<?php

declare( strict_types = 1 );

use WooCommerce\PayPalCommerce\WcSubscriptions\Helper\SubscriptionHelper;
use WooCommerce\PayPalCommerce\WcGateway\Gateway\CreditCardGateway;
use WooCommerce\PayPalCommerce\WcGateway\Gateway\PayPalGateway;

/**
 * Class SubscriptionModule
 */
class SUMOSubs_WC_PayPal_Payments_Helper extends SubscriptionHelper {

    /**
     * Whether the current product is a subscription.
     *
     * @return bool
     */
    public function current_product_is_subscription(): bool {
        $product = wc_get_product();
        return $product && sumo_is_subscription_product( $product->get_id() );
    }

    /**
     * Whether the current cart contains subscriptions.
     *
     * @return bool
     */
    public function cart_contains_subscription(): bool {
        return sumo_is_cart_contains_subscription_items();
    }

    /**
     * Checks if order contains subscription.
     *
     * @param int $order_id The order ID.
     * @return boolean Whether order is a subscription or not.
     */
    public function has_subscription( $order_id ): bool {
        return sumo_order_contains_subscription( $order_id );
    }

    /**
     * Whether only automatic payment gateways are accepted.
     *
     * @return bool
     */
    public function accept_only_automatic_payment_gateways(): bool {
        return SUMOSubs_Payment_Gateways::gateway_requires_auto_renewals( PayPalGateway::ID ) || SUMOSubs_Payment_Gateways::gateway_requires_auto_renewals( CreditCardGateway::ID );
    }

    /**
     * Checks if cart contains only one item.
     *
     * @return bool
     */
    public function cart_contains_only_one_item(): bool {
        $cart = WC()->cart;
        if ( ! $cart || $cart->is_empty() ) {
            return false;
        }

        if ( count( $cart->get_cart() ) > 1 ) {
            return false;
        }

        return true;
    }

    /**
     * Whether pay for order contains subscriptions.
     *
     * @return bool
     */
    public function order_pay_contains_subscription(): bool {
        if ( ! is_wc_endpoint_url( 'order-pay' ) ) {
            return false;
        }

        global $wp;
        $order_id = ( int ) $wp->query_vars[ 'order-pay' ];
        if ( 0 === $order_id ) {
            return false;
        }

        return $this->has_subscription( $order_id );
    }

    /**
     * Returns previous order transaction from the given subscription.
     *
     * @param int $subscription_id
     * @param string  $vault_token_id Vault token id.
     * @return string
     */
    public function previous_transaction( $subscription_id, string $vault_token_id ): string {
        $orders = sumosubs_get_related_orders_for_subscription( $subscription_id );
        if ( ! $orders || ! $vault_token_id ) {
            return '';
        }
        // Sort orders by oder ID descending.
        rsort( $orders );
        $current_order = wc_get_order( array_shift( $orders ) );
        if ( ! $current_order instanceof WC_Order ) {
            return '';
        }
        foreach ( $orders as $order_id ) {
            $order = wc_get_order( $order_id );
            if ( is_a( $order, 'WC_Order' ) && in_array( $order->get_status(), array( 'processing', 'completed' ), \true ) && $current_order->get_payment_method() === $order->get_payment_method() ) {
                $transaction_id = $order->get_transaction_id();
                $tokens         = $order->get_payment_tokens();
                foreach ( $tokens as $token ) {
                    $wc_token = \WC_Payment_Tokens::get( $token );
                    if ( $transaction_id && $wc_token instanceof \WC_Payment_Token && $wc_token->get_token() === $vault_token_id ) {
                        return $transaction_id;
                    }
                }
            }
        }
        return '';
    }

    /**
     * Checks whether subscription needs subscription intent.
     *
     * @param string $subscription_mode The subscription mode.
     * @return bool
     */
    public function need_subscription_intent( string $subscription_mode ): bool {
        if ( $this->cart_contains_subscription() ) {
            return \false;
        }

        return parent::need_subscription_intent( $subscription_mode );
    }
}
