<?php

declare( strict_types = 1 );

use WooCommerce\PayPalCommerce\Vendor\Dhii\Container\ServiceProvider;
use WooCommerce\PayPalCommerce\Vendor\Dhii\Modular\Module\ModuleInterface;
use Psr\Log\LoggerInterface;
use WooCommerce\PayPalCommerce\ApiClient\Exception\RuntimeException;
use WooCommerce\PayPalCommerce\Vaulting\PaymentTokenRepository;
use WooCommerce\PayPalCommerce\WcGateway\Gateway\PayPalGateway;
use WooCommerce\PayPalCommerce\WcGateway\Gateway\CreditCardGateway;
use WooCommerce\PayPalCommerce\Vendor\Interop\Container\ServiceProviderInterface;
use WooCommerce\PayPalCommerce\Vendor\Psr\Container\ContainerInterface;

/**
 * Class SubscriptionModule
 */
class SUMOSubs_WC_PayPal_Payments_Module implements ModuleInterface {

    /**
     * {@inheritDoc}
     */
    public function setup(): ServiceProviderInterface {
        return new ServiceProvider(
                array(
            // Backward compatibility with version 2.4.2 or lower.
            'subscription.helper' => static function( ContainerInterface $container ): SUMOSubs_WC_PayPal_Payments_Helper {
                return new SUMOSubs_WC_PayPal_Payments_Helper( $container->get( 'wcgateway.settings' ) );
            },
            'sumosubs-subscription.renewal-handler' => static function( ContainerInterface $container ): SUMOSubs_WC_PayPal_Payments_Renewal_Handler {
                return new SUMOSubs_WC_PayPal_Payments_Renewal_Handler(
                $container->get( 'woocommerce.logger.woocommerce' ),
                $container->get( 'vaulting.repository.payment-token' ),
                $container->get( 'api.endpoint.order' ),
                $container->get( 'api.factory.purchase-unit' ),
                $container->get( 'api.factory.shipping-preference' ),
                $container->get( 'api.factory.payer' ),
                $container->get( 'onboarding.environment' ),
                $container->get( 'wcgateway.settings' ),
                $container->get( 'wcgateway.processor.authorized-payments' ),
                $container->get( 'wcgateway.funding-source.renderer' ),
                $container->get( 'wc-subscriptions.helpers.real-time-account-updater' ),
                $container->get( 'wc-subscriptions.helper' )
                );
            },
                ),
                array()
        );
    }

    /**
     * {@inheritDoc}
     */
    public function run( ContainerInterface $c ): void {

        // Add integration for SUMO Subscriptions.
        add_filter( 'woocommerce_payment_gateway_supports', array( $this, 'register_supports' ), 10, 3 );

        add_filter( 'sumosubscriptions_is_' . PayPalGateway::ID . '_preapproval_status_valid', '__return_true' );
        add_filter( 'sumosubscriptions_is_' . CreditCardGateway::ID . '_preapproval_status_valid', '__return_true' );

        add_filter(
                'sumosubscriptions_is_' . PayPalGateway::ID . '_preapproved_payment_transaction_success',
                function( $bool, $subscription_id, $renew_order ) use( $c ) {
                    $this->renew( $renew_order, $c );

                    return $this->pay_renew( $renew_order, $c );
                },
                10,
                3
        );

        add_filter(
                'sumosubscriptions_is_' . CreditCardGateway::ID . '_preapproved_payment_transaction_success',
                function( $bool, $subscription_id, $renew_order ) use( $c ) {
                    $this->renew( $renew_order, $c );

                    return $this->pay_renew( $renew_order, $c );
                },
                10,
                3
        );

        add_action(
                'woocommerce_order_status_changed',
                function( $order_id ) use( $c ) {
                    $order = wc_get_order( $order_id );

                    if ( ! sumo_order_contains_subscription( $order ) ) {
                        return;
                    }

                    $payment_method = $order->get_payment_method();
                    if ( empty( $payment_method ) ) {
                        $payment_method = isset( $_REQUEST[ 'payment_method' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'payment_method' ] ) ) : '';
                    }

                    // Double check subscription payment method.
                    if ( empty( $payment_method ) || ! in_array( $payment_method, array( PayPalGateway::ID, CreditCardGateway::ID ), true ) ) {
                        return;
                    }

                    $payment_token_repository = $c->get( 'vaulting.repository.payment-token' );
                    $logger                   = $c->get( 'woocommerce.logger.woocommerce' );

                    $this->add_payment_token_id( $order, $payment_token_repository, $logger, $payment_method );
                },
                0
        );

        $this->maybe_remove_action_scheduler_filter();
    }

    /**
     * Handles a Subscription product renewal.
     *
     * @param \WC_Order               $order     WooCommerce order.
     * @param ContainerInterface|null $container The container.
     * @return void
     */
    protected function renew( $order, $container ) {
        if ( $order instanceof WC_Order ) {
            $handler = $container->get( 'sumosubs-subscription.renewal-handler' );
            return $handler->renew( $order );
        }

        return false;
    }

    /**
     * Handles a Subscription product renewal.
     *
     * @param \WC_Order               $order     WooCommerce order.
     * @param ContainerInterface|null $container The container.
     * @return void
     */
    protected function pay_renew( $order, $container ) {
        if ( $order instanceof WC_Order ) {
            $handler = $container->get( 'sumosubs-subscription.renewal-handler' );
            return $handler->pay_renew( $order );
        }

        return false;
    }

    /**
     * Adds Payment token ID to order.
     *
     * @param \WC_Order    $order 
     * @param PaymentTokenRepository $payment_token_repository The payment repository.
     * @param LoggerInterface        $logger                   The logger.
     */
    protected function add_payment_token_id( \WC_Order $order, PaymentTokenRepository $payment_token_repository, LoggerInterface $logger, $payment_method = '' ) {
        try {
            $tokens = $payment_token_repository->all_for_user_id( $order->get_user_id() );
            if ( $tokens ) {
                $latest_token_id = end( $tokens )->id() ? end( $tokens )->id() : '';

                sumo_save_subscription_payment_info( $order, array(
                    'payment_type'   => sumo_get_subscription_order_payment( $order, 'payment_type' ),
                    'payment_method' => $payment_method,
                    'payment_key'    => $latest_token_id,
                ) );
            }
        } catch ( RuntimeException $error ) {
            $message = sprintf(
                    // translators: %1$s is the payment token Id, %2$s is the error message.
                    __(
                            'Could not add token Id to order %1$s: %2$s',
                            'sumosubscriptions'
                    ),
                    $order->get_id(),
                    $error->getMessage()
            );

            $logger->log( 'warning', $message );
        }
    }

    /**
     * Add SUMO Subscriptions support to PayPal Standard.
     *
     * @param bool          $support Tell if the feature is support.
     * @param string        $feature Feature to check.
     * @param object|string $gateway Current gateway.
     *
     * @return bool
     */
    public function register_supports( $support, $feature, $gateway ) {
        $gateway = is_object( $gateway ) ? $gateway->id : $gateway;

        if ( ! in_array( $gateway, array( PayPalGateway::ID, CreditCardGateway::ID ), true ) ) {
            return $support;
        }

        $supports = array(
            'sumo_subscriptions',
        );
        return in_array( $feature, $supports, true ) ? true : $support;
    }

    /**
     * Remove filter action_scheduler_before_execute added from WooCommerce\PayPalCommerce\Subscription\SubscriptionModule
     * to avoid errors with subscription renew process.
     *
     * @return void
     */
    protected function maybe_remove_action_scheduler_filter() {
        global $wp_filter;

        if ( empty( $wp_filter[ 'action_scheduler_before_execute' ] ) || ! class_exists( 'ReflectionFunction' ) ) {
            return;
        }

        foreach ( $wp_filter[ 'action_scheduler_before_execute' ] as $priority => $callbacks ) {
            foreach ( $callbacks as $id => $callback ) {
                if ( empty( $callback[ 'function' ] ) || ! is_object( $callback[ 'function' ] ) ) {
                    continue;
                }

                $function = new ReflectionFunction( $callback[ 'function' ] );
                if (
                        $function->getClosureThis() instanceof WooCommerce\PayPalCommerce\Subscription\SubscriptionModule ||
                        $function->getClosureThis() instanceof WooCommerce\PayPalCommerce\PayPalSubscriptions\PayPalSubscriptionsModule
                ) {
                    unset( $wp_filter[ 'action_scheduler_before_execute' ]->callbacks[ $priority ][ $id ] );
                }
            }
        }
    }
}
