<?php

namespace SPC_Pro;

use SPC\Services\SDK_Integrations;
use SPC_Pro\Modules\Frontend;
use SPC_Pro\Modules\HTML_Modifier;
use SPC_Pro\Modules\Settings_Manager;
use SPC_Pro\Modules\Fallback_Cache;
use SPC_Pro\Services\Cloudflare_Transform_Rule;
use SW_CLOUDFLARE_PAGECACHE;

/**
 * Feature Loader
 */
class Loader {
	const SPC_PRO_TRANSLATION_CACHE_KEY_PREFIX = 'spc_pro_translations';
	const SPC_FREE_SLUG                        = 'wp-cloudflare-page-cache';

	/**
	 * Pro modules.
	 *
	 * @var array $pro_modules Pro modules.
	 */
	private $pro_modules = [];

	/**
	 * Loader constructor.
	 */
	public function __construct( SW_CLOUDFLARE_PAGECACHE $plugin ) {
		$this->pro_modules['settings_manager_pro'] = new Settings_Manager();
		$this->pro_modules['transform_rule']       = new Cloudflare_Transform_Rule( $plugin );
		$this->pro_modules['fallback_cache']       = new Fallback_Cache( $plugin );

		if ( ! \SPC\Loader::can_process_html() ) {
			return;
		}

		$this->pro_modules['html_modifier_pro'] = new HTML_Modifier();
		$this->pro_modules['frontend_pro']      = new Frontend();
	}

	/**
	 * Initialize.
	 *
	 * @return void
	 */
	public function init() {
		add_filter( 'spc_loaded_modules', [ $this, 'add_modules' ] );
		add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'ensure_translations' ], 11 );
		add_filter( 'themeisle_sdk_ran_promos', '__return_true' );
		add_filter( 'spc_lazyload_behaviours', [ $this, 'add_lazyload_behaviours' ] );
		$sdk = new SDK_Integrations();

		add_filter( $sdk->get_product_key() . '_hide_license_field', '__return_true' );
		add_filter( $sdk->get_product_key() . '_lc_no_valid_string', [ $this, 'replace_notice_link' ] );
	}

	/**
	 * Replace the notice link.
	 *
	 * @param string $message The message.
	 *
	 * @return string
	 */
	public function replace_notice_link( $message ) {
		return str_replace( '<a href="%s">', '<a href="' . admin_url( 'admin.php?page=super-page-cache-license' ) . '">', $message );
	}

	/**
	 * Add lazyload behaviour for viewport.
	 *
	 * @param array $behaviours Behaviours.
	 *
	 * @return array
	 */
	public function add_lazyload_behaviours( $behaviours ) {
		$behaviours[ Frontend::LAZY_LOAD_BEHAVIOUR_VIEWPORT ] = true;

		return $behaviours;
	}

	/**
	 * Add additional pro modules.
	 *
	 * @param array $loaded_modules Loaded modules.
	 *
	 * @return array
	 */
	public function add_modules( array $loaded_modules ) {
		return array_merge( $loaded_modules, $this->pro_modules );
	}

	/**
	 * Ensure translations are downloaded from the Free version as we reuse them.
	 *
	 * @param mixed $_transient_data Transient data containing plugin translations.
	 *
	 * @return mixed
	 */
	public function ensure_translations( $_transient_data ) {
		if (
			! is_object( $_transient_data )
			|| ! isset( $_transient_data->translations )
			|| ! is_array( $_transient_data->translations )
		) {
			return $_transient_data;
		}

		$locale        = get_locale();
		$transient_key = self::SPC_PRO_TRANSLATION_CACHE_KEY_PREFIX . '_' . $locale;

					// Check transient cache first
					$translations_check = get_transient( $transient_key );
		if ( $translations_check === false ) {

			// Get the latest update date and download URL.
			$response = wp_remote_post(
				'https://api.wordpress.org/translations/plugins/1.0/',
				array(
					'headers' => array(
						'Content-Type' => 'application/x-www-form-urlencoded',
					),
					'body'    => array(
						'wp_version' => get_bloginfo( 'version' ),
						'locale'     => $locale,
						'version'    => SWCFPC_VERSION,
						'slug'       => self::SPC_FREE_SLUG,
					),
				)
			);

			if ( is_wp_error( $response ) ) {
				return $_transient_data;
			}

			$body = json_decode( wp_remote_retrieve_body( $response ), true );

			if ( isset( $body['translations'] ) && is_array( $body['translations'] ) ) {
				$translations_check = $body['translations'];

				set_transient( $transient_key, $translations_check, DAY_IN_SECONDS );
			}
		}

		if ( ! is_array( $translations_check ) ) {
			return $_transient_data;
		}

		foreach ( $translations_check as $translation ) {
			$has_translation = false;

			// Check if plugin translation was already loaded.
			foreach ( $_transient_data->translations as $existing ) {
				if (
					self::SPC_FREE_SLUG === $existing['slug']
					&& $existing['language'] === $translation['language']
				) {
					$has_translation = true;
					break;
				}
			}

			if ( ! $has_translation ) {
				$_transient_data->translations[] = array(
					'language'   => $translation['language'],
					'updated'    => $translation['updated'],
					'package'    => $translation['package'],
					'slug'       => self::SPC_FREE_SLUG,
					'autoupdate' => true,
					'type'       => 'plugin',
					'version'    => 'latest',
				);
			} elseif ( ! empty( $existing ) && strtotime( $existing['updated'] ) < strtotime( $translation['updated'] ) ) {
				$existing['updated'] = $translation['updated'];
			}
		}

					return $_transient_data;
	}
}
