<?php

namespace SPC_Pro\Modules;

use SPC\Modules\Module_Interface;
use SPC\Services\Settings_Store;
use SPC\Utils\Helpers;
use SPC_Pro\Modules\BgOptimizer\Lazyload;
use SPC_Pro\Constants;
use SPC\Constants as CoreConstants;
use SPC_Pro\Modules\PageProfiler\Profile;
use SPC_Pro\Modules\Preload\Links;

/**
 * HTML Modifier
 */
class HTML_Modifier implements Module_Interface {
	private const DEFAULT_CONFIG = [
		'defer_js'          => false,
		'delay_js'          => false,
		'lazyload_viewport' => false,
		'unused_css'        => false,
	];

	/**
	 * Page profiler.
	 *
	 * @var \SPC_Pro\Modules\PageProfiler\Profile
	 */
	private Profile $page_profiler;

	const DELAY_ATTR = 'data-spc-src';

	/**
	 * Initialize the module.
	 *
	 * @return void
	 */
	public function init() {
		if ( ! \SPC\Loader::can_process_html() ) {
			return;
		}

		add_filter( 'swcfpc_normal_fallback_cache_html', [ $this, 'alter_cached_html' ], 9, 2 );
		add_filter( 'swcfpc_curl_fallback_cache_html', [ $this, 'alter_cached_html' ], 9, 2 );

		add_filter( 'swcfpc_normal_fallback_cache_html', [ $this, 'add_preload_links' ], 11, 2 );
		add_filter( 'swcfpc_curl_fallback_cache_html', [ $this, 'add_preload_links' ], 11, 2 );

		if ( Settings_Store::get_instance()->is_lazyload_viewport_enabled() ) {
			add_filter( 'spc_html_modifier_parser_img', [ $this, 'alter_parser_img' ], 10, 2 );
		}

		add_filter( 'spc_lazyload_bg_lazyload_css', [ $this, 'alter_lazyload_bg_lazyload_css' ], 10, 2 );

		// Only create profiler if not already set (allows test injection)
		if ( ! isset( $this->page_profiler ) ) {
			$this->page_profiler = new Profile();
		}
	}

	/**
	 * Set the page profiler instance (for testing purposes).
	 *
	 * @param Profile $page_profiler The page profiler instance.
	 * @return void
	 */
	public function set_page_profiler( Profile $page_profiler ): void {
		$this->page_profiler = $page_profiler;
	}
	/**
	 * Add preload links to the HTML.
	 *
	 * @param string $html The HTML.
	 * @param string $key The cache key.
	 *
	 * @return string
	 */
	public function add_preload_links( $html, $key ) {
		if ( Links::get_links_count() > 0 ) {
			$html = str_replace(
				'</head>',
				Links::get_links_html() . '</head>',
				$html
			);

		}
		return $html;
	}
	/**
	 * Alter the lazyload background lazyload CSS.
	 *
	 * @param string $css The CSS.
	 * @param array $selectors The selectors.
	 *
	 * @return string
	 */
	public function alter_lazyload_bg_lazyload_css( $css, $selectors ) {
		$profile_id = Profile::generate_id();
		if ( $this->page_profiler->exists_all( $profile_id ) ) {
			return Lazyload::get_personalized_css( $this->page_profiler->get_profile_data( $profile_id ), $selectors );
		}
		return $css;
	}
	/**
	 * Alter HTML before saving into cache.
	 *
	 * @param string $html HTML content.
	 *
	 * @return string
	 */
	public function alter_cached_html( $html, $key = null ) {
		$defer_js          = (bool) Settings_Store::get_instance()->get( Constants::SETTING_DEFER_JS );
		$delay_js          = (bool) Settings_Store::get_instance()->get( Constants::SETTING_DELAY_JS );
		$lazyload_viewport = ( Settings_Store::get_instance()->get( CoreConstants::SETTING_LAZY_LOAD_BEHAVIOUR ) === Frontend::LAZY_LOAD_BEHAVIOUR_VIEWPORT );

		$unused_css = (bool) Settings_Store::get_instance()->get( CoreConstants::SETTING_UNUSED_CSS );

		return ! $defer_js && ! $delay_js && ! $lazyload_viewport && ! $unused_css ? $html : $this->parse_html(
			$html,
			wp_parse_args(
				[
					'defer_js'          => $defer_js,
					'delay_js'          => $delay_js,
					'lazyload_viewport' => $lazyload_viewport,
					'unused_css'        => $unused_css,
				],
				self::DEFAULT_CONFIG
			),
			$key
		);
	}

	/**
	 * Parse the HTML and add required attributes.
	 *
	 * @param string $html The HTML content.
	 * @param array  $config The config to use when parsing.
	 * @param string $key The cache key.
	 *
	 * @return string
	 */
	private function parse_html( $html, $config, $key ) {
		$parser = new \WP_HTML_Tag_Processor( $html );

		$scripts_to_wrap = [];

		$profile_id = Profile::generate_id( $key );
		// @phpstan-ignore-next-line - next_tag can receive a tag name as a string.
		while ( $parser->next_tag( [ 'SCRIPT', 'STYLE', 'LINK' ] ) ) {
			if ( $config['unused_css'] ) {
				$this->handle_unused_css( $parser, $profile_id );
			}
			if ( $parser->get_tag() === 'SCRIPT' ) {
				if ( $config['defer_js'] ) {
					$this->handle_js_defer( $parser, $scripts_to_wrap );
				}
				if ( $config['delay_js'] ) {
					$this->handle_js_delay( $parser );
				}
			}
		}

		$updated = $parser->get_updated_html();

		foreach ( $scripts_to_wrap as $inline_script ) {
			$updated = str_replace(
				$inline_script,
				sprintf( 'document.addEventListener("DOMContentLoaded", function() {%s});', $inline_script ),
				$updated
			);
		}

		if ( ( $config['lazyload_viewport'] || $config['unused_css'] ) && ! empty( Helpers::get_current_url() ) ) {
			if ( ! $this->page_profiler->exists_all( $profile_id ) ) {
				$js_optimizer = Frontend::get_optimizer_script( false );
				$missing      = $this->page_profiler->missing_devices( $profile_id );
				$time         = time();
				$page         = base64_encode( Helpers::get_current_absolute_url() );
				$hmac         = wp_hash( $profile_id . $time . $page, 'nonce' );
				$js_optimizer = str_replace(
					[ Profile::PLACEHOLDER, Profile::PLACEHOLDER_MISSING, Profile::PLACEHOLDER_TIME, Profile::PLACEHOLDER_HMAC, Profile::PLACEHOLDER_URL ],
					[ $profile_id, implode( ',', $missing ), (string) $time, $hmac, $page ],
					$js_optimizer
				);
				$updated      = str_replace(
					Frontend::get_optimizer_script(),
					$js_optimizer,
					$updated
				);
			} else {
				$updated = str_replace(
					Frontend::get_optimizer_script(),
					'',
					$updated
				);
			}
			Profile::set_current_profile_id( $profile_id );
			$this->page_profiler->set_current_profile_data();
		}
		if ( $config['unused_css'] && ! $this->is_current_url_excluded( CoreConstants::SETTING_UNUSED_CSS_EXCLUDED_PATHS ) && $this->page_profiler->exists_all( $profile_id ) ) {
			$css_lazy_loader = require_once SWCFPC_PLUGIN_PATH . 'pro/assets/build/css-lazy-loader.php';
			$updated         = str_replace(
				[ '<head>', '</body>' ],
				[
					'<head><style data-critical-css>' . $this->page_profiler->get_critical_css() . '</style>',
					'<script type="text/javascript"> ' . $css_lazy_loader . '</script></body>',
				],
				$updated
			);
		}
		return $updated;
	}

	/**
	 * Alter the parser for the img tag.
	 *
	 * @param \WP_HTML_Tag_Processor $parser The parser.
	 * @param int                    $id The ID of the image.
	 *
	 * @return \WP_HTML_Tag_Processor The parser.
	 */
	public function alter_parser_img( $parser, $id ) {

		if ( $this->page_profiler->is_in_all_viewports( $id ) ) {
			$parser->set_attribute( 'data-spc-skip-lazyload', true );
			Links::add_link(
				[
					'url'      => $parser->get_attribute( 'src' ),
					'priority' => 'high',
					'srcset'   => $parser->get_attribute( 'srcset' ),
					'sizes'    => $parser->get_attribute( 'sizes' ),
				]
			);
		}
		if ( $this->page_profiler->is_lcp_image_in_all_viewports( $id ) ) {
			$parser->set_attribute( 'data-spc-lcp-image', true );
			$parser->set_attribute( 'data-spc-skip-lazyload', true );
			Links::add_link(
				[
					'url'      => $parser->get_attribute( 'src' ),
					'priority' => 'high',
					'srcset'   => $parser->get_attribute( 'srcset' ),
					'sizes'    => $parser->get_attribute( 'sizes' ),
				]
			);
		}
		if ( $parser->get_attribute( 'data-spc-skip-lazyload' ) ) {
			return $parser;
		}

		//We remove the fetchpriority attribute if the profile is complete and the image should be lazyloaded.
		if ( $this->page_profiler->exists_all( Profile::get_current_profile_id() ) ) {
			$parser->remove_attribute( 'fetchpriority' );
		}
		return $parser;
	}
	/**
	 * Handle unused CSS.
	 *
	 * @param \WP_HTML_Tag_Processor $parser The parser.
	 * @param string                 $profile_id The profile ID.
	 *
	 * @return void
	 */
	private function handle_unused_css( $parser, $profile_id ) {
		if ( $this->is_current_url_excluded( CoreConstants::SETTING_UNUSED_CSS_EXCLUDED_PATHS ) ) {
			return;
		}
		// If the tag has the data-spc-skip-ucss attribute, skip the unused CSS processing.
		if ( $parser->get_attribute( 'data-spc-skip-ucss' ) ) {
			return;
		}

		$is_profile_complete = $this->page_profiler->exists_all( $profile_id );
		if ( $parser->get_tag() === 'LINK' && $parser->get_attribute( 'rel' ) === 'stylesheet' ) {
			if ( $this->is_excluded_external_file( $parser->get_attribute( 'href' ), CoreConstants::SETTING_UNUSED_CSS_EXCLUDED_CSS, Constants::UNUSED_CSS_EXTERNAL_EXCLUSIONS ) ) {
				$parser->set_attribute( 'data-spc-skip-ucss', true );
				return;
			}
			if ( $is_profile_complete ) {
				$parser->set_attribute( 'rel', 'lazy-stylesheet' );
			}
		} elseif ( $parser->get_tag() === 'LINK' && $parser->get_attribute( 'rel' ) === 'preload' && $parser->get_attribute( 'as' ) === 'style' ) {
			if ( $this->is_excluded_external_file( $parser->get_attribute( 'href' ), CoreConstants::SETTING_UNUSED_CSS_EXCLUDED_CSS, Constants::UNUSED_CSS_EXTERNAL_EXCLUSIONS ) ) {
				$parser->set_attribute( 'data-spc-skip-ucss', true );
				return;
			}
			if ( $is_profile_complete ) {
				$parser->set_attribute( 'rel', 'lazy-preload' );
			}
		} elseif ( $parser->get_tag() === 'STYLE' ) {
			if ( $this->is_tag_id_content_excluded( $parser, CoreConstants::SETTING_UNUSED_CSS_EXCLUDED_CSS ) ) {
				$parser->set_attribute( 'data-spc-skip-ucss', true );
				return;
			}
			if ( $is_profile_complete ) {
				$parser->set_attribute( 'type', 'text/lazy-css' );
			}
		}
	}
	/**
	 * Handle JS defer.
	 *
	 * @param \WP_HTML_Tag_Processor $parser The parser.
	 * @param array                  $scripts_to_wrap The scripts to wrap. Passed by reference.
	 *
	 * @return void
	 */
	private function handle_js_defer( $parser, &$scripts_to_wrap ) {
		$src = $parser->get_attribute( 'src' );

		if ( $parser->get_attribute( 'type' ) === 'module' ) {
			return;
		}

		if ( ! empty( $src ) && ! $this->is_excluded_src( $src, Constants::DEFER_JS_EXTERNAL_EXCLUSIONS ) ) {

			$id = $parser->get_attribute( 'id' );

			$should_defer = apply_filters( 'spc_defer_script', true, $src, $id );

			if ( ! $should_defer ) {
				return;
			}

			$parser->set_attribute( 'defer', true );

			return;
		}

		$inner_text = $parser->get_modifiable_text();

		if ( empty( $inner_text ) ) {
			return;
		}

		$jquery_markers = [ 'jQuery', '$(', '$.' ];
		$jquery_skip    = [ 'DOMContentLoaded', 'document.write' ];

		$contains_jquery_marker = array_filter(
			$jquery_markers,
			function ( $marker ) use ( $inner_text ) {
				return strpos( $inner_text, $marker ) !== false;
			}
		);

		if ( ! $contains_jquery_marker ) {
			return;
		}

		$has_jquery_skip = array_filter(
			$jquery_skip,
			function ( $ignore ) use ( $inner_text ) {
				return strpos( $inner_text, $ignore ) !== false;
			}
		);

		if ( $has_jquery_skip ) {
			return;
		}

		$scripts_to_wrap[] = $inner_text;
	}

	/**
	 * Check if the tag ID or content is excluded from the setting.
	 *
	 * @param \WP_HTML_Tag_Processor $parser The parser.
	 * @param string $setting_name The name of the setting to check.
	 *
	 * @return bool True if the tag ID or content is excluded, false otherwise.
	 */
	private function is_tag_id_content_excluded( $parser, $setting_name ) {
		$excluded_content = Settings_Store::get_instance()->get( $setting_name );
		if ( ! is_array( $excluded_content ) ) {
			$excluded_content = [];
		}
		$id      = $parser->get_attribute( 'id' );
		$content = $parser->get_modifiable_text();
		if ( empty( $id ) && empty( $content ) ) {
			return false;
		}
		return ! empty(
			array_filter(
				$excluded_content,
				function ( $exclude ) use ( $id, $content ) {
					return ( ! empty( $id ) && strpos( $id, $exclude ) !== false ) || ( ! empty( $content ) && strpos( $content, $exclude ) !== false );
				}
			)
		);
	}
	/**
	 * Check if the current URL is excluded from the setting.
	 *
	 * @param string $setting_name The name of the setting to check.
	 *
	 * @return bool True if the current URL is excluded, false otherwise.
	 */
	private function is_current_url_excluded( $setting_name ) {
		// Check for excluded paths.
		$excluded_paths = Settings_Store::get_instance()->get( $setting_name );

		if ( ! is_array( $excluded_paths ) ) {
			$excluded_paths = [];
		}

		$is_excluded_path = array_filter(
			$excluded_paths,
			function ( $exclude ) {
				return strpos( $_SERVER['REQUEST_URI'], $exclude ) !== false;
			}
		);

		if ( $is_excluded_path ) {
			return true;
		}

		return false;
	}
	/**
	 * Handle JS delay.
	 *
	 * @param \WP_HTML_Tag_Processor $parser The parser.
	 *
	 * @return void
	 */
	private function handle_js_delay( $parser ) {
		if ( $this->is_current_url_excluded( Constants::SETTING_DELAY_JS_EXCLUDED_PATHS ) ) {
			return;
		}

		// Get the source.
		$source = $parser->get_attribute( 'src' );
		if ( empty( $source ) || $this->is_excluded_src( $source, Constants::DELAY_JS_EXTERNAL_EXCLUSIONS ) ) {
			return;
		}

		if ( $this->is_excluded_external_file( $source, Constants::SETTING_DELAY_JS_EXCLUDED_FILES ) ) {
			return;
		}

		$parser->set_attribute( self::DELAY_ATTR, $source );
		$parser->remove_attribute( 'src' );
	}

	/**
	 * Check if the external file is excluded from the setting.
	 *
	 * @param string $source The source URL to check.
	 * @param string $setting_name The name of the setting to check.
	 *
	 * @return bool True if the external file is excluded, false otherwise.
	 */
	private function is_excluded_external_file( $source, $setting_name, $extra_exclusions = [] ) {
		$excluded_files = Settings_Store::get_instance()->get( $setting_name );
		if ( ! is_array( $excluded_files ) ) {
			$excluded_files = [];
		}
		$excluded_files = array_merge( $excluded_files, $extra_exclusions );
		if ( empty( $excluded_files ) ) {
			return false;
		}
		return ! empty(
			array_filter(
				$excluded_files,
				function ( $exclude ) use ( $source ) {
					return strpos( $source, trim( $exclude ) ) !== false;
				}
			)
		);
	}

	/**
	 * Check if a given src matches any of the exclusion patterns in a given list.
	 *
	 * @param string   $src The source URL to check.
	 * @param string[] $exclusion_list The list of exclusion patterns.
	 *
	 * @return bool True if the src matches any exclusion pattern, false otherwise.
	 */
	private function is_excluded_src( $src, $exclusion_list ) {
		foreach ( $exclusion_list as $pattern ) {
			if ( preg_match( '#' . $pattern . '#', $src ) ) {
				return true;
			}
		}

		return false;
	}
}
