<?php

namespace SPC_Pro\Modules\PageProfiler;

use SPC\Utils\Helpers;

/**
 * Class Profile
 *
 * Handles page profiling functionality for SPC, including storage and retrieval
 * of above-fold image data for different device types.
 *
 * @package SPC_Pro\Modules\PageProfiler
 */
class Profile {

	/**
	 * Placeholder used to identify where a profile ID should be inserted.
	 */
	const PLACEHOLDER = '###pageprofileid###';

	/**
	 * Placeholder used to identify where a profile HMAC should be inserted.
	 */
	const PLACEHOLDER_HMAC = '###profilehmac###';
	/**
	 * Placeholder used to identify where a profile time should be inserted.
	 */
	const PLACEHOLDER_TIME = '###profiletime###';

	/**
	 * Placeholder used to indicate a missing profile ID.
	 */
	const PLACEHOLDER_MISSING = '###pageprofileidmissing###';

	const PLACEHOLDER_URL = '###pageurl###';
	/**
	 * Device type constant for mobile devices.
	 */
	const DEVICE_TYPE_MOBILE = 1;

	/**
	 * Device type constant for desktop devices.
	 *
	 * @var int
	 */
	const DEVICE_TYPE_DESKTOP = 2;

	/**
	 * Stores the current profile ID being processed.
	 *
	 * @var string|null
	 */
	private static $current_profile_id = null;

	/**
	 * Stores the current profile data for all device types.
	 *
	 * @var array
	 */
	private static $current_profile_data = [];

	/**
	 * The storage handler instance.
	 *
	 * @var Storage\Base
	 */
	private $storage;

	/**
	 * Constructor.
	 *
	 * Initializes the storage handler based on the provided filter.
	 *
	 * @throws \Exception If an invalid storage class is provided.
	 */
	public function __construct() {
		/**
		 * Filter the storage class.
		 * Allows to change the storage class to a different one, i.e a database storage class/file storage class etc.
		 *
		 * @param string $storage_class The storage class.
		 *
		 * @return string The storage class.
		 */
		$storage_class = apply_filters( 'spc_page_profiler_storage', wp_using_ext_object_cache() ? Storage\ObjectCache::class : Storage\Transients::class );
		if ( ! is_subclass_of( $storage_class, Storage\Base::class ) ) {
			throw new \Exception( 'Invalid storage class' );
		}
		$this->storage = new $storage_class();
		add_action( 'swcfpc_fallback_cache_purged_all', [ $this, 'delete_all' ] );
		add_action( 'swcfpc_fallback_cache_purged_url', [ $this, 'delete' ] );
	}

	/**
	 * Generate a unique ID for the page profile.
	 * @param string $key The cache key to generate the ID for.
	 * @return string
	 */
	public static function generate_id( $key = null ) {
		if ( empty( $key ) ) {
			$key = function_exists( 'swcfpc_fallback_cache_get_current_page_cache_key' ) ? swcfpc_fallback_cache_get_current_page_cache_key() : '';
		}
		return sha1( $key );
	}
	/**
	 * Delete all profile data.
	 * @return void
	 */
	public function delete_all() {
		if ( apply_filters( 'spc_page_profiler_dont_delete_all', false ) ) {
			return;
		}
		$this->storage->delete_all();
	}

	/**
	 * Delete profile data for a specific key.
	 * @param string $key The key to delete.
	 * @return void
	 */
	public function delete( $key ) {
		if ( apply_filters( 'spc_page_profiler_dont_delete_url', false, $key ) ) {
			return;
		}
		$this->storage->delete( $key );
	}

	/**
	 * Stores above-fold image data for a specific profile ID and device type.
	 *
	 * @param string                                                                              $id The profile ID.
	 * @param string                                                                              $device_type The device type constant.
	 * @param array<string>                                                                       $above_fold_images Array of above-fold images.
	 * @param array<string, array<string, array<int, string>>>                                    $af_bg_selectors Array of above-fold background selectors.
	 *                                        Array structure:
	 *                                        [
	 *                                            'css_selector' => [
	 *                                                'above_the_fold_selector' => [
	 *                                                    0 => 'background_image_url',
	 *                                                    1 => 'background_image_url',
	 *                                                    ...
	 *                                                ],
	 *                                                ...
	 *                                            ],
	 *                                            ...
	 *                                        ] Array of above-fold background selectors.
	 * @param array{imageId?: string, bgSelector?: string, bgUrls?: array<string>, type?: string} $lcp_data LCP (Largest Contentful Paint) data.
	 *                                                                   where 'imageId' is the element identifier,
	 *                                                                   'bgSelector' is the selector,
	 *                                                                   'bgUrls' is an array of URLs
	 *                                                                   'type' is the type of the LCP element.
	 * @param array $critical_css { css: array } Structured critical CSS data for server-side merging.
	 *
	 * @return void
	 */
	public function store( string $id, string $device_type, array $above_fold_images, $af_bg_selectors = [], $lcp_data = [], $critical_css = [] ) {
		if ( ! in_array( (int) $device_type, self::get_active_devices(), true ) ) {
			return;
		}

		// store $above_fold_images as image_id => true to faster access.
		$above_fold_images = array_fill_keys( $above_fold_images, true );

		$this->storage->store(
			$id . '_' . $device_type,
			[
				'af'  => $above_fold_images,
				'bg'  => $af_bg_selectors,
				'cc'  => $critical_css,
				'lcp' => $lcp_data,
			]
		);
	}

	/**
	 * Checks if profile data exists for all active device types.
	 *
	 * @param string $id The profile ID to check.
	 *
	 * @return bool True if data exists for all device types, false otherwise.
	 */
	public function exists_all( $id ): bool {
		foreach ( self::get_active_devices() as $device ) {
			if ( ! $this->exists( $id, $device ) ) {
				return false;
			}
		}

		return true;
	}


	/**
	 * Gets a list of device types that are missing profile data.
	 *
	 * @param string $id The profile ID to check.
	 *
	 * @return array List of device types missing profile data.
	 */
	public function missing_devices( $id ): array {
		$missing = [];
		foreach ( self::get_active_devices() as $device ) {
			if ( ! $this->exists( $id, $device ) ) {
				$missing[] = $device;
			}
		}

		return $missing;
	}

	/**
	 * Checks if profile data exists for a specific device type.
	 *
	 * @param string $id The profile ID to check.
	 * @param int    $device The device type constant.
	 *
	 * @return bool True if data exists, false otherwise.
	 */
	public function exists( $id, $device ): bool {
		return $this->storage->get( $id . '_' . $device ) !== false;
	}

	/**
	 * Gets the current profile ID being processed.
	 *
	 * @return string The current profile ID or null if not set.
	 */
	public static function get_current_profile_id(): string {
		return self::$current_profile_id;
	}

	/**
	 * Sets the current profile ID.
	 *
	 * @param string $id The profile ID to set as current.
	 *
	 * @return void
	 */
	public static function set_current_profile_id( $id ): void {
		self::$current_profile_id = $id;
	}

	/**
	 * Gets the current profile data for all device types.
	 *
	 * @return array The current profile data.
	 */
	public static function get_current_profile_data(): array {
		return self::$current_profile_data;
	}

	/**
	 * Sets the current profile data by loading it from storage.
	 *
	 * @return array The loaded profile data.
	 * @throws \Exception If current profile ID is not set.
	 */
	public function set_current_profile_data(): array {
		if ( empty( self::get_current_profile_id() ) ) {
			throw new \Exception( 'Current profile ID is not set' );
		}
		if ( ! empty( self::$current_profile_data ) ) {
			return self::$current_profile_data;
		}
		self::$current_profile_data = [
			self::DEVICE_TYPE_MOBILE  => $this->storage->get( self::get_current_profile_id() . '_' . self::DEVICE_TYPE_MOBILE ),
			self::DEVICE_TYPE_DESKTOP => $this->storage->get( self::get_current_profile_id() . '_' . self::DEVICE_TYPE_DESKTOP ),
		];

		return self::$current_profile_data;
	}

	/**
	 * Checks if an image is in the viewport of all device types.
	 *
	 * @param int $image_id The image ID to check.
	 *
	 * @return bool True if the image is in the viewport of all device types, false otherwise.
	 */
	public function is_in_all_viewports( int $image_id ): bool {
		foreach ( self::get_active_devices() as $device ) {
			// If the data is not available for the device, return false.
			if ( empty( self::$current_profile_data[ $device ] ?? null ) ) {
				return false;
			}
			// If the image is not in the viewport of the device, return false.
			if ( ! ( self::$current_profile_data[ $device ]['af'][ $image_id ] ?? false ) ) {
				return false;
			}
		}

		// If the image is in the viewport of all device types, return true.
		return true;
	}
	public function get_critical_css(): string {
		$merged_structure = [];

		foreach ( self::get_active_devices() as $device ) {
			if ( empty( self::$current_profile_data[ $device ] ?? null ) ) {
				continue;
			}

			$css_data = self::$current_profile_data[ $device ]['cc'] ?? null;
			if ( ! $css_data ) {
				continue;
			}

			// Get structured CSS data directly
			$device_structure = $css_data['css'] ?? null;
			if ( $device_structure && is_array( $device_structure ) ) {
				$merged_structure = $this->merge_structured_css( $merged_structure, $device_structure );
			}
		}

		// Generate final CSS from merged structure
		if ( ! empty( $merged_structure ) ) {
			return $this->generate_final_css( $merged_structure );
		}

		return '';
	}

	/**
	 * Merge structured CSS from multiple devices
	 *
	 * @param array $merged Current merged structure
	 * @param array $device_structure Structure from a single device
	 * @return array Merged structure
	 */
	private function merge_structured_css( array $merged, array $device_structure ): array {
		// Merge all media queries
		$all_media_queries = array_unique( array_merge( array_keys( $merged ), array_keys( $device_structure ) ) );

		foreach ( $all_media_queries as $media_query ) {
			if ( ! isset( $merged[ $media_query ] ) ) {
				$merged[ $media_query ] = [];
			}

			$device_selectors = $device_structure[ $media_query ] ?? [];

			// Merge selectors within media query
			foreach ( $device_selectors as $selector => $properties ) {
				if ( ! isset( $merged[ $media_query ][ $selector ] ) ) {
					$merged[ $media_query ][ $selector ] = [];
				}

				// Merge properties within selector
				$merged[ $media_query ][ $selector ] = array_merge(
					$merged[ $media_query ][ $selector ],
					$properties
				);
			}
		}

		return $merged;
	}

	/**
	 * Generate final CSS from merged structured CSS
	 *
	 * @param array $merged_structure Merged CSS structure
	 * @return string Final CSS
	 */
	private function generate_final_css( array $merged_structure ): string {
		// Flatten all rules from all media queries into a single array with their order
		$all_rules = [];
		foreach ( $merged_structure as $media_query => $selectors ) {
			foreach ( $selectors as $selector => $data ) {
				if ( isset( $data['_type'] ) && isset( $data['_cssText'] ) ) {
					$all_rules[] = [
						'media'    => $media_query,
						'selector' => $selector,
						'data'     => $data,
						'order'    => $data['_order'] ?? PHP_INT_MAX,
					];
				}
			}
		}

		// Sort by order to preserve CSS cascade
		usort(
			$all_rules,
			function ( $a, $b ) {
				return $a['order'] <=> $b['order'];
			}
		);

		// Generate CSS maintaining order
		$css                 = '';
		$import_rules        = '';
		$current_media_query = null;
		$media_css           = '';

		foreach ( $all_rules as $rule ) {
			$media_query = $rule['media'];
			$selector    = $rule['selector'];
			$data        = $rule['data'];
			$css_text    = $data['_cssText'];

			// Handle default (no media query) rules
			if ( $media_query === 'default' ) {
				if ( $data['_type'] === 'import' ) {
					$import_rules .= $css_text . "\n";
				} elseif ( in_array( $data['_type'], [ 'font-face', 'keyframes', 'other' ], true ) ) {
					$css .= $css_text . "\n";
				} elseif ( $data['_type'] === 'declarations' ) {
					// Browser-parsed CSS declarations
					$css .= $selector . " {\n  " . str_replace( ';', ";\n  ", $css_text ) . "\n}\n";
				} elseif ( $data['_type'] === 'fallback' ) {
					// Fallback: use entire CSS rule as-is
					$css .= $css_text . "\n";
				}
			} else {
				// Handle media query rules
				// If we're starting a new media query, close the previous one
				if ( $current_media_query !== null && $current_media_query !== $media_query ) {
					if ( ! empty( $media_css ) ) {
						$css      .= '@media ' . $current_media_query . " {\n" . $media_css . "}\n";
						$media_css = '';
					}
				}

				$current_media_query = $media_query;

				// Add rule to current media query
				if ( $data['_type'] === 'declarations' ) {
					// Browser-parsed CSS declarations
					$media_css .= '  ' . $selector . " {\n    " . str_replace( ';', ";\n    ", $css_text ) . "\n  }\n";
				} elseif ( $data['_type'] === 'fallback' ) {
					// Fallback: use entire CSS rule as-is (but indent appropriately)
					$indented_css = preg_replace( '/^/m', '  ', $css_text );
					$media_css   .= $indented_css . "\n";
				} else {
					// Other types within media queries
					$media_css .= '  ' . $css_text . "\n";
				}
			}
		}

		// Close the last media query if any
		if ( $current_media_query !== null && ! empty( $media_css ) ) {
			$css .= '@media ' . $current_media_query . " {\n" . $media_css . "}\n";
		}

		// Prepend import rules at the very beginning
		return $import_rules . $css;
	}

	/**
	 * Checks if the LCP image is in the viewport of all device types.
	 *
	 * @param int $image_id The image ID to check.
	 *
	 * @return bool True if the LCP image is in the viewport of all device types, false otherwise.
	 */
	public function is_lcp_image_in_all_viewports( int $image_id ): bool {
		foreach ( self::get_active_devices() as $device ) {
			if ( ( ( self::$current_profile_data[ $device ]['lcp']['type'] ?? '' ) === 'img' ) && ( self::$current_profile_data[ $device ]['lcp']['imageId'] === $image_id ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Checks if an image is in the viewport of any device type.
	 *
	 * @param mixed $image_id The image ID to check.
	 *
	 * @return int|false The device type if the image is in the viewport, false otherwise.
	 */
	public function is_in_any_viewport( $image_id ) {
		foreach ( self::get_active_devices() as $device ) {
			if ( self::$current_profile_data[ $device ]['af'][ $image_id ] ?? false ) {
				return $device;
			}
		}

		return false;
	}

	/**
	 * Gets the profile data for a specific ID.
	 *
	 * @param string $id The profile ID to get data for.
	 *
	 * @return array The profile data.
	 */
	public function get_profile_data( $id ) {
		$profile_data = [];
		foreach ( self::get_active_devices() as $device ) {
			$profile_data[ $device ] = $this->storage->get( $id . '_' . $device );
		}

		return $profile_data;
	}

	/**
	 * Resets the current profile ID and data.
	 *
	 * @return void
	 */
	public static function reset_current_profile() {
		self::$current_profile_id   = null;
		self::$current_profile_data = [];
	}

	/**
	 * Gets the list of active device types supported by the profiler.
	 *
	 * @return array Array of device type constants.
	 */
	public static function get_active_devices(): array {
		return [
			self::DEVICE_TYPE_MOBILE,
			self::DEVICE_TYPE_DESKTOP,
		];
	}
}
