<?php
namespace SPC_Pro\Modules\Preload;

/**
 * Class Links
 *
 * @package SPC_Pro\Modules\Preload
 */
class Links {
	/**
	 * Links map that contains the url as the key and the url data as the value.
	 *
	 * @var array $links The array of links to preload.
	 */
	private static $links = [];
	/**
	 * Ids map of the images that are preloaded.
	 *
	 * @var array $ids The array of ids to preload.
	 */
	private static $ids = [];
	/**
	 * Add a link to the preload array.
	 *
	 * @param array $url_data The array of url data to add.
	 */
	public static function add_link( array $url_data ) {
		if ( ! isset( $url_data['url'] ) ) {
			return;
		}
		self::$links[ crc32( $url_data['url'] ) ] = $url_data;
	}

	/**
	 * Add an id to the preload array.
	 *
	 * @param int    $id The id to add.
	 * @param string $priority The priority of the id.
	 */
	public static function add_id( int $id, string $priority = 'auto' ) {
		self::$ids[ $id ] = $priority;
	}

	/**
	 * Check if an id is preloaded.
	 *
	 * @param int $id The id to check.
	 * @return string|false The priority of the id or false if it is not preloaded.
	 */
	public static function is_preloaded( int $id ) {
		return self::$ids[ $id ] ?? false;
	}

	/**
	 * Get the links.
	 *
	 * @return array The links.
	 */
	public static function get_links(): array {
		return self::$links;
	}

	/**
	 * Get the links count.
	 *
	 * @return int The links count.
	 */
	public static function get_links_count(): int {
		return count( self::$links );
	}

	/**
	 * Get the links html.
	 *
	 * @return string The links html.
	 */
	public static function get_links_html(): string {
		// generate image preload links for all links
		$links = [];
		foreach ( self::$links as $link ) {
			$url = esc_url( $link['url'] );
			if ( empty( $url ) ) {
				continue;
			}
			$preload = '<link rel="preload" media="screen" href="' . $url . '" ';
			if ( isset( $link['priority'] ) && $link['priority'] !== 'auto' ) {
				$preload .= 'fetchpriority="' . esc_attr( $link['priority'] ) . '" ';
			}
			// Add imagesrcset if available
			if ( isset( $link['srcset'] ) && ! empty( $link['srcset'] ) ) {
				$preload .= 'imagesrcset="' . esc_attr( $link['srcset'] ) . '" ';
			}
			// Add imagesizes if available
			if ( isset( $link['sizes'] ) && ! empty( $link['sizes'] ) ) {
				$preload .= 'imagesizes="' . esc_attr( $link['sizes'] ) . '" ';
			}
			// Complete the preload tag
			$preload .= 'as="image">';

			$links[] = $preload;
		}
		return implode( "\n", $links ) . "\n";
	}
}
