<?php

namespace SPC_Pro\Services;

use SPC\Modules\Module_Interface;
use SPC\Services\Cloudflare_Rule;
use SPC_Pro\Builders\Transform_Rule;
use SPC_Pro\Constants;
use SW_CLOUDFLARE_PAGECACHE;

class Cloudflare_Transform_Rule extends Cloudflare_Rule implements Module_Interface {
	/**
	 * @var Transform_Rule $rule_builder Rule builder.
	 */
	private $rule_builder;

	/**
	 * Constructor
	 *
	 * @param SW_CLOUDFLARE_PAGECACHE $plugin The plugin main instance.
	 */
	public function __construct( SW_CLOUDFLARE_PAGECACHE $plugin ) {
		parent::__construct( $plugin );

		$this->rule_builder = new Transform_Rule();
		$this->set_rule_type( self::RULE_TYPE_TRANSFORM );
	}

	public function get_ruleset_id_setting_slug(): string {
		return Constants::SETTING_TRANSFORM_RULESET_ID;
	}

	public function get_rule_id_setting_slug(): string {
		return Constants::SETTING_TRANSFORM_RULE_ID;
	}

	public function init() {
		add_action( 'swcfpc_before_cache_rule_sync_start', [ $this, 'sync_rule_if_diff' ] );
		add_action( 'swcfpc_after_create_cache_rule', [ $this, 'create_transform_rule' ] );
		add_action( 'swcfpc_after_delete_cache_rule', [ $this, 'delete_transform_rule' ] );
	}

	public function sync_rule_if_diff() {
		$this->log( 'sync_transform_rule', 'Syncing transform rule...' );
		$cloudflare_handler = $this->plugin->get_cloudflare_handler();
		if ( ! $cloudflare_handler->is_enabled() ) {
			$this->log( 'sync_transform_rule', 'Cloudflare is not enabled. Skipping transform rule sync.' );
			return;
		}

		if ( empty( $this->rule_id ) && empty( $this->ruleset_id ) ) {
			$this->log( 'sync_transform_rule', 'Rule and ruleset not defined. Creating transform rule...' );
			$this->create_transform_rule();

			return;
		}

		$existing_rule = $this->get_rule();

		// There's no existing rule, or the expression is not explicitly set.
		if (
			! isset( $existing_rule['expression'] ) ||
			! isset( $existing_rule['action_parameters'], $existing_rule['action_parameters']['uri'], $existing_rule['action_parameters']['uri']['query'], $existing_rule['action_parameters']['uri']['query']['expression'] )
		) {
			$this->log( 'sync_transform_rule', 'Existing rule not found.' );
			return;
		}

		// The rule is identical to what we have already.
		if (
			$existing_rule['expression'] === $this->rule_builder->build() &&
			$existing_rule['action_parameters']['uri']['query']['expression'] === $this->rule_builder->get_regex_expression()
		) {
			$this->log( 'sync_transform_rule', 'Rule is identical. Skipping sync.' );
			return;
		}

		$this->update_rule();
	}

	/**
	 * Create the transform rule.
	 *
	 * @return void
	 */
	public function create_transform_rule() {
		if ( ! $this->plugin->get_cloudflare_handler()->has_cache_rule() ) {
			$this->log( 'create_transform_rule', 'Cloudflare is not enabled or cache rule is not set. Skipping transform rule creation' );

			return;
		}

		$this->ruleset_id = $this->get_ruleset_id();

		if ( empty( $this->ruleset_id ) ) {
			$this->log( 'create_transform_rule', 'Ruleset not defined, creating...' );

			$this->ruleset_id = $this->create_ruleset();
			$this->plugin->set_single_config( $this->get_ruleset_id_setting_slug(), $this->ruleset_id );
		}

		if ( empty( $this->ruleset_id ) ) {
			$this->log( 'create_transform_rule', 'Failed to create transform ruleset.' );

			return;
		}

		$this->log( 'create_transform_rule', 'Creating transform rule...' );

		$this->rule_id = $this->create_rule();
		$this->plugin->set_single_config( $this->get_rule_id_setting_slug(), $this->rule_id );
		$this->plugin->update_config();

		if ( empty( $this->rule_id ) ) {
			update_option( \SPC\Constants::KEY_RULE_UPDATE_FAILED, true );
			$this->log( 'create_transform_rule', 'Failed to create transform rule.' );
		} else {
			$this->log( 'create_transform_rule', 'Created transform rule.' );
		}
	}

	/**
	 * Delete the transform rule.
	 *
	 * @return void
	 */
	public function delete_transform_rule() {
		$status = $this->delete_rule();

		if ( ! $status ) {

			return;
		}

		$this->plugin->set_single_config( $this->get_rule_id_setting_slug(), '' );
		$this->plugin->set_single_config( $this->get_ruleset_id_setting_slug(), '' );
		$this->plugin->update_config();
	}

	/**
	 * Gets the rule arguments. To be used when creating the rule.
	 *
	 * @return array
	 */
	protected function get_rule_args(): array {
		return [
			'action'            => 'rewrite',
			'action_parameters' => [
				'uri' => [
					'query' => [
						'expression' => $this->rule_builder->get_regex_expression(),
					],
				],
			],
			'description'       => $this->build_rule_description(),
			'enabled'           => true,
			'expression'        => $this->rule_builder->build(),
		];
	}
}
