<?php
/**
 * PayPal Settings Class.
 *
 * @package sureforms-pro
 * @since 2.4.0
 */

namespace SRFM_Pro\Inc\Business\Payments\PayPal;

use SRFM\Inc\Helper;
use SRFM_Pro\Inc\Business\Payments\PayPal\Helper as PayPal_Helper;
use SRFM_Pro\Inc\Helper as Pro_Helper;
use SRFM_Pro\Inc\Traits\Get_Instance;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * PayPal Settings.
 *
 * @since 2.4.0
 */
class Settings {
	use Get_Instance;

	/**
	 * Constructor.
	 *
	 * @since 2.4.0
	 */
	public function __construct() {
		// Register hooks.
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );

		// Register AJAX handlers.
		add_action( 'wp_ajax_srfm_paypal_get_connect_url', [ $this, 'ajax_get_paypal_connect_url' ] );
		add_action( 'wp_ajax_srfm_paypal_disconnect', [ $this, 'ajax_disconnect_paypal' ] );
		add_action( 'wp_ajax_srfm_paypal_onboard_complete', [ $this, 'handle_paypal_onboard_complete' ] );
		add_action( 'wp_ajax_srfm_paypal_create_webhook', [ $this, 'ajax_create_paypal_webhook' ] );
		add_action( 'wp_ajax_srfm_paypal_delete_webhook', [ $this, 'ajax_delete_paypal_webhook' ] );

		// Extend payments data with PayPal settings.
		add_filter( 'srfm_admin_localize_payments_data', [ $this, 'add_paypal_to_payments_data' ] );

		// Register PayPal in payment methods registry.
		add_filter( 'srfm_payment_methods_registry', [ $this, 'register_paypal_payment_method' ] );

		// Enqueue PayPal payment provider scripts.
		add_action( 'srfm_enqueue_block_scripts', [ $this, 'enqueue_paypal_scripts' ], 10, 1 );

		// Add PayPal onboarding URLs to payment settings.
		add_filter( 'srfm_get_payments_settings', [ $this, 'add_paypal_connect_urls_to_settings' ], 10, 1 );
	}

	/**
	 * Enqueue PayPal payment provider scripts.
	 *
	 * @since 2.4.0
	 * @param array $args Array containing block_name and attr.
	 * @return void
	 */
	public function enqueue_paypal_scripts( $args ) {
		$block_name = $args['block_name'] ?? '';
		$attr       = isset( $args['attr'] ) && is_array( $args['attr'] ) ? $args['attr'] : [];

		if ( 'payment' !== $block_name ) {
			return;
		}

		$payment_methods = isset( $attr['paymentMethods'] ) && is_array( $attr['paymentMethods'] ) ? $attr['paymentMethods'] : [ 'stripe' ];
		$has_paypal      = in_array( 'paypal', $payment_methods, true );

		if ( ! $has_paypal ) {
			return;
		}

		// Check if PayPal is connected before loading scripts.
		if ( ! PayPal_Helper::is_paypal_connected() ) {
			return;
		}

		// Enqueue PayPal SDK.
		$client_id = PayPal_Helper::get_paypal_client_id();
		$currency  = PayPal_Helper::get_currency();

		if ( ! empty( $client_id ) ) {
			// Determine payment type and intent.
			$payment_type = isset( $attr['paymentType'] ) && is_string( $attr['paymentType'] ) ? $attr['paymentType'] : 'one-time';

			$query_args = [
				'client-id'  => $client_id,
				'currency'   => strtoupper( $currency ),
				'components' => 'buttons',
			];

			// Add vault parameter only for subscriptions (required for recurring billing).
			if ( 'subscription' === $payment_type ) {
				$query_args['vault']  = 'true';
				$query_args['intent'] = 'subscription';
			} else {
				$query_args['intent'] = 'capture';
			}

			$sdk_url = add_query_arg( $query_args, 'https://www.paypal.com/sdk/js' );

			// phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion -- PayPal SDK URL includes version in query args.
			wp_enqueue_script(
				SRFM_PRO_SLUG . '-paypal-sdk',
				$sdk_url,
				[],
				null,
				true
			);

			// Add data-namespace attribute for conflict avoidance.
			add_filter(
				'script_loader_tag',
				static function ( $tag, $handle ) {
					if ( SRFM_PRO_SLUG . '-paypal-sdk' === $handle ) {
						$tag = str_replace( ' src', ' data-namespace="paypal_sdk" src', $tag );
					}
					return $tag;
				},
				10,
				2
			);
		}

		wp_enqueue_script(
			SRFM_PRO_SLUG . '-paypal-frontend',
			SRFM_PRO_URL . 'dist/package/business/paypal-frontend.js',
			[ SRFM_PRO_SLUG . '-paypal-sdk' ], // Only depends on PayPal SDK now.
			SRFM_PRO_VER,
			true
		);

		// Localize script for PayPal payment functionality.
		wp_localize_script(
			SRFM_PRO_SLUG . '-paypal-frontend',
			'srfm_paypal_frontend',
			[
				'ajax_url'     => admin_url( 'admin-ajax.php' ),
				'paypal_nonce' => wp_create_nonce( 'srfm_paypal_payment_nonce_frontend' ),
				'messages'     => [
					'invalid_payment_amount'    => __( 'Invalid payment amount. Please try again.', 'sureforms-pro' ),
					'invalid_email'             => __( 'Please enter a valid email address.', 'sureforms-pro' ),
					'paypal_not_configured'     => __( 'PayPal is not configured. Please contact the site administrator for assistance or refresh the page and try again.', 'sureforms-pro' ),
					'fail_to_process_payment'   => __( 'Failed to process the payment. Please try again or contact the site administrator for assistance.', 'sureforms-pro' ),
					'payment_cancelled'         => __( 'Payment cancelled. Please try again or contact the site administrator for assistance.', 'sureforms-pro' ),
					'fail_to_initialize_paypal' => __( 'Failed to initialize PayPal. Please try again or contact the site administrator for assistance.', 'sureforms-pro' ),
					'window_closed'             => __( 'The window was closed. Please try again or contact the site administrator for assistance.', 'sureforms-pro' ),
				],
			]
		);
	}

	/**
	 * Enqueue PayPal payment settings assets.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	public function enqueue_assets() {
		$this->enqueue_form_editor_assets();

		$current_screen = get_current_screen();

		if ( is_null( $current_screen ) ) {
			return;
		}

		// Enqueue only on SureForms global settings page.
		$is_screen_sureforms_form_settings = Helper::validate_request_context( 'sureforms_form_settings', 'page' );

		if ( ! $is_screen_sureforms_form_settings ) {
			return;
		}

		$this->enqueue_scripts();
	}

	/**
	 * AJAX handler to get PayPal connect URL.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	public function ajax_get_paypal_connect_url() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'sureforms-pro' ) ] );
		}

		// Verify nonce for AJAX request (nonce should be sent as 'nonce' in POST).
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'srfm_paypal_admin' ) ) {
			wp_send_json_error( [ 'message' => __( 'Security check failed.', 'sureforms-pro' ) ] );
		}

		// Get mode from request (test/sandbox or live).
		$mode = isset( $_POST['mode'] ) && 'live' === $_POST['mode'] ? 'live' : 'sandbox';

		// Call the existing method with mode parameter.
		$response = $this->get_paypal_signup_url( $mode );

		// Return AJAX response.
		if ( is_array( $response ) && isset( $response['success'] ) && $response['success'] ) {
			wp_send_json_success( $response );
		} else {
			$error_message = $response['message'] ?? __( 'Failed to generate PayPal connect URL.', 'sureforms-pro' );
			wp_send_json_error( [ 'message' => $error_message ] );
		}
	}

	/**
	 * AJAX handler to disconnect PayPal account.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	public function ajax_disconnect_paypal() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'sureforms-pro' ) ] );
		}

		// Verify nonce for AJAX request (nonce should be sent as 'nonce' in POST).
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'srfm_paypal_admin' ) ) {
			wp_send_json_error( [ 'message' => __( 'Security check failed.', 'sureforms-pro' ) ] );
		}

		// Call the existing method.
		$updated = $this->disconnect_paypal();

		// Return AJAX response.
		if ( $updated ) {
			wp_send_json_success(
				[
					'message' => __( 'PayPal account disconnected successfully.', 'sureforms-pro' ),
					'updated' => $updated,
				]
			);
		} else {
			wp_send_json_error( [ 'message' => __( 'Failed to disconnect PayPal account.', 'sureforms-pro' ) ] );
		}
	}

	/**
	 * Handle PayPal onboarding completion via AJAX.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	public function handle_paypal_onboard_complete() {
		// Validate required parameters.
		if ( ! isset( $_POST['authCode'] ) || ! isset( $_POST['sharedId'] ) ) {
			wp_send_json_error( [ 'message' => __( 'Missing required parameters.', 'sureforms-pro' ) ] );
		}

		// Verify nonce for AJAX request (nonce should be sent as 'nonce' in POST).
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'srfm_paypal_admin' ) ) {
			wp_send_json_error( [ 'message' => __( 'Security check failed.', 'sureforms-pro' ) ] );
		}

		$auth_code = sanitize_text_field( wp_unslash( $_POST['authCode'] ) );
		$shared_id = sanitize_text_field( wp_unslash( $_POST['sharedId'] ) );
		$mode      = isset( $_POST['mode'] ) && 'live' === $_POST['mode'] ? 'live' : 'sandbox';

		try {
			// Step 1: Exchange authCode and sharedId for access token.
			$access_token = $this->get_onboarding_access_token( $auth_code, $shared_id, $mode );

			if ( empty( $access_token ) ) {
				wp_send_json_error( [ 'message' => __( 'Failed to get access token from PayPal.', 'sureforms-pro' ) ] );
			}

			// Step 2: Fetch and store merchant credentials.
			$credentials_result = $this->fetch_and_store_credentials( $mode, $access_token );

			if ( ! $credentials_result['status'] ) {
				wp_send_json_error(
					[
						'message' => $credentials_result['message']
							?? __( 'Failed to fetch PayPal credentials.', 'sureforms-pro' ),
					]
				);
			}

			// Step 3: Create webhook for the connected mode.
			Webhook::create_webhook( $mode );

			wp_send_json_success(
				[
					'message' => __( 'PayPal account connected successfully!', 'sureforms-pro' ),
					'mode'    => $mode,
				]
			);

		} catch ( \Exception $e ) {
			wp_send_json_error(
				[
					'message' => $e->getMessage(),
				]
			);
		}
	}

	/**
	 * Get PayPal connect URL for first party integration.
	 *
	 * @param string $mode PayPal mode (live or sandbox). Defaults to 'sandbox'.
	 * @since 2.4.0
	 * @return array Response array with success status and URL.
	 */
	public function get_paypal_signup_url( $mode = 'sandbox' ) {
		// Normalize mode: 'test' becomes 'sandbox', 'live' stays 'live'.
		$environment = 'live' === $mode ? 'live' : 'sandbox';

		// Generate tracking ID and seller nonce for security.
		$tracking_id  = $this->generate_random_string( 32 );
		$seller_nonce = $this->get_seller_nonce( $environment );

		// Build return URL with proper args.
		$redirect_url = PayPal_Helper::get_paypal_settings_url();
		$return_url   = add_query_arg(
			[
				'srfm-paypal-connect-nonce' => $seller_nonce,
				'paypal-process-onboard'    => '1',
				'environment'               => $environment,
			],
			$redirect_url
		);

		$args2 = [
			'environment'  => $environment,
			'tracking_id'  => $tracking_id,
			'seller_nonce' => $seller_nonce,
			'return_url'   => $return_url,
		];

		$result = $this->get_paypal_signup_url_from_middle_ware( $args2 );

		// Check if result has error.
		if ( ! isset( $result['success'] ) || ! $result['success'] ) {
			return $result;
		}

		// Save the partner id in the option.
		if ( isset( $result['partner_id'] ) ) {
			update_option( 'srfm_paypal_partner_id_' . $environment, $result['partner_id'] );
		}

		return [
			'success' => true,
			'url'     => $result['signup_link'],
		];
	}

	/**
	 * Get PayPal connect URL - internal method using middleware.
	 *
	 * @param array $args2 Arguments array.
	 * @since 2.4.0
	 * @return array Response array.
	 */
	public function get_paypal_signup_url_from_middle_ware( $args2 ) {
		// Validate required arguments.
		if (
			! isset( $args2['seller_nonce'] ) ||
			! isset( $args2['environment'] ) ||
			! isset( $args2['tracking_id'] ) ||
			! isset( $args2['return_url'] )
		) {
			return [
				'success' => false,
				'type'    => 'invalid_request',
				'message' => 'Invalid Request',
			];
		}

		// Prepare request data for middleware.
		$request_data = [
			'seller_nonce' => $args2['seller_nonce'],
			'environment'  => $args2['environment'],
			'tracking_id'  => $args2['tracking_id'],
			'return_url'   => $args2['return_url'],
		];

		$encoded_data = wp_json_encode( $request_data );
		if ( false === $encoded_data || '' === $encoded_data ) {
			return [
				'success' => false,
				'type'    => 'invalid_request',
				'message' => __( 'Failed to encode request data.', 'sureforms-pro' ),
			];
		}

		$encoded_data = base64_encode( $encoded_data ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Required for PayPal Basic Auth header

		// Send request to middleware.
		$response = wp_remote_post(
			PayPal_Helper::middle_ware_base_url() . 'connect-url/create',
			[
				'body'    => $encoded_data,
				'headers' => [
					'Content-Type' => 'application/json',
				],
				'timeout' => 30,
			]
		);

		// Handle wp_remote_post errors.
		if ( is_wp_error( $response ) ) {
			return [
				'success' => false,
				'type'    => 'middleware_error',
				'message' => $response->get_error_message(),
			];
		}

		// Get response body.
		$body = wp_remote_retrieve_body( $response );
		$code = wp_remote_retrieve_response_code( $response );

		// Decode JSON response.
		$decoded = json_decode( $body, true );

		// Handle JSON decode errors.
		if ( null === $decoded && JSON_ERROR_NONE !== json_last_error() ) {
			return [
				'success' => false,
				'type'    => 'invalid_response',
				'message' => 'Failed to parse middleware response: ' . json_last_error_msg(),
			];
		}

		if ( ! is_array( $decoded ) ) {
			return [
				'success' => false,
				'type'    => 'invalid_response',
				'message' => 'Failed to parse middleware response: ' . json_last_error_msg(),
			];
		}

		// Handle non-200 HTTP responses.
		if ( 200 !== $code ) {
			return [
				'success' => false,
				'type'    => $decoded['code'] ?? 'http_error',
				'message' => $decoded['message'] ?? __( 'Middleware Error', 'sureforms-pro' ),
			];
		}

		// Validate response structure.
		if ( ! isset( $decoded['success'] ) || ! $decoded['success'] ) {
			return [
				'success' => false,
				'type'    => $decoded['code'] ?? 'middleware_error',
				'message' => $decoded['message'] ?? __( 'Middleware request failed', 'sureforms-pro' ),
			];
		}

		// Extract links from response.
		if ( ! isset( $decoded['links'] ) || ! is_array( $decoded['links'] ) ) {
			return [
				'success' => false,
				'type'    => 'generate_signup_link',
				'message' => __( 'Links not found in middleware response', 'sureforms-pro' ),
			];
		}

		// Find action_url link and extract signup URL.
		$signup_link = null;
		foreach ( $decoded['links'] as $link ) {
			if ( isset( $link['rel'] ) && 'action_url' === $link['rel'] && ! empty( $link['href'] ) ) {
				// Add &displayMode=minibrowser as requested.
				$signup_link = $link['href'] . '&displayMode=minibrowser';
				break;
			}
		}

		// Return error if no signup link found.
		if ( empty( $signup_link ) ) {
			return [
				'success' => false,
				'type'    => 'generate_signup_link',
				'message' => __( 'Signup link not found in middleware response', 'sureforms-pro' ),
			];
		}

		// Return success response with partner_id and signup_link.
		return [
			'success'     => true,
			'partner_id'  => $decoded['partner_id'] ?? '',
			'signup_link' => $signup_link,
		];
	}

	/**
	 * Disconnect PayPal account.
	 *
	 * @since 2.4.0
	 * @return bool True on success, false on failure.
	 */
	public function disconnect_paypal() {
		$settings = PayPal_Helper::get_all_paypal_settings();
		if ( ! is_array( $settings ) ) {
			$settings = PayPal_Helper::get_default_paypal_settings();
		}

		$settings['paypal_sandbox_connected']  = false;
		$settings['paypal_live_connected']     = false;
		$settings['paypal_account_id']         = '';
		$settings['paypal_account_email']      = '';
		$settings['paypal_live_client_id']     = '';
		$settings['paypal_live_client_secret'] = '';
		$settings['paypal_test_client_id']     = '';
		$settings['paypal_test_client_secret'] = '';
		$settings['webhook_test_secret']       = '';
		$settings['webhook_test_url']          = '';
		$settings['webhook_test_id']           = '';
		$settings['webhook_live_secret']       = '';
		$settings['webhook_live_url']          = '';
		$settings['webhook_live_id']           = '';
		$settings['account_name']              = '';

		PayPal_Helper::update_all_paypal_settings( $settings );

		return true;
	}

	/**
	 * Add PayPal settings to the localized payments data.
	 *
	 * This filter extends the global payments data with PayPal-specific settings,
	 * making them available in JavaScript via window.srfm_admin.payments.
	 *
	 * @param array $payments_data The existing payments data.
	 * @since 2.4.0
	 * @return array The extended payments data with PayPal settings.
	 */
	public function add_paypal_to_payments_data( $payments_data ) {
		// Add payment mode to payments data.
		if ( ! isset( $payments_data['payment_mode'] ) ) {
			$payments_data['payment_mode'] = \SRFM\Inc\Payments\Payment_Helper::get_payment_mode();
		}

		// Get all PayPal settings including webhook details.
		$paypal_settings = PayPal_Helper::get_all_paypal_settings();

		// Get PayPal supported currency codes.
		$paypal_currencies_data = PayPal_Helper::get_all_paypal_currencies_data();
		$supported_currencies   = array_keys( $paypal_currencies_data );

		// Add PayPal settings with webhook details and supported currencies.
		$payments_data['paypal'] = [
			'paypal_sandbox_connected' => PayPal_Helper::is_paypal_sandbox_connected(),
			'paypal_live_connected'    => PayPal_Helper::is_paypal_live_connected(),
			'webhook_test_id'          => $paypal_settings['webhook_test_id'] ?? '',
			'webhook_test_url'         => $paypal_settings['webhook_test_url'] ?? '',
			'webhook_live_id'          => $paypal_settings['webhook_live_id'] ?? '',
			'webhook_live_url'         => $paypal_settings['webhook_live_url'] ?? '',
			'account_name'             => $paypal_settings['account_name'] ?? '',
			'paypal_account_id'        => $paypal_settings['paypal_account_id'] ?? '',
			'supported_currencies'     => $supported_currencies,
			'settings_url'             => admin_url( 'admin.php?page=sureforms_form_settings&tab=payments-settings&subpage=payment-methods&gateway=paypal' ),
		];

		return $payments_data;
	}

	/**
	 * Add PayPal onboarding URLs to payment settings.
	 *
	 * Pre-fetches the PayPal connect URLs for both sandbox and live modes
	 * and adds them to payment settings to avoid loading delay when the component renders.
	 *
	 * @param array $payments_settings The existing payment settings.
	 * @since 2.4.0
	 * @return array The extended payment settings with PayPal connect URLs.
	 */
	public function add_paypal_connect_urls_to_settings( $payments_settings ) {
		// Initialize PayPal settings array if not exists.
		if ( ! isset( $payments_settings['payment_settings']['paypal'] ) || ! is_array( $payments_settings['payment_settings']['paypal'] ) ) {
			$payments_settings['payment_settings']['paypal'] = [];
		}

		// Fetch sandbox URL if not connected.
		if ( ! PayPal_Helper::is_paypal_sandbox_connected() ) {
			$sandbox_response = $this->get_paypal_signup_url( 'sandbox' );
			if ( is_array( $sandbox_response ) && isset( $sandbox_response['success'] ) && $sandbox_response['success'] && isset( $sandbox_response['url'] ) ) {
				$payments_settings['payment_settings']['paypal']['paypal_sandbox_connect_url'] = $sandbox_response['url'];
			}
		}

		// Fetch live URL if not connected.
		if ( ! PayPal_Helper::is_paypal_live_connected() ) {
			$live_response = $this->get_paypal_signup_url( 'live' );
			if ( is_array( $live_response ) && isset( $live_response['success'] ) && $live_response['success'] && isset( $live_response['url'] ) ) {
				$payments_settings['payment_settings']['paypal']['paypal_live_connect_url'] = $live_response['url'];
			}
		}

		return $payments_settings;
	}

	/**
	 * Register PayPal in the payment methods registry.
	 *
	 * @param array $methods Existing payment methods.
	 * @since 2.4.0
	 * @return array Modified payment methods array.
	 */
	public function register_paypal_payment_method( $methods ) {
		// Check if PayPal is connected before registering.
		if ( ! PayPal_Helper::is_paypal_connected() ) {
			return $methods;
		}

		ob_start();
		?>
			<div class="srfm-paypal-button"></div>
			<div class="srfm-paypal-placeholder-message">
				<?php esc_html_e( 'PayPal selected for checkout', 'sureforms-pro' ); ?>
			</div>
		<?php
		$place_holder_content = ob_get_clean();
		$place_holder_content = ! empty( $place_holder_content ) ? $place_holder_content : '';

		// Add PayPal to the registry.
		$methods['paypal'] = [
			'id'                   => 'paypal',
			'label'                => __( 'PayPal', 'sureforms-pro' ),
			'description'          => __( 'Pay with PayPal account', 'sureforms-pro' ),
			'icon'                 => 'paypal',
			'enabled'              => true,
			'container_class'      => 'srfm-paypal-button-container',
			'place_holder_content' => $place_holder_content,
		];

		return $methods;
	}

	/**
	 * AJAX handler to create PayPal webhook.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	public function ajax_create_paypal_webhook() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'sureforms-pro' ) ] );
		}

		// Verify nonce for AJAX request.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'srfm_paypal_admin' ) ) {
			wp_send_json_error( [ 'message' => __( 'Security check failed.', 'sureforms-pro' ) ] );
		}

		// Get mode from request (test or live).
		$mode = isset( $_POST['mode'] ) && 'live' === $_POST['mode'] ? 'live' : 'test';

		// Create webhook.
		$result = Webhook::create_webhook( $mode );

		if ( ! empty( $result['success'] ) ) {
			// Format response to match JavaScript expectations.
			// JavaScript expects: response.data.webhook_details[mode].
			wp_send_json_success(
				[
					'message'         => $result['message'] ?? __( 'Webhook created successfully.', 'sureforms-pro' ),
					'webhook_details' => [
						$mode => [
							'webhook_id'     => $result['webhook_id'] ?? '',
							'webhook_url'    => $result['webhook_url'] ?? '',
							'webhook_secret' => '', // PayPal uses signature-based verification, not a shared secret.
						],
					],
				]
			);
		} else {
			wp_send_json_error(
				[
					'message' => $result['message'] ?? __( 'Failed to create webhook.', 'sureforms-pro' ),
				]
			);
		}
	}

	/**
	 * AJAX handler to delete PayPal webhook.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	public function ajax_delete_paypal_webhook() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => __( 'Insufficient permissions.', 'sureforms-pro' ) ] );
		}

		// Verify nonce for AJAX request.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'srfm_paypal_admin' ) ) {
			wp_send_json_error( [ 'message' => __( 'Security check failed.', 'sureforms-pro' ) ] );
		}

		// Get mode from request (test or live).
		$mode = isset( $_POST['mode'] ) && 'live' === $_POST['mode'] ? 'live' : 'test';

		// Delete webhook.
		$result = Webhook::delete_webhook( $mode );

		if ( ! empty( $result['success'] ) ) {
			wp_send_json_success(
				[
					'message' => $result['message'] ?? __( 'Webhook deleted successfully.', 'sureforms-pro' ),
				]
			);
		} else {
			wp_send_json_error(
				[
					'message' => $result['message'] ?? __( 'Failed to delete webhook.', 'sureforms-pro' ),
				]
			);
		}
	}

	/**
	 * Enqueue PayPal form editor assets.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	private function enqueue_form_editor_assets() {
		$current_screen = get_current_screen();

		$is_page_id_payment_page = isset( $current_screen->id ) && 'sureforms_page_sureforms_payments' === $current_screen->id;
		$is_block_editor         = ! is_null( $current_screen ) && $current_screen->is_block_editor();

		// Check if we're in the block editor.
		if ( $is_page_id_payment_page || $is_block_editor ) {
			// Get asset file path and data.
			$script_dep_path = SRFM_PRO_DIR . 'dist/package/business/paypal-form-editor.asset.php';
			$script_dep_data = file_exists( $script_dep_path )
				? include $script_dep_path
				: [
					'dependencies' => [],
					'version'      => SRFM_PRO_VER,
				];

			// Ensure dependencies are arrays.
			$script_dep = is_array( $script_dep_data['dependencies'] ) ? $script_dep_data['dependencies'] : [];

			// Enqueue script.
			wp_enqueue_script(
				SRFM_PRO_SLUG . '-paypal-form-editor', // Handle.
				SRFM_PRO_URL . 'dist/package/business/paypal-form-editor.js',
				$script_dep, // Dependencies.
				$script_dep_data['version'], // Version.
				true // Enqueue in footer.
			);

			// Register script translations.
			Pro_Helper::register_script_translations( SRFM_PRO_SLUG . '-paypal-form-editor' );
		}
	}

	/**
	 * Enqueue PayPal payment settings scripts.
	 *
	 * @since 2.4.0
	 * @return void
	 */
	private function enqueue_scripts() {
		$scripts = [
			[
				'unique_file'        => 'paypal-payment-settings',
				'unique_handle'      => 'paypal-payment-settings',
				'extra_dependencies' => [],
			],
		];

		foreach ( $scripts as $script ) {
			$script_dep_path = SRFM_PRO_DIR . 'dist/package/business/' . $script['unique_file'] . '.asset.php';
			$script_dep_data = file_exists( $script_dep_path )
				? include $script_dep_path
				: [
					'dependencies' => [],
					'version'      => SRFM_PRO_VER,
				];

			// Ensure dependencies are arrays.
			$script_dep = array_merge(
				is_array( $script_dep_data['dependencies'] ) ? $script_dep_data['dependencies'] : [],
				is_array( $script['extra_dependencies'] ) ? $script['extra_dependencies'] : []
			);

			// Enqueue script.
			wp_enqueue_script(
				SRFM_PRO_SLUG . '-' . $script['unique_handle'], // Handle.
				SRFM_PRO_URL . 'dist/package/business/' . $script['unique_file'] . '.js',
				$script_dep, // Dependencies.
				$script_dep_data['version'], // Version.
				true // Enqueue in footer.
			);

			// Localize script with PayPal admin nonce (for AJAX use).
			wp_localize_script(
				SRFM_PRO_SLUG . '-' . $script['unique_handle'],
				'srfm_paypal',
				[
					'admin_nonce' => wp_create_nonce( 'srfm_paypal_admin' ),
				]
			);
		}

		// Register script translations.
		Pro_Helper::register_script_translations( SRFM_PRO_SLUG . '-' . $script['unique_handle'] );
	}

	/**
	 * Generate random string.
	 *
	 * @param int $length Random string length.
	 * @since 2.4.0
	 * @return string
	 */
	private function generate_random_string( $length = 64 ) {
		$chars  = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
		$max    = strlen( $chars ) - 1;
		$string = '';
		for ( $i = 0; $i < $length; $i++ ) {
			$string .= $chars[ wp_rand( 0, $max ) ];
		}
		return $string;
	}

	/**
	 * Create and get seller nonce for PayPal Connect.
	 *
	 * @param string $mode PayPal mode (live or sandbox).
	 * @since 2.4.0
	 * @return string
	 */
	private function get_seller_nonce( $mode ) {
		$option_name = 'srfm_paypal_connect_seller_nonce_' . $mode;
		$get_option  = get_option( $option_name );

		if ( is_string( $get_option ) && ! empty( $get_option ) ) {
			return $get_option;
		}
			$seller_nonce = $this->generate_random_string();
			update_option( $option_name, $seller_nonce );
			return $seller_nonce;
	}

	/**
	 * Get access token for onboarding by exchanging authCode and sharedId.
	 *
	 * @param string $auth_code PayPal auth code.
	 * @param string $shared_id PayPal shared id.
	 * @param string $mode      PayPal mode (sandbox or live).
	 * @since 2.4.0
	 * @return string Access token or empty string on failure.
	 */
	private function get_onboarding_access_token( $auth_code, $shared_id, $mode ) {
		$option_name  = 'srfm_paypal_connect_seller_nonce_' . $mode;
		$seller_nonce = get_option( $option_name );

		// Delete the seller nonce after use for security.
		delete_option( $option_name );

		// Determine PayPal API base URL.
		$base_url = 'live' === $mode
			? 'https://api-m.paypal.com/'
			: 'https://api-m.sandbox.paypal.com/';

		// Prepare request to exchange authorization code for access token.
		$args = [
			'headers' => [
				'Authorization'                 => 'Basic ' . base64_encode( $shared_id ), // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Required for PayPal Basic Auth header
				'Content-Type'                  => 'application/x-www-form-urlencoded',
				'PayPal-Partner-Attribution-Id' => PayPal_Helper::paypal_bn_code(),
			],
			'body'    => [
				'grant_type'    => 'authorization_code',
				'code'          => $auth_code,
				'code_verifier' => $seller_nonce,
			],
		];

		$response = wp_remote_post( $base_url . 'v1/oauth2/token', $args );

		if ( is_wp_error( $response ) ) {
			return '';
		}

		$body             = wp_remote_retrieve_body( $response );
		$decoded_response = json_decode( $body );

		if ( ! is_object( $decoded_response ) ) {
			return '';
		}

		if ( ! empty( $decoded_response->access_token ) ) {
			return $decoded_response->access_token;
		}

		return '';
	}

	/**
	 * Fetch merchant credentials and store them in settings.
	 *
	 * @param string $mode         PayPal mode (sandbox or live).
	 * @param string $access_token PayPal access token.
	 * @since 2.4.0
	 * @return array Status array with 'status' key and optional 'message' key.
	 */
	private function fetch_and_store_credentials( $mode, $access_token ) {
		$result = [ 'status' => false ];

		// Determine PayPal API base URL.
		$base_url = 'live' === $mode
			? 'https://api-m.paypal.com/'
			: 'https://api-m.sandbox.paypal.com/';

		// Prepare request headers.
		$args = [
			'headers' => [
				'Authorization'                 => 'Bearer ' . $access_token,
				'Content-Type'                  => 'application/json',
				'PayPal-Partner-Attribution-Id' => PayPal_Helper::paypal_bn_code(),
			],
		];

		// Get partner ID.
		$partner_id = get_option( 'srfm_paypal_partner_id_' . $mode );

		if ( empty( $partner_id ) || ! is_string( $partner_id ) ) {
			$result['message'] = __( 'Partner ID not found.', 'sureforms-pro' );
			return $result;
		}

		// Fetch merchant credentials from PayPal.
		$response = wp_remote_get( $base_url . "v1/customer/partners/{$partner_id}/merchant-integrations/credentials/", $args );

		if ( is_wp_error( $response ) ) {
			$result['message'] = $response->get_error_message();
			return $result;
		}

		$body     = wp_remote_retrieve_body( $response );
		$response = json_decode( $body );

		// Validate response and check for required credentials.
		if ( is_object( $response ) && ! empty( $response->client_id ) && ! empty( $response->client_secret ) && ! empty( $response->payer_id ) ) {

			// Check merchant onboarding status.
			$onboarding_check = $this->check_merchant_onboarding_status( $args, $partner_id, $response->payer_id, $mode );

			if ( ! $onboarding_check['status'] ) {
				return $onboarding_check;
			}

			// Get current settings.
			$settings = PayPal_Helper::get_all_paypal_settings();

			// Update settings based on mode.
			if ( 'live' === $mode ) {
				$settings['paypal_live_client_id']     = $response->client_id;
				$settings['paypal_live_client_secret'] = $response->client_secret;
				$settings['paypal_account_id']         = $response->payer_id;
				$settings['paypal_live_connected']     = true;
			} else {
				$settings['paypal_test_client_id']     = $response->client_id;
				$settings['paypal_test_client_secret'] = $response->client_secret;
				$settings['paypal_account_id']         = $response->payer_id;
				$settings['paypal_sandbox_connected']  = true;
			}

			// Get account email if available.
			if ( ! empty( $response->primary_email ) ) {
				$settings['paypal_account_email'] = $response->primary_email;
			}

			// Get account name if available.
			if ( ! empty( $response->legal_name ) ) {
				$settings['account_name'] = $response->legal_name;
			}

			// Save settings.
			PayPal_Helper::update_all_paypal_settings( $settings );

			$result['status'] = true;
		}

		return $result;
	}

	/**
	 * Check merchant onboarding status to ensure account is ready to receive payments.
	 *
	 * @param array  $args            Request headers.
	 * @param string $partner_id      PayPal partner ID.
	 * @param string $seller_payer_id PayPal seller payer ID.
	 * @param string $mode            PayPal mode (sandbox or live).
	 * @since 2.4.0
	 * @return array Status array.
	 */
	private function check_merchant_onboarding_status( $args, $partner_id, $seller_payer_id, $mode ) {
		// Determine PayPal API base URL.
		$base_url = 'live' === $mode
			? 'https://api-m.paypal.com/'
			: 'https://api-m.sandbox.paypal.com/';

		$response = wp_remote_get( $base_url . "v1/customer/partners/{$partner_id}/merchant-integrations/{$seller_payer_id}", $args );

		if ( is_wp_error( $response ) ) {
			return [ 'status' => false ];
		}

		$body     = wp_remote_retrieve_body( $response );
		$response = json_decode( $body, true );

		// Validate response structure.
		if ( ! is_array( $response ) || ( ! isset( $response['payments_receivable'] ) || ! isset( $response['primary_email_confirmed'] ) ) ) {
			return [ 'status' => false ];
		}

		// Check if merchant can receive payments and email is confirmed.
		if ( true === $response['payments_receivable'] && true === $response['primary_email_confirmed'] ) {
			return [ 'status' => true ];
		}

		// Build error message for incomplete onboarding.
		$issues = [];

		if ( true !== $response['payments_receivable'] ) {
			$issues[] = __( 'Account is not set up to receive payments', 'sureforms-pro' );
		}

		if ( true !== $response['primary_email_confirmed'] ) {
			$issues[] = __( 'Primary email is not confirmed', 'sureforms-pro' );
		}

		return [
			'status'  => false,
			'message' => sprintf(
				/* translators: %s: comma-separated list of issues */
				__( 'PayPal account setup incomplete: %s. Please complete your PayPal account setup and try again.', 'sureforms-pro' ),
				implode( ', ', $issues )
			),
		];
	}
}
