<?php
/**
 * Google Sheets Payload Transformer
 *
 * Transforms form data to Google Sheets API format.
 *
 * @package SureForms
 * @since 2.3.0
 */

namespace SRFM_Pro\Inc\Pro\Native_Integrations\Transformers;

use SRFM\Inc\Helper;
use SRFM_Pro\Inc\Pro\Native_Integrations\Interfaces\Payload_Transformer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Google Sheets Transformer class.
 *
 * @since 2.3.0
 */
class Google_Sheets_Transformer implements Payload_Transformer {
	/**
	 * Transform payload for Google Sheets integration
	 *
	 * @param array $form_data Original form data.
	 * @param array $field_mappings Field mappings configuration.
	 * @param array $context Additional context data (workflow, action_config).
	 * @return array Transformed payload.
	 * @since 2.3.0
	 */
	public function transform( $form_data, $field_mappings = [], $context = [] ) {
		// Unset unused parameters to avoid warnings.
		unset( $field_mappings );
		unset( $context );

		$exclude_fields = [
			'spreadsheet_id',
			'sheet_name',
			'value_input_option',
			'insert_data_option',
			'include_headers',
			'field_mappings',
		];

		$row_values = [];

		foreach ( $form_data as $field_key => $field_value ) {
			if ( ! in_array( $field_key, $exclude_fields, true ) ) {
				$row_values[] = Helper::get_string_value( $field_value );
			}
		}

		// Return in Google Sheets API format.
		return [
			'values' => [ $row_values ],
		];
	}

	/**
	 * Prepare test payload with sample data
	 *
	 * @param array $workflow_fields Fields from the workflow configuration.
	 * @param array $field_definitions Field definitions from the action configuration.
	 * @return array Test payload.
	 * @since 2.3.0
	 */
	public function prepare_test_payload( $workflow_fields, $field_definitions ) {
		// Added unset to avoid warnings.
		unset( $workflow_fields );
		unset( $field_definitions );

		// Generate sample data for testing.
		$test_data = [
			'Test Name',
			'test@example.com',
			'+1234567890',
			'Test Message Content',
		];

		return [
			'values' => [ $test_data ],
		];
	}

	/**
	 * Get integration name this transformer handles
	 *
	 * @return string Integration name.
	 * @since 2.3.0
	 */
	public function get_integration_name() {
		return 'google_sheets';
	}
}
