<?php
/**
 * PayPal Payments API Class.
 *
 * Provides clean, reusable methods for PayPal payment operations.
 * This class handles all payment-related API endpoints including:
 * - Orders (create, capture, authorize)
 * - Refunds
 * - Subscriptions
 *
 * @package sureforms-pro
 * @since 2.4.0
 */

namespace SRFM_Pro\Inc\Business\Payments\PayPal;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * PayPal Payments API.
 *
 * Usage example:
 * ```php
 * // Create order
 * $result = API_Payments::create_order( $order_data );
 *
 * // Capture order
 * $result = API_Payments::capture_order( $order_id );
 *
 * // Refund payment
 * $result = API_Payments::refund_capture( $capture_id, $refund_data );
 * ```
 *
 * @since 2.4.0
 */
class API_Payments {
	/**
	 * Use PayPal API client trait for HTTP communication.
	 */
	use Client;

	/**
	 * Create a PayPal order.
	 *
	 * Creates a new order in PayPal system. This is typically the first step
	 * in processing a payment.
	 *
	 * @param array       $body Order data (purchase units, payer info, etc.).
	 * @param string|null $mode Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with order ID and status.
	 */
	public static function create_order( $body, $mode = null ) {
		return self::request( 'v2/checkout/orders', $body, 'POST', $mode );
	}

	/**
	 * Capture payment for an order.
	 *
	 * Captures the payment for an authorized order. This finalizes the payment
	 * and transfers funds from the customer to the merchant.
	 *
	 * @param string      $order_id PayPal order ID.
	 * @param array       $body     Optional capture parameters.
	 * @param string|null $mode     Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with capture details and status.
	 */
	public static function capture_order( $order_id, $body = [], $mode = null ) {
		return self::request( "v2/checkout/orders/{$order_id}/capture", $body, 'POST', $mode );
	}

	/**
	 * Authorize payment for an order.
	 *
	 * Authorizes payment for an order without capturing it immediately.
	 * The funds are reserved but not transferred until captured later.
	 *
	 * @param string      $order_id PayPal order ID.
	 * @param array       $body     Optional authorization parameters.
	 * @param string|null $mode     Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with authorization details.
	 */
	public static function authorize_order( $order_id, $body = [], $mode = null ) {
		return self::request( "v2/checkout/orders/{$order_id}/authorize", $body, 'POST', $mode );
	}

	/**
	 * Refund a captured payment.
	 *
	 * Refunds a previously captured payment (full or partial).
	 * Requires the capture ID from the original payment.
	 *
	 * @param string      $capture_id PayPal capture ID (transaction ID).
	 * @param array       $body       Refund data (amount, currency, note).
	 * @param string|null $mode       Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with refund details and status.
	 */
	public static function refund_capture( $capture_id, $body, $mode = null ) {
		return self::request( "v2/payments/captures/{$capture_id}/refund", $body, 'POST', $mode );
	}

	/**
	 * Create a PayPal subscription.
	 *
	 * Creates a new subscription for recurring payments.
	 *
	 * @param array       $body Subscription data (plan ID, subscriber info, etc.).
	 * @param string|null $mode Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with subscription ID and status.
	 */
	public static function create_subscription( $body, $mode = null ) {
		return self::request( 'v1/billing/subscriptions', $body, 'POST', $mode );
	}

	/**
	 * Get subscription details.
	 *
	 * Retrieves complete details about a subscription including status,
	 * billing info, and payment history.
	 *
	 * @param string      $subscription_id PayPal subscription ID.
	 * @param string|null $mode            Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with subscription details.
	 */
	public static function get_subscription( $subscription_id, $mode = null ) {
		return self::request( "v1/billing/subscriptions/{$subscription_id}", [], 'GET', $mode );
	}

	/**
	 * Get order details.
	 *
	 * Retrieves complete details about an order including payment status,
	 * payer information, and purchase units.
	 *
	 * @param string      $order_id PayPal order ID.
	 * @param string|null $mode     Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with order details.
	 */
	public static function get_order( $order_id, $mode = null ) {
		return self::request( "v2/checkout/orders/{$order_id}", [], 'GET', $mode );
	}

	/**
	 * Create billing agreement token.
	 *
	 * Creates a billing agreement token for reference transactions.
	 * Used for subscription payment methods.
	 *
	 * @param array       $body Billing agreement data.
	 * @param string|null $mode Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with billing agreement token.
	 */
	public static function create_billing_agreement_token( $body, $mode = null ) {
		return self::request( 'v1/billing-agreements/agreement-tokens', $body, 'POST', $mode );
	}

	/**
	 * Create billing agreement.
	 *
	 * Creates a billing agreement from a token for recurring payments.
	 *
	 * @param string      $billing_token Billing agreement token.
	 * @param string|null $mode          Payment mode ('test' or 'live'). If null, uses current mode.
	 * @since 2.4.0
	 * @return array Response with billing agreement details.
	 */
	public static function create_billing_agreement( $billing_token, $mode = null ) {
		return self::request( 'v1/billing-agreements/agreements', [ 'token_id' => $billing_token ], 'POST', $mode );
	}
}
