<?php
/**
 * Transformer Factory
 *
 * Factory class to create payload transformer instances.
 *
 * @package SureForms
 * @since 1.13.0
 */

namespace SRFM_Pro\Inc\Pro\Native_Integrations\Factories;

use SRFM_Pro\Inc\Pro\Native_Integrations\Interfaces\Payload_Transformer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Transformer Factory class.
 *
 * @since 1.13.0
 */
class Transformer_Factory {
	/**
	 * Transformer instances cache
	 *
	 * @var array
	 * @since 1.13.0
	 */
	private static $instances = [];

	/**
	 * Integrations that have specific transformer implementations
	 *
	 * @var array
	 * @since 1.13.0
	 */
	private static $transformer_map = [
		'kit'           => 'SRFM_Pro\Inc\Pro\Native_Integrations\Transformers\Kit_Transformer',
		'agile-crm'     => 'SRFM_Pro\Inc\Pro\Native_Integrations\Transformers\Agile_CRM_Transformer',
		'keap'          => 'SRFM_Pro\Inc\Pro\Native_Integrations\Transformers\Keap_Transformer',
		'google_sheets' => 'SRFM_Pro\Inc\Pro\Native_Integrations\Transformers\Google_Sheets_Transformer',
		'notion'        => 'SRFM_Pro\Inc\Pro\Native_Integrations\Transformers\Notion_Transformer',
	];

	/**
	 * Create transformer instance for integration
	 *
	 * @param string $integration_name Integration name.
	 * @return Payload_Transformer|null
	 * @since 1.13.0
	 */
	public static function create( $integration_name ) {
		$integration_key = strtolower( $integration_name );

		// Return cached instance if available.
		if ( isset( self::$instances[ $integration_key ] ) ) {
			return self::$instances[ $integration_key ];
		}

		// Check if we have a specific transformer for this integration.
		if ( isset( self::$transformer_map[ $integration_key ] ) ) {
			$transformer_class = self::$transformer_map[ $integration_key ];

			// Load transformer class if not already loaded.
			self::load_transformer_class( $integration_key, $transformer_class );

			if ( class_exists( $transformer_class ) ) {
				$transformer = new $transformer_class();

				// Ensure the created instance implements the required interface.
				if ( $transformer instanceof Payload_Transformer ) {
					self::$instances[ $integration_key ] = $transformer;
					return $transformer;
				}
			}
		}

		// No specific transformer found.
		return null;
	}

	/**
	 * Check if integration has a specific transformer
	 *
	 * @param string $integration_name Integration name.
	 * @return bool True if transformer exists, false otherwise.
	 * @since 1.13.0
	 */
	public static function has_transformer( $integration_name ) {
		$integration_key = strtolower( $integration_name );
		return isset( self::$transformer_map[ $integration_key ] );
	}

	/**
	 * Get all available transformers
	 *
	 * @return array Array of integration names that have transformers.
	 * @since 1.13.0
	 */
	public static function get_available_transformers() {
		return array_keys( self::$transformer_map );
	}

	/**
	 * Register a new transformer for an integration
	 *
	 * @param string $integration_name Integration name.
	 * @param string $transformer_class Transformer class name.
	 * @return void
	 * @since 1.13.0
	 */
	public static function register_transformer( $integration_name, $transformer_class ) {
		$integration_key                           = strtolower( $integration_name );
		self::$transformer_map[ $integration_key ] = $transformer_class;

		// Clear cached instance if it exists.
		if ( isset( self::$instances[ $integration_key ] ) ) {
			unset( self::$instances[ $integration_key ] );
		}
	}

	/**
	 * Clear all cached transformer instances
	 *
	 * @return void
	 * @since 1.13.0
	 */
	public static function clear_cache() {
		self::$instances = [];
	}

	/**
	 * Load transformer class file if not already loaded
	 *
	 * @param string $integration_key Integration key.
	 * @param string $transformer_class Transformer class name.
	 * @return void
	 * @since 1.13.0
	 */
	private static function load_transformer_class( $integration_key, $transformer_class ) {
		// Skip if class already exists.
		if ( class_exists( $transformer_class ) ) {
			return;
		}

		// Map integration keys to their transformer file paths.
		$transformer_files = [
			'kit'           => SRFM_PRO_DIR . 'inc/pro/native-integrations/transformers/kit-transformer.php',
			'agile-crm'     => SRFM_PRO_DIR . 'inc/pro/native-integrations/transformers/agile-crm-transformer.php',
			'keap'          => SRFM_PRO_DIR . 'inc/pro/native-integrations/transformers/keap-transformer.php',
			'google_sheets' => SRFM_PRO_DIR . 'inc/pro/native-integrations/transformers/google-sheets-transformer.php',
			'notion'        => SRFM_PRO_DIR . 'inc/pro/native-integrations/transformers/notion-transformer.php',
		];

		if ( isset( $transformer_files[ $integration_key ] ) ) {
			$file_path = $transformer_files[ $integration_key ];

			if ( file_exists( $file_path ) ) {
				require_once $file_path;
			}
		}
	}
}
