<?php
/**
 * Abstract Integration Provider
 *
 * Base class for all integration providers.
 *
 * @package SureForms
 * @since 1.13.0
 */

namespace SRFM_Pro\Inc\Pro\Native_Integrations;

use SRFM_Pro\Inc\Pro\Native_Integrations\Utils\Integration_Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Abstract Integration Provider class.
 *
 * @since 1.13.0
 */
abstract class Integration_Provider {
	/**
	 * Provider name/type (e.g., 'mailchimp', 'constantcontact')
	 *
	 * @var string
	 * @since 1.13.0
	 */
	protected $provider_type;

	/**
	 * Integration configuration from JSON
	 *
	 * @var array
	 * @since 1.13.0
	 */
	protected $config;

	/**
	 * Constructor
	 *
	 * @param string $provider_type Provider type.
	 * @param array  $config Integration configuration.
	 * @since 1.13.0
	 */
	public function __construct( $provider_type, $config ) {
		$this->provider_type = $provider_type;
		$this->config        = $config;
	}

	/**
	 * Test integration connection
	 *
	 * @param array $credentials User credentials.
	 * @param array $auth_config Authentication configuration.
	 * @return array Test result with success status and message.
	 * @since 1.13.0
	 */
	abstract public function test_connection( $credentials, $auth_config = [] );

	/**
	 * Fetch select options (like lists, campaigns, etc.)
	 *
	 * @param array $credentials User credentials.
	 * @param array $api_config API configuration.
	 * @param array $parameters Optional parameters for the API call.
	 * @return array Options array.
	 * @since 1.13.0
	 */
	abstract public function fetch_select_options( $credentials, $api_config, $parameters = [] );

	/**
	 * Fetch dynamic fields for a specific action
	 *
	 * @param array $credentials User credentials.
	 * @param array $api_config API configuration.
	 * @param array $parameters Parameters for the API call.
	 * @return array Fields array.
	 * @since 1.13.0
	 */
	abstract public function fetch_dynamic_fields( $credentials, $api_config, $parameters = [] );

	/**
	 * Get provider type
	 *
	 * @return string Provider type.
	 * @since 1.13.0
	 */
	public function get_provider_type() {
		return $this->provider_type;
	}

	/**
	 * Get provider configuration
	 *
	 * @return array Provider configuration.
	 * @since 1.13.0
	 */
	public function get_config() {
		return $this->config;
	}

	/**
	 * Map provider-specific field types to SureForms field types
	 *
	 * @param string $provider_type Provider field type.
	 * @return string SureForms field type.
	 * @since 1.13.0
	 */
	abstract protected function map_field_type( $provider_type );

	/**
	 * Process API response based on response type
	 *
	 * @param array  $response API response data.
	 * @param string $response_type Type of response to process.
	 * @param array  $config API configuration.
	 * @return array Processed response.
	 * @since 1.13.0
	 */
	abstract protected function process_api_response( $response, $response_type, $config );

	/**
	 * Replace placeholders in URL with actual values
	 *
	 * @param string $url URL with placeholders.
	 * @param array  $replacements Key-value pairs for replacement.
	 * @return string URL with placeholders replaced.
	 * @since 1.13.0
	 */
	protected function replace_placeholders( $url, $replacements ) {
		return Integration_Utils::replace_placeholders( $url, $replacements );
	}

	/**
	 * Make HTTP request to provider API
	 *
	 * @param string                $url API endpoint URL.
	 * @param string                $method HTTP method.
	 * @param array<string, string> $headers Request headers.
	 * @param array|string|null     $body Request body.
	 * @return array<string,mixed>|\WP_Error Response array or WP_Error on failure.
	 * @since 1.13.0
	 */
	protected function make_api_request( $url, $method = 'GET', $headers = [], $body = null ) {
		$args = [
			'method'  => $method,
			'headers' => $headers,
			'timeout' => 30,
		];

		if ( $body && in_array( $method, [ 'POST', 'PUT', 'PATCH' ], true ) ) {
			if ( is_array( $body ) ) {
				$encoded_body = wp_json_encode( $body );
				$args['body'] = false !== $encoded_body ? $encoded_body : '';
			} else {
				$args['body'] = $body;
			}
		}

		return wp_remote_request( $url, $args );
	}

	/**
	 * Extract server prefix from API key (for providers like Mailchimp)
	 *
	 * @param string $api_key API key.
	 * @return string Server prefix or empty string.
	 * @since 1.13.0
	 */
	protected function extract_server_prefix( $api_key ) {
		$parts = explode( '-', $api_key );
		return $parts[1] ?? '';
	}
}
