<?php
/**
 * WordPress Action Base Class
 *
 * Provides a standardized base for all WordPress plugin integration actions.
 *
 * @package SureForms
 * @since 1.13.0
 */

namespace SRFM_Pro\Inc\Pro\Native_Integrations;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Abstract WordPress Action Base Class
 *
 * @since 1.13.0
 */
abstract class WordPress_Action {
	/**
	 * Integration name (e.g., 'LatePoint', 'WooCommerce')
	 *
	 * @var string
	 * @since 1.13.0
	 */
	protected $integration = '';

	/**
	 * Action name (e.g., 'create_customer', 'create_order')
	 *
	 * @var string
	 * @since 1.13.0
	 */
	protected $action = '';

	/**
	 * Error data storage
	 *
	 * @var array|null
	 * @since 1.13.0
	 */
	protected $error_data = null;

	/**
	 * Constructor - Auto-registers the action
	 *
	 * @since 1.13.0
	 */
	public function __construct() {
		if ( empty( $this->integration ) || empty( $this->action ) ) {
			return;
		}

		$action_name = $this->get_action_name();
		add_action( $action_name, [ $this, 'action_handler' ] );
	}

	/**
	 * Action handler - processes the action and stores result globally
	 *
	 * @param array $data Form submission data.
	 * @return void
	 * @throws \Exception When plugin is not active or execution fails.
	 * @since 1.13.0
	 */
	public function action_handler( $data ) {
		try {
			// Validate required plugin.
			if ( ! $this->is_plugin_active() ) {
				throw new \Exception(
					sprintf(
						// translators: %s: Integration name.
						__( '%s plugin is not active.', 'sureforms-pro' ),
						$this->integration
					)
				);
			}

			// Execute the action.
			$result = $this->execute( $data );

			// Store result for retrieval by provider.
			$this->store_result( $result );

		} catch ( \Exception $e ) {
			$error = new \WP_Error(
				'action_execution_failed',
				sprintf(
					// translators: %s: Error message.
					__( 'Action execution failed: %s', 'sureforms-pro' ),
					$e->getMessage()
				)
			);

			$this->store_result( $error );
		}
	}

	/**
	 * Get the full WordPress action name
	 *
	 * @return string
	 * @since 1.13.0
	 */
	protected function get_action_name() {
		return 'srfm_' . strtolower( $this->integration ) . '_' . $this->action;
	}

	/**
	 * Validate required fields
	 *
	 * @param array $data Form data.
	 * @param array $required_fields List of required field keys.
	 * @return true|\WP_Error
	 * @since 1.13.0
	 */
	protected function validate_required_fields( $data, $required_fields ) {
		if ( ! is_array( $data ) ) {
			return new \WP_Error( 'invalid_data', __( 'Invalid data provided.', 'sureforms-pro' ) );
		}

		$missing_fields = [];
		foreach ( $required_fields as $field ) {
			if ( empty( $data[ $field ] ) ) {
				$missing_fields[] = $field;
			}
		}

		if ( ! empty( $missing_fields ) ) {
			return new \WP_Error(
				'missing_required_fields',
				sprintf(
					// translators: %s: Comma-separated list of missing fields.
					__( 'Missing required fields: %s', 'sureforms-pro' ),
					implode( ', ', $missing_fields )
				)
			);
		}

		return true;
	}

	/**
	 * Check if the required plugin is active
	 *
	 * @return bool
	 * @since 1.13.0
	 */
	abstract protected function is_plugin_active();

	/**
	 * Execute the action logic
	 *
	 * @param array $data Form submission data.
	 * @return array|\WP_Error
	 * @since 1.13.0
	 */
	abstract protected function execute( $data );

	/**
	 * Store result in global storage for provider retrieval
	 *
	 * @param mixed $result Action result.
	 * @return void
	 * @since 1.13.0
	 */
	private function store_result( $result ) {
		if ( ! isset( $GLOBALS['srfm_action_results'] ) ) {
			$GLOBALS['srfm_action_results'] = [];
		}

		$action_name                                    = $this->get_action_name();
		$GLOBALS['srfm_action_results'][ $action_name ] = $result;
	}
}
