<?php
/**
 * Save and Resume Module Utils
 * This class handles all the utility functions related to the Save and Resume module in the SureForms Pro plugin.
 *
 * @since 2.2.0
 * @package sureforms-pro
 */

namespace SRFM_Pro\Inc\Pro\Save_Resume;

use SRFM\Inc\Helper;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Save and Resume Module Utils
 *
 * @since 2.2.0
 */
class Utils {
	/**
	 * Sanitizes the data for the Save and Resume module.
	 *
	 * This function takes a JSON string of Save and Resume data, decodes it, validates structure,
	 * sanitizes each field appropriately, and returns a JSON-encoded string of the sanitized data.
	 *
	 * @param string $data The JSON string containing Save and Resume data.
	 * @since 2.2.0
	 * @return string Sanitized JSON string of Save and Resume data.
	 */
	public static function save_resume_meta_sanitizer( $data ) {
		if ( empty( $data ) || ! is_string( $data ) ) {
			return '';
		}

		$data = json_decode( $data, true );

		if ( ! is_array( $data ) || json_last_error() !== JSON_ERROR_NONE ) {
			// If the JSON is invalid, return an empty array as JSON.
			return '';
		}

		$sanitized = [
			'status'                => isset( $data['status'] ) ? wp_validate_boolean( $data['status'] ) : false,
			'linkText'              => isset( $data['linkText'] ) ? sanitize_text_field( $data['linkText'] ) : __( 'Save & Continue', 'sureforms-pro' ),
			'confirmationMessage'   => isset( $data['confirmationMessage'] ) ? Helper::strip_js_attributes( $data['confirmationMessage'] ) : '',
			'afterEmailSendMessage' => isset( $data['afterEmailSendMessage'] ) ? Helper::strip_js_attributes( $data['afterEmailSendMessage'] ) : '',
			'emailBody'             => isset( $data['emailBody'] ) ? Helper::strip_js_attributes( $data['emailBody'] ) : '',
		];

		// Return the sanitized data as a JSON string.
		$return = wp_json_encode( $sanitized, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );
		return $return ? $return : '';
	}

	/**
	 * Get file prefix based on debug mode.
	 *
	 * @return string
	 * @since 2.2.0
	 */
	public static function get_file_prefix() {
		return defined( 'SRFM_DEBUG' ) && SRFM_DEBUG ? '' : '.min';
	}

	/**
	 * Get directory name based on debug mode.
	 *
	 * @return string
	 * @since 2.2.0
	 */
	public static function get_dir_name() {
		return defined( 'SRFM_DEBUG' ) && SRFM_DEBUG ? 'unminified' : 'minified';
	}

	/**
	 * Get css URI based on directory name.
	 *
	 * @param string $dir_name Directory name, either 'minified' or 'unminified'.
	 * @return string
	 * @since 2.2.0
	 */
	public static function get_css_uri( $dir_name ) {
		return SRFM_PRO_URL . 'assets/css/' . $dir_name . '/package/pro/';
	}

	/**
	 * Get RTL suffix for CSS files.
	 *
	 * @return string
	 * @since 2.2.0
	 */
	public static function get_rtl_suffix() {
		return is_rtl() ? '-rtl' : '';
	}

	/**
	 * Get the default confirmation message.
	 *
	 * @return string The HTML confirmation message.
	 * @since 2.2.0
	 */
	public static function get_default_confirmation_message() {
		ob_start();
		?>
		<p><span style="font-size: 24px;"><strong><?php esc_html_e( 'We\'ve Saved Your Progress!', 'sureforms-pro' ); ?></strong></span></p><p><?php esc_html_e( 'Your responses are safe! Use the link below to return and complete your form at any time within the next 30 days.', 'sureforms-pro' ); ?></p><p>{resume_url}</p><p><?php esc_html_e( 'Or, you can just enter your email, and we\'ll send the link straight to your inbox.', 'sureforms-pro' ); ?></p><p>{save_email_input}</p>
		<?php
		$output = ob_get_clean();
		return trim( false !== $output ? $output : '' );
	}

	/**
	 * Get the default after email send message.
	 *
	 * @return string The HTML after email send message.
	 * @since 2.2.0
	 */
	public static function get_default_after_email_send_message() {
		ob_start();
		?>
		<p><strong><span style="font-size: 24px;"><?php esc_html_e( 'Link Sent Successfully!', 'sureforms-pro' ); ?></span></strong></p><p><?php esc_html_e( 'We\'ve sent a copy of your save link to', 'sureforms-pro' ); ?> <strong>{save_email}</strong></p><p><?php esc_html_e( 'Check your inbox and spam folder, just in case, to continue your form later.', 'sureforms-pro' ); ?></p>
		<?php
		$output = ob_get_clean();
		return trim( false !== $output ? $output : '' );
	}

	/**
	 * Get the default email body message.
	 *
	 * @return string The HTML email body message.
	 * @since 2.2.0
	 */
	public static function get_default_email_body() {
		ob_start();
		?>
		<p><strong><span style="font-size: 18px;"><?php esc_html_e( 'Hey There!', 'sureforms-pro' ); ?></span></strong></p><p><span><?php esc_html_e( 'Thank you for using SureForms!', 'sureforms-pro' ); ?></span></p><p><span><?php esc_html_e( 'Your progress for the form has been saved successfully.', 'sureforms-pro' ); ?></span></p><p><span><?php esc_html_e( 'Click the button below anytime within the next 30 days to resume and complete your form from any device:', 'sureforms-pro' ); ?></span></p><p>{resume_url}</p><p><span><?php esc_html_e( 'Please note: This link is unique to you, so keep it safe. It will expire in 30 days and should not be shared with others.', 'sureforms-pro' ); ?></span></p><p><?php esc_html_e( 'Best,', 'sureforms-pro' ); ?><br><?php esc_html_e( 'The SureForms Team', 'sureforms-pro' ); ?></p>
		<?php
		$output = ob_get_clean();
		return trim( false !== $output ? $output : '' );
	}
}
