<?php
/**
 * It is Main File to load all Notice, Upgrade Menu and all
 *
 * @link       https://posimyth.com/
 * @since      5.3.3
 *
 * @package    Theplus
 * @subpackage ThePlus/Notices
 * */

namespace Theplus\Notices;

/**
 * Exit if accessed directly.
 * */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'Tp_Memorylimit_Popup' ) ) {

	/**
	 * This class used for only load All Notice Files
	 *
	 * @since 5.3.3
	 */
	class Tp_Memorylimit_Popup {

		/**
		 * Singleton Instance of the Class.
		 *
		 * @since 5.3.3
		 * @access private
		 * @static
		 * @var null|instance $instance An instance of the class or null if not instantiated yet.
		 */
		private static $instance = null;

		/**
		 * Singleton Instance Creation Method.
		 *
		 * This public static method ensures that only one instance of the class is loaded or can be loaded.
		 * It follows the Singleton design pattern to create or return the existing instance of the class.
		 *
		 * @since 5.3.3
		 * @access public
		 * @static
		 * @return self Instance of the class.
		 */
		public static function instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor Method for Compatibility Checks and Actions Initialization.
		 *
		 * This constructor method is automatically triggered when the class is instantiated.
		 * It performs compatibility checks to ensure basic requirements are met and initiates
		 * necessary actions such as setting up deactivation feedback and adding AJAX hooks.
		 *
		 * @since 5.3.3
		 * @access public
		 */
		public function __construct() {

			if ( ! get_option( 'tp_memorylimit_popup_dismiss' ) ) {
            	add_action( 'elementor/editor/before_enqueue_scripts', array( $this, 'tp_enqueue_memorylimit_popup_scripts' ) );
				add_action( 'elementor/editor/before_enqueue_styles', array( $this, 'tp_enqueue_memorylimit_popup_styles' ) );
			}

			add_action( 'wp_ajax_theplus_memory_popup_dismiss', array( $this, 'theplus_memory_limit_popup_dismiss' ) );
		}

        public function tp_enqueue_memorylimit_popup_scripts() {
            wp_enqueue_script( 'plus-memorylimit-popup-js', $this->pathurl_security( THEPLUS_URL . 'includes/notices/memory-popup/plus-memorylimit-popup.js' ), array(), L_THEPLUS_VERSION, true );

			wp_localize_script( 
				'plus-memorylimit-popup-js', 
				'tp_memory_limit_popup',
				array(
					'nonce'    => wp_create_nonce( 'tp_memory_popup' ),
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'manager_link' => admin_url( 'admin.php?page=elementor-element-manager' ),
					'memory_limit' => ini_get('memory_limit'),
				)
			);
        }

		public function tp_enqueue_memorylimit_popup_styles() {
            wp_enqueue_style( 'plus-memorylimit-popup', THEPLUS_URL . 'includes/notices/memory-popup/plus-memorylimit-popup.css', array(), THEPLUS_VERSION, false );
		}

        public function pathurl_security( $url ) {
			return preg_replace( array( '/^http:/', '/^https:/', '/(?!^)\/\//' ), array( '', '', '/' ), $url );
		}

		/**
		 * It's is use for Save key in database
		 * TAPG Notice and TAG Popup Dismisse
		 *
		 * @since 6.3.11
		 */
		public function theplus_memory_limit_popup_dismiss() {
			$get_security = ! empty( $_POST['security'] ) ? sanitize_text_field( wp_unslash( $_POST['security'] ) ) : '';

			if ( ! isset( $get_security ) || empty( $get_security ) || ! wp_verify_nonce( $get_security, 'tp_memory_popup' ) ) {
				wp_die( 'Security checked!' );
			}

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_send_json_error( __( 'You are not allowed to do this action', 'theplus' ) );
			}

			$get_type = ! empty( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : '';

			update_option( 'tp_memorylimit_popup_dismiss', true );

			wp_send_json_success();
		}

	}

	Tp_Memorylimit_Popup::instance();
}
