<?php
/**
 * The file that defines the core plugin class
 *
 * @link    https://posimyth.com/
 * @since   6.4.1
 *
 * @package the-plus-addons-for-elementor-page-builder
 */

namespace TheplusAddons\Widgets;

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;

if ( ! defined( 'WPINC' ) ) {
	exit;
}

/*
 * Plus Continuous Animation Theplus.
 * 
 * @since 6.4.1
 */
if ( ! class_exists( 'Tpaep_Plus_Contianimation' ) ) {

	/**
	 * Define Tpaep_Plus_Contianimation class
	 */
	class Tpaep_Plus_Contianimation {

		/**
		 * A reference to an instance of this class.
		 *
		 * @since 6.4.1
		 * @var   object
		 */
		private static $instance = null;

		/**
		 * Returns a singleton instance of the class.
		 *
		 * This method ensures that only one instance of the class is created (singleton pattern).
		 * If an instance doesn't exist, it creates one using the provided shortcodes.
		 *
		 * @since 6.4.1
		 *
		 * @param array $shortcodes Optional. An array of shortcodes to initialize the instance with.
		 * @return self The single instance of the class.
		 */
		public static function get_instance( $shortcodes = array() ) {

			if ( null === self::$instance ) {
				self::$instance = new self( $shortcodes );
			}

			return self::$instance;
		}

        private array $containers_data = [];

		/**
		 * Initalize integration hooks
		 *
		 * @return void
		 * 
		 * @since 6.4.1
		 */
		public function __construct() {

            add_action( 'elementor/element/common/_section_style/after_section_end', array( $this, 'tpaep_register_controls' ), 10, 2 );
			add_filter( 'elementor/widget/render_content', array( $this, 'tpaep_render_element' ), 10, 2 );
		}

		 /**
		 * Register controls for the Continuous Animation feature
		 *
		 * @since 6.4.1
		 */
        public function tpaep_register_controls( $element, $section_id ) {

            if ( ! $element || ! is_object( $element ) ) {
                return; 
            }

            $element->start_controls_section(
                'plus_section_contianimation_section',
                array(
                    'label' => esc_html__( 'Continuous Animation', 'theplus' ),
                    'tab'   => Controls_Manager::TAB_ADVANCED,
                )
            );

            $element->add_control(
				'plus_continuous_animation',
				array(
					'label'       => esc_html__( 'Continuous Animation', 'theplus' ),
					'type'        => Controls_Manager::SWITCHER,
					'label_on'    => esc_html__( 'Yes', 'theplus' ),
					'label_off'   => esc_html__( 'No', 'theplus' ),
					'render_type' => 'template',
					'separator'   => 'before',
					'description' => wp_kses_post(
						sprintf(
							'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
							esc_html__( 'Turn this on to keep the animation playing continuously. It helps draw attention to the element, and you can set different animations for normal view and hover.', 'theplus' ),
							esc_url( L_THEPLUS_TPDOC . 'global-continuous-animation-widget-in-elementor/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
							esc_html__( 'Learn More', 'theplus' ),
						)
					),
				)
			);
			$element->add_control(
				'plus_animation_effect',
				array(
					'label'       => esc_html__( 'Animation Effect', 'theplus' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => 'pulse',
					'options'     => array(
						'pulse'    => esc_html__( 'Pulse', 'theplus' ),
						'floating' => esc_html__( 'Floating', 'theplus' ),
						'tossing'  => esc_html__( 'Tossing', 'theplus' ),
						'rotating' => esc_html__( 'Rotating', 'theplus' ),
					),
					'render_type' => 'template',
					'condition'   => array(
						'plus_continuous_animation' => 'yes',
					),
				)
			);
			$element->add_control(
				'plus_animation_hover',
				array(
					'label'       => esc_html__( 'Hover Animation', 'theplus' ),
					'type'        => Controls_Manager::SWITCHER,
					'label_on'    => esc_html__( 'Yes', 'theplus' ),
					'label_off'   => esc_html__( 'No', 'theplus' ),
					'render_type' => 'template',
					'condition'   => array(
						'plus_continuous_animation' => 'yes',
					),
				)
			);
			$element->add_control(
				'plus_animation_duration',
				array(
					'label'      => esc_html__( 'Duration Time', 'theplus' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => 's',
					'range'      => array(
						's' => array(
							'min'  => 0.5,
							'max'  => 50,
							'step' => 0.1,
						),
					),
					'default'    => array(
						'unit' => 's',
						'size' => 2.5,
					),
					'selectors'  => array(
						'{{WRAPPER}} .plus-conti-animation' => 'animation-duration: {{SIZE}}{{UNIT}};-webkit-animation-duration: {{SIZE}}{{UNIT}};',
					),
					'condition'  => array(
						'plus_continuous_animation' => 'yes',
					),
				)
			);
			$element->add_control(
				'plus_transform_origin',
				array(
					'label'       => esc_html__( 'Transform Origin', 'theplus' ),
					'type'        => Controls_Manager::SELECT,
					'default'     => 'center center',
					'options'     => array(
						'top left'      => esc_html__( 'Top Left', 'theplus' ),
						'top center"'   => esc_html__( 'Top Center', 'theplus' ),
						'top right'     => esc_html__( 'Top Right', 'theplus' ),
						'center left'   => esc_html__( 'Center Left', 'theplus' ),
						'center center' => esc_html__( 'Center Center', 'theplus' ),
						'center right'  => esc_html__( 'Center Right', 'theplus' ),
						'bottom left'   => esc_html__( 'Bottom Left', 'theplus' ),
						'bottom center' => esc_html__( 'Bottom Center', 'theplus' ),
						'bottom right'  => esc_html__( 'Bottom Right', 'theplus' ),
					),
					'selectors'   => array(
						'{{WRAPPER}} .plus-conti-animation' => '-webkit-transform-origin: {{VALUE}};-moz-transform-origin: {{VALUE}};-ms-transform-origin: {{VALUE}};-o-transform-origin: {{VALUE}};transform-origin: {{VALUE}};',
					),
					'render_type' => 'template',
					'condition'   => array(
						'plus_continuous_animation' => 'yes',
						'plus_animation_effect'     => 'rotating',
					),
				)
			);

            $element->end_controls_section();
        }

		/**
		 * Apply Continuous Animation settings before rendering the widget.
		 *
		 * @since 6.4.1
		 */
        public function tpaep_render_element( $content, $element ) {
            
            $settings  = $element->get_settings_for_display();
            $widget_id = $element->get_id();

            $plus_contianimation = ! empty( $settings['plus_continuous_animation'] ) ? $settings['plus_continuous_animation'] : '';

            if ( empty( $plus_contianimation ) || 'yes' !== $plus_contianimation ) {
				return $content;
			}

            if ( 'yes' === $plus_contianimation ) {
                $uid_widget = 'plus' . $widget_id;

                $plus_animation_hover  = ! empty( $settings['plus_animation_hover'] ) ? $settings['plus_animation_hover'] : '';
                $plus_animation_effect = ! empty( $settings['plus_animation_effect'] ) ? $settings['plus_animation_effect'] : '';

                if ( 'yes' === $plus_animation_hover ) {
                    $animation_class = 'hover_';
                } else { 
                    $animation_class = 'image-';
                }

                $continuous_animation = $animation_class . $plus_animation_effect;

                $plus_before_content = '';
                $plus_after_content = '';

                $plus_before_content .= '<div id="' . esc_attr( $uid_widget ) . '" class=" plus-conti-animation ' . $continuous_animation . ' ">';

                $plus_after_content  .= '</div>';

                $plus_output = $plus_before_content . $content . $plus_after_content;

                return $plus_output;
            }

			return $content;
        }
	}
}

Tpaep_Plus_Contianimation::get_instance();