<?php
/**
 * The file that defines the core plugin class
 *
 * @link    https://posimyth.com/
 * @since   6.4.5
 *
 * @package the-plus-addons-for-elementor-page-builder
 */

namespace ElementPack\Includes\DynamicContent;

if ( ! defined( 'WPINC' ) ) {
	exit; // Exit if accessed directly
}

/*
 * Dynamic Tag option Theplus.
 * 
 * This full module registers all dynamic tags (Post, Site, User, Archive, WooCommerce).
 * 
 * @since 6.4.5
 */
if ( ! class_exists( 'Tpaep_Dynamic_Tag' ) ) {

    /**
	 * Class Tpaep_Dynamic_Tag
     * 
     * Handles registering dynamic tag groups and their individual tag classes.
     * 
     * @since 6.4.5
	 */
    class Tpaep_Dynamic_Tag{

		/**
		 * Holds the singleton instance of this class.
		 *
		 * @since 6.4.5
		 * @var   object
		 */
		private static $instance = null;

		/**
		 * Returns a singleton instance of the class.
		 *
		 * Ensures only one instance is created during execution.
		 *
		 * @since 6.4.5
		 *
		 * @param array $shortcodes Optional. An array of shortcodes to initialize the instance with.
		 * @return self The single instance of the class.
		 */
		public static function get_instance( $shortcodes = array() ) {

			if ( null === self::$instance ) {
				self::$instance = new self( $shortcodes );
			}

			return self::$instance;
		}

        /**
		 * Dynamic tag mappings for text-based tag classes.
		 *
		 * Key = tag slug  
		 * Value = class file name (autoloaded manually)
		 *
		 * @var array
		 */
        private $tp_dynamic_tags_text = array(
            /** User Tags*/
            'user-info' => 'ThePlus_Dynamic_Tag_User_Info',
            'user-meta' => 'ThePlus_Dynamic_Tag_User_Meta',

            /** Archive Tags*/
            'archive-title' => 'ThePlus_Dynamic_Tag_Archive_Title',
            'archive-meta'  => 'ThePlus_Dynamic_Tag_Archive_Meta',

            'archive-description' => 'ThePlus_Dynamic_Tag_Archive_Description',
        );

        /**
		 * Image-related dynamic tags.
		 *
         * @since 6.4.5
		 * @var array
		 */
        private $tp_dynamic_tags_image = [
            /** User Tags*/
            'user-avatar' => 'ThePlus_Dynamic_Tag_User_Avatar',

            /** Archive Tags*/
            // 'archive-image' => 'ThePlus_Dynamic_Tag_Archive_Image',
        ];

        /**
		 * URL-based dynamic tags.
		 *
         * @since 6.4.5
		 * @var array
		 */
        private $tp_dynamic_tags_url = [
            /** User Tags*/
            'user-url' => 'ThePlus_Dynamic_Tag_User_URL',
            'user-login-logout-url' => 'ThePlus_Dynamic_Tag_User_Login_Logout_URL',

            /** Archive Tags*/
            'archive-url' => 'ThePlus_Dynamic_Tag_Archive_URL',
        ];

        /**
		 * WooCommerce text dynamic tags.
		 *
		 * Loaded only when WooCommerce is active.
		 *
         * @since 6.4.5
		 * @var array
		 */
        private $tp_dynamic_tags_woo_text = [
            'product-id'    => 'ThePlus_Dynamic_Tag_Product_ID',
            'product-title' => 'ThePlus_Dynamic_Tag_Product_Title',
            'product-price' => 'ThePlus_Dynamic_Tag_Product_Price',

            'product-category'    => 'ThePlus_Dynamic_Tag_Product_Category',
            'product-description' => 'ThePlus_Dynamic_Tag_Product_Description',
            
            'product-short-description' => 'ThePlus_Dynamic_Tag_Product_Short_Description',

            'product-sku'  => 'ThePlus_Dynamic_Tag_Product_SKU',
            'product-slug' => 'ThePlus_Dynamic_Tag_Product_Slug',
            'product-tags' => 'ThePlus_Dynamic_Tag_Product_Tags',
            'product-type' => 'ThePlus_Dynamic_Tag_Product_Type',
            'product-sale' => 'ThePlus_Dynamic_Tag_Product_Sale',

            'product-rating'    => 'ThePlus_Dynamic_Tag_Product_Rating',
            'product-shipping'  => 'ThePlus_Dynamic_Tag_Product_Shipping',
            'product-attribute' => 'ThePlus_Dynamic_Tag_Product_Attribute',
            
            'product-purchase-note' => 'ThePlus_Dynamic_Tag_Product_Purchase_Note',
            'product-stock-status'  => 'ThePlus_Dynamic_Tag_Product_Stock_Status',
            'product-stock-qty'     => 'ThePlus_Dynamic_Tag_Product_Stock_Quantity',
        ];

        /**
		 * WooCommerce image-based tags.
		 *
         * @since 6.4.5
		 * @var array
		 */
        private $tp_dynamic_tags_woo_image = [
            'product-featured-image' => 'ThePlus_Dynamic_Tag_Product_Featured_Image',
            // 'product-gallery-image'  => 'ThePlus_Dynamic_Tag_Product_Gallery',
        ];

        /**
		 * WooCommerce URL-based tags.
		 *
         * @since 6.4.5
		 * @var array
		 */
        private $tp_dynamic_tags_woo_url = [
            'product-url'     => 'ThePlus_Dynamic_Tag_Product_URL',
            'product-cat-url' => 'ThePlus_Dynamic_Tag_Product_Category_URL',
            'product-tag-url' => 'ThePlus_Dynamic_Tag_Product_Tag_URL',

            'product-add-to-cart-url' => 'ThePlus_Dynamic_Tag_Product_Add_To_Cart_URL',
            'product-checkout-url'    => 'ThePlus_Dynamic_Tag_Product_Checkout_URL',
            'product-back-shop-url'   => 'ThePlus_Dynamic_Tag_Back_To_Shop_URL',
        ];

        /**
		 * ACF based tags.
		 *
         * @since 6.4.5
		 * @var array
		 */
        private $tp_dynamic_tags_acf = [
            'acf-text'    => 'ThePlus_Dynamic_Tag_ACF_Text',
            'acf-url'     => 'ThePlus_Dynamic_Tag_ACF_URL',
            'acf-image'   => 'ThePlus_Dynamic_Tag_ACF_Image',
            'acf-gallery' => 'ThePlus_Dynamic_Tag_ACF_Gallery',
        ];

        /**
		 * Constructor.
		 *
		 * Registers tag groups and individual tags on Elementor load.
		 *
		 * @since 6.4.5
		 */
        public function __construct(){
            add_action('elementor/dynamic_tags/register', [$this, 'tpae_reg_dynamic_tag_group'], 1);
            add_action('elementor/dynamic_tags/register', [$this, 'tpae_reg_dynamic_tag']);
        }

        /**
		 * Register Dynamic Tag Groups (Post, Site, User, Archive, WooCommerce).
         * 
         * @since 6.4.5
		 *
		 * @param object $dynamic_ele Elementor dynamic tag manager.
		 */
        public function tpae_reg_dynamic_tag_group( $dynamic_ele ) {
            $dynamic_ele->register_group(
                'plus-opt-user',
                [
                    'title' => esc_html__( 'Plus - User', 'theplus' )
                ]
            );

            $dynamic_ele->register_group(
                'plus-opt-archive',
                [
                    'title' => esc_html__( 'Plus - Archive', 'theplus' )
                ]
            );

            if ( class_exists( 'WooCommerce' ) ) {
                $dynamic_ele->register_group(
                    'plus-opt-woocommerce',
                    [
                        'title' => esc_html__( 'Plus - WooCommerce', 'theplus' )
                    ]
                );
            }

            if ( function_exists( 'acf_get_field_groups' ) ) {
                $dynamic_ele->register_group(
                    'plus-opt-acf',
                    [
                        'title' => esc_html__( 'Plus - ACF', 'theplus' )
                    ]
                );
            }
        }

        /**
		 * Registers all dynamic tags (text, image, url, WooCommerce).
		 *
         * @since 6.4.5
         * 
		 * @param object $dynamic_ele Elementor dynamic tag manager.
		 */
        public function tpae_reg_dynamic_tag( $dynamic_ele ) {
            $this->tpae_register_text_tags( $dynamic_ele );

            $this->tpae_register_image_tags( $dynamic_ele );

            $this->tpae_register_url_tags( $dynamic_ele );

            $this->tpae_register_woo_text_tags( $dynamic_ele );

            $this->tpae_register_woo_image_tags( $dynamic_ele );

            $this->tpae_register_woo_url_tags( $dynamic_ele );

            $this->tpae_register_acf_tags( $dynamic_ele );
        }
        
        /**
		 * Load and register text-based tag classes.
         * 
         * @since 6.4.5
		 */
        private function tpae_register_text_tags( $dynamic_tags_manager ){
            foreach ( $this->tp_dynamic_tags_text as $tag => $class ) {
                $file = THEPLUS_PATH . 'modules/extensions/dynamic-tag/tags/text/' . $tag . '.php';
                if ( file_exists( $file ) ) {
                    include( $file );
                    if ( class_exists( $class ) ) {
                        $dynamic_tags_manager->register( new $class() );
                    }
                }
            }
        }

        /**
		 * Load and register image tags.
         * 
         * @since 6.4.5
		 */
        private function tpae_register_image_tags( $dynamic_tags_manager ) {
            foreach ( $this->tp_dynamic_tags_image as $tag => $class ) {
                $file = THEPLUS_PATH . 'modules/extensions/dynamic-tag/tags/image/' . $tag . '.php';
                if ( file_exists( $file ) ) {
                    include( $file );
                    if ( class_exists( $class ) ) {
                        $dynamic_tags_manager->register( new $class() );
                    }
                }
            }
        }

        /**
		 * Load and register URL-based dynamic tags.
         * 
         * @since 6.4.5
		 */
        private function tpae_register_url_tags( $dynamic_tags_manager ) {
            foreach ( $this->tp_dynamic_tags_url as $tag => $class ) {
                $file = THEPLUS_PATH . 'modules/extensions/dynamic-tag/tags/url/' . $tag . '.php';
                if ( file_exists( $file ) ) {
                    include( $file );
                    if ( class_exists( $class ) ) {
                        $dynamic_tags_manager->register( new $class() );
                    }
                }
            }
        }

        /**
		 * Register WooCommerce text tags.
         * 
         * @since 6.4.5
		 */
        private function tpae_register_woo_text_tags( $dynamic_tags_manager ) {
            if ( class_exists( 'WooCommerce' ) ) {
                foreach ( $this->tp_dynamic_tags_woo_text as $tag => $class ) {
                    $file = THEPLUS_PATH . 'modules/extensions/dynamic-tag/tags/woocommerce/text/' . $tag . '.php';
                    if ( file_exists( $file ) ) {
                        include( $file );
                        if ( class_exists( $class ) ) {
                            $dynamic_tags_manager->register( new $class() );
                        }
                    }
                }
            }
        }

        /**
		 * Register WooCommerce image tags.
         * 
         * @since 6.4.5
		 */
        private function tpae_register_woo_image_tags( $dynamic_tags_manager ) {
            if ( class_exists( 'WooCommerce' ) ) {
                foreach ( $this->tp_dynamic_tags_woo_image as $tag => $class ) {
                    $file = THEPLUS_PATH . 'modules/extensions/dynamic-tag/tags/woocommerce/image/' . $tag . '.php';
                    if ( file_exists( $file ) ) {
                        include( $file );
                        if ( class_exists( $class ) ) {
                            $dynamic_tags_manager->register( new $class() );
                        }
                    }
                }
            }
        }

        /**
		 * Register WooCommerce URL-based tags.
         * 
         * @since 6.4.5
		 */
        private function tpae_register_woo_url_tags( $dynamic_tags_manager ) {
            if ( class_exists( 'WooCommerce' ) ) {
                foreach ( $this->tp_dynamic_tags_woo_url as $tag => $class ) {
                    $file = THEPLUS_PATH . 'modules/extensions/dynamic-tag/tags/woocommerce/url/' . $tag . '.php';
                    if ( file_exists( $file ) ) {
                        include( $file );
                        if ( class_exists( $class ) ) {
                            $dynamic_tags_manager->register( new $class() );
                        }
                    }
                }
            }
        }

        /**
		 * Register ACF based tags.
         * 
         * @since 6.4.5
		 */
        private function tpae_register_acf_tags( $dynamic_tags_manager ) {
            if ( function_exists( 'acf_get_field_groups' ) ) {
                foreach ( $this->tp_dynamic_tags_acf as $tag => $class ) {
                    $file = THEPLUS_PATH . 'modules/extensions/dynamic-tag/tags/acf/' . $tag . '.php';
                    if ( file_exists( $file ) ) {
                        include( $file );
                        if ( class_exists( $class ) ) {
                            $dynamic_tags_manager->register( new $class() );
                        }
                    }
                }
            }
        }

    }
}

/** Initialize the dynamic tag system.*/
Tpaep_Dynamic_Tag::get_instance();
