<?php

use Elementor\Core\DynamicTags\Data_Tag;
use Elementor\Modules\DynamicTags\Module;
use Elementor\Controls_Manager;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plus Addons Dynamic Tag - ACF Gallery Field
 *
 * Provides a dynamic tag for Elementor to output an ACF gallery field value.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_ACF_Gallery extends Data_Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-acf-gallery';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'ACF Gallery Field', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-acf' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::GALLERY_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
	public function is_settings_required(): bool {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'acf_field_key',
			[
				'label'  => esc_html__( 'Key', 'theplus' ),
				'type'   => Controls_Manager::SELECT,
				'groups' => $this->get_gallery_groups(),
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Select the ACF field you want to display.', 'theplus' )
					),
				),
			]
		);
	}

    /**
     * Get ACF gallery field groups and their text-safe fields.
     *
     * @since 6.4.5
     * @return void
     */
	protected function get_gallery_groups(): array {

		if ( ! function_exists( 'acf_get_field_groups' ) ) {
			return [];
		}

		$groups = [];

		foreach ( acf_get_field_groups() as $group ) {

			$fields = acf_get_fields( $group['key'] );
			if ( empty( $fields ) ) {
				continue;
			}

			$options = [];

			foreach ( $fields as $field ) {
				if ( $field['type'] === 'gallery' ) {
					$options[ $field['name'] ] = $field['label'];
				}
			}

			if ( $options ) {
				$groups[] = [
					'label'   => $group['title'],
					'options' => $options,
				];
			}
		}

		return $groups;
	}

    /**
     * Render the dynamic selected ACF gallery field on frontend.
     *
     * Gets the current selected ACF gallery field and prints its value.
     *
     * @since 6.4.5
     * @return void
     */
	public function get_value( array $options = [] ) {

        $key = $this->get_settings_for_display( 'acf_field_key' );

        if ( empty( $key ) || ! function_exists( 'get_field' ) ) {
            return [];
        }

        // Get gallery value
        if ( is_tax() || is_category() || is_tag() ) {
            $value = get_field( $key, 'term_' . get_queried_object_id() );
        } else {
            $value = get_field( $key, get_the_ID() );
        }

        if ( empty( $value ) || ! is_array( $value ) ) {
            return [];
        }

        $images = [];

        foreach ( $value as $image ) {

            // ACF Gallery image array
            if ( is_array( $image ) && ! empty( $image['ID'] ) ) {
                $images[] = [
                    'id' => (int) $image['ID'],
                ];
            }

            // ACF Gallery image ID
            if ( is_numeric( $image ) ) {
                $images[] = [
                    'id' => (int) $image,
                ];
            }
        }

        return $images;
    }

}
