<?php

use Elementor\Core\DynamicTags\Data_Tag;
use Elementor\Modules\DynamicTags\Module;
use Elementor\Controls_Manager;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plus Addons Dynamic Tag - ACF Image Field
 *
 * Provides a dynamic tag for Elementor to output an ACF image field value.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_ACF_Image extends Data_Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-acf-image';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'ACF Image Field', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-acf' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [ 
			Module::IMAGE_CATEGORY 
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
	public function is_settings_required(): bool {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'acf_field_key',
			[
				'label'  => esc_html__( 'ACF Image Field', 'theplus' ),
				'type'   => Controls_Manager::SELECT,
				'groups' => $this->get_acf_image_groups(),
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Select the ACF field you want to display.', 'theplus' )
					),
				),
			]
		);
	}

    /**
     * Get ACF image field groups and their text-safe fields.
     *
     * @since 6.4.5
     * @return void
     */
	protected function get_acf_image_groups(): array {

		if ( ! function_exists( 'acf_get_field_groups' ) ) {
			return [];
		}

		$groups = [];

		foreach ( acf_get_field_groups() as $group ) {

			$fields = acf_get_fields( $group['key'] );
			if ( empty( $fields ) ) {
				continue;
			}

			$options = [];

			foreach ( $fields as $field ) {
				if ( $field['type'] === 'image' ) {
					$options[ $field['name'] ] = $field['label'];
				}
			}

			if ( $options ) {
				$groups[] = [
					'label'   => $group['title'],
					'options' => $options,
				];
			}
		}

		return $groups;
	}

    /**
     * Render the dynamic selected ACF image field on frontend.
     *
     * Gets the current selected ACF image field and prints its value.
     *
     * @since 6.4.5
     * @return void
     */
	public function get_value( array $options = [] ) {

		$key = $this->get_settings_for_display( 'acf_field_key' );

		if ( empty( $key ) || ! function_exists( 'get_field' ) ) {
			return [];
		}

		if ( is_tax() || is_category() || is_tag() ) {
			$image = get_field( $key, 'term_' . get_queried_object_id() );
		} else {
			$image = get_field( $key, get_the_ID() );
		}

		if ( is_array( $image ) && ! empty( $image['ID'] ) ) {
			return [
				'id'  => (int) $image['ID'],
				'url' => $image['url'] ?? '',
			];
		}

		if ( is_numeric( $image ) ) {
			return [
				'id'  => (int) $image,
				'url' => wp_get_attachment_url( $image ),
			];
		}

		if ( is_string( $image ) ) {
			$id = attachment_url_to_postid( $image );
			return [
				'id'  => $id ?: null,
				'url' => $image,
			];
		}

		return [];
	}
}
