<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;
use Elementor\Controls_Manager;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - ACF URL Field
 *
 * Provides a dynamic tag for Elementor to output an ACF URL field value.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_ACF_URL extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-acf-url';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'ACF URL Field', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-acf' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::URL_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
	public function is_settings_required(): bool {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
        $this->add_control(
            'acf_field_key',
            [
                'label'  => esc_html__( 'Key', 'theplus' ),
                'type'   => Controls_Manager::SELECT,
                'groups' => $this->get_acf_url_field_groups(),
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Select the ACF field you want to display.', 'theplus' )
					),
				),
            ]
        );
	}

    /**
     * Get ACF URL field groups and their text-safe fields.
     *
     * @since 6.4.5
     * @return void
     */
	protected function get_acf_url_field_groups(): array {

        if ( ! function_exists( 'acf_get_field_groups' ) ) {
            return [];
        }

        $groups = [];

        $field_groups = acf_get_field_groups();

        foreach ( $field_groups as $acf_group ) {

            if ( empty( $acf_group['key'] ) || empty( $acf_group['title'] ) ) {
                continue;
            }

            $fields = acf_get_fields( $acf_group['key'] );

            if ( empty( $fields ) || ! is_array( $fields ) ) {
                continue;
            }

            $options = [];

            foreach ( $fields as $field ) {

                if (
                    empty( $field['name'] ) ||
                    empty( $field['label'] ) ||
                    $field['type'] !== 'url'
                ) {
                    continue;
                }

                $options[ $field['name'] ] = $field['label'];
            }

            if ( ! empty( $options ) ) {
                $groups[] = [
                    'label'   => esc_html( $acf_group['title'] ),
                    'options' => $options,
                ];
            }
        }

        return $groups;
    }

    /**
     * Render the dynamic selected ACF URL field on frontend.
     *
     * Gets the current selected ACF URL field and prints its value.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$field_name = $this->get_settings( 'acf_field_key' );

		if ( empty( $field_name ) || ! function_exists( 'get_field' ) ) {
			return;
		}

		$url = '';

		if ( is_tax() || is_category() || is_tag() ) {
			$term = get_queried_object();
			$url  = get_field( $field_name, $term );
		} else {
			$url = get_field( $field_name, get_the_ID() );
		}

		if ( empty( $url ) ) {
			return;
		}

		echo esc_url( $url );
	}
}
