<?php

use Elementor\Core\DynamicTags\Data_Tag;
use Elementor\Modules\DynamicTags\Module;
use Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Archive Image
 *
 * Provides a dynamic tag for Elementor to output the current archive image.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Archive_Image extends Data_Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name() {
		return 'plus-tag-archive-image';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title() {
		return __( 'Archive Image', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group() {
		return [ 'plus-opt-archive' ];
	}

    /**
     * Defines the category type (Image, Media) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories() {
		return [
			Module::IMAGE_CATEGORY,
			Module::MEDIA_CATEGORY,
		];
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls() {
		$this->add_control(
			'fallback_image',
			[
				'label' => __( 'Fallback Image', 'theplus' ),
				'type'  => \Elementor\Controls_Manager::MEDIA,
			]
		);
	}

	/**
	 * Return array(id + url)
	 */
	public function get_value( array $options = [] ) {

		$image_id  = 0;
		$image_url = '';

		$queried = get_queried_object();

		if ( is_author() ) {
			$author_id = get_queried_object_id();

			// Check user meta for image (common meta keys)
			$meta_keys = [ 'archive_image', 'author_image', 'profile_image' ];
			foreach ( $meta_keys as $key ) {
				$val = get_user_meta( $author_id, $key, true );
				if ( $val ) {
					if ( is_numeric( $val ) ) {
						$image_id  = (int) $val;
						$image_url = wp_get_attachment_image_url( $image_id, 'full' );
					} else {
						$image_url = esc_url_raw( $val );
					}
				}
			}

			// Fallback to avatar
			if ( empty( $image_url ) ) {
				$image_url = get_avatar_url( $author_id );
			}
		} elseif ( is_category() || is_tag() || is_tax() ) {

			$term_id = $queried->term_id ?? 0;

			// Check common term meta keys
			$meta_keys = [ 'archive_image', 'thumbnail_id', 'term_image', 'category_image' ];

			foreach ( $meta_keys as $key ) {
				$value = get_term_meta( $term_id, $key, true );

				if ( $value ) {
					if ( is_numeric( $value ) ) {
						$image_id  = (int) $value;
						$image_url = wp_get_attachment_image_url( $image_id, 'full' );
					} else {
						$image_url = esc_url_raw( $value );
					}
				}
			}
		}

		if ( empty( $image_url ) ) {
			$fallback = $this->get_settings( 'fallback_image' );
			if ( ! empty( $fallback['url'] ) ) {
				return [
					'id'  => $fallback['id'] ?? 0,
					'url' => $fallback['url']
				];
			}
		}

		if ( empty( $image_url ) ) {
			$image_url = \Elementor\Utils::get_placeholder_image_src();
		}

		return [
			'id'  => $image_id,
			'url' => $image_url,
		];
	}

    /**
     * Render the dynamic archive image on frontend.
     *
     * Gets the current archive image and prints it.
     *
     * @since 6.4.5
     * @return void
     */
	public function render() {
		$value = $this->get_value();
		echo esc_url( $value['url'] ?? '' );
	}
}
