<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Archive Meta
 *
 * Provides a dynamic tag for Elementor to output the current archive meta.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Archive_Meta extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
    public function get_name(): string {
        return 'plus-tag-archive-meta';
    }

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
    public function get_title(): string {
        return esc_html__( 'Archive Meta', 'theplus' );
    }

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
    public function get_group(): array {
        return ['plus-opt-archive'];
    }

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
    public function get_categories(): array {
        return [
            Module::TEXT_CATEGORY,
        ];
    }

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
    protected function register_controls(): void {
        $this->add_control(
            'meta_key',
            [
                'label'       => esc_html__( 'Meta Key', 'theplus' ),
                'type'        => \Elementor\Controls_Manager::TEXT,
                'placeholder' => esc_html__( 'Enter meta key', 'theplus' ),
                'ai'          => 'false',
                'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Show meta value based on key', 'theplus' )
					),
				),
            ]
        );
    }

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
    protected function register_advanced_section(): void {}

    /**
     * Render the dynamic archive meta on frontend.
     *
     * Gets the current archive meta and prints it.
     *
     * @since 6.4.5
     * @return void
     */
    public function render(): void {

        $key = $this->get_settings('meta_key'); 

        if ( empty( $key ) ) {
            return;
        }

        $meta_value = '';

        if ( is_category() || is_tag() || is_tax() ) {
            $term_id    = get_queried_object_id();
            $meta_value = get_term_meta( $term_id, $key, true );
        } elseif ( is_author() ) {
            $author_id  = get_queried_object_id();
            $meta_value = get_user_meta( $author_id, $key, true );
        }

        if ( is_array( $meta_value ) ) {
            $meta_value = implode( ', ', $meta_value );
        }

        echo wp_kses_post( $this->apply_word_limit( $meta_value ) );
    }

}
