<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Archive Title
 *
 * Provides a dynamic tag for Elementor to output the current archive title.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Archive_Title extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-archive-title';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Archive Title', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-archive' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {

		$this->add_control(
			'include_context',
			[
				'label'   => esc_html__( 'Include Context', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);
	}

    /**
     * Render the dynamic archive title on frontend.
     *
     * Gets the current archive title and prints it.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

        $include_context = $this->get_settings('include_context') === 'yes';
        $title = '';

        if ( is_archive() || is_home() || is_post_type_archive() || is_category() || is_tag() || is_tax() || is_author() || is_date() ) {

            $archive_title = get_the_archive_title();

            if ( ! $include_context ) {
                $archive_title = wp_strip_all_tags( $archive_title );
                $archive_title = preg_replace('/^[^:]+:\s*/u', '', $archive_title);
            }

            $title = $archive_title;
        } elseif ( is_singular() ) {

            $post_id = get_the_ID();
            $post_type = get_post_type( $post_id );

            $singular_title = get_the_title( $post_id );

            if ( $include_context ) {
                $obj = get_post_type_object( $post_type );

                if ( $obj && isset($obj->labels->singular_name) ) {
                    $label = $obj->labels->singular_name;
                    $title = $label . ': ' . $singular_title;
                } else {
                    $title = $singular_title;
                }

            } else {
                $title = $singular_title;
            }
        } else {
            $qo = get_queried_object();

            if ( isset($qo->name) ) {
                $title = $qo->name;
            } elseif ( isset($qo->post_title) ) {
                $title = $qo->post_title;
            }
        }

        echo wp_kses_post( $title );
    }
}
