<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - User Info
 *
 * Provides a dynamic tag for Elementor to output the current user info.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_User_Info extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-user-info';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'User Info', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-user' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'user_field',
			[
				'label'   => esc_html__( 'User Field', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => 'display_name',
				'options' => [
					'id'           => esc_html__( 'ID', 'theplus' ),
					'name'         => esc_html__( 'Name', 'theplus' ),
					'email'        => esc_html__( 'Email', 'theplus' ),
					'website'      => esc_html__( 'Website', 'theplus' ),
					'display_name' => esc_html__( 'Display Name', 'theplus' ),
					'nickname'     => esc_html__( 'Nickname', 'theplus' ),
					'first_name'   => esc_html__( 'First Name', 'theplus' ),
					'last_name'    => esc_html__( 'Last Name', 'theplus' ),
					'description'  => esc_html__( 'Description', 'theplus' ),
					'role'         => esc_html__( 'Role', 'theplus' ),
				],
			]
		);
	}

	/**
	 * Resolve the user ID based on the current context.
	 *
	 * @since 6.4.5
	 * @return int|false User ID or false if no user found.
	 */
	protected function resolve_user_id() {

		if ( ! is_user_logged_in() ) {
			return false;
		}

		if ( is_author() ) {
			$author_id = get_queried_object_id();
			if ( $author_id ) {
				return (int) $author_id;
			}
		}

		if ( is_singular() ) {
			$post_id = get_the_ID();
			if ( $post_id ) {
				$author_id = get_post_field( 'post_author', $post_id );
				if ( $author_id ) {
					return (int) $author_id;
				}
			}
		}

		$qo = get_queried_object();
		if ( $qo ) {
			if ( isset( $qo->ID ) && user_can( $qo->ID, 'read' ) ) {
				return (int) $qo->ID;
			}
			if ( isset( $qo->user_id ) && $qo->user_id ) {
				return (int) $qo->user_id;
			}
		}

		$current = get_current_user_id();
		if ( $current ) {
			return (int) $current;
		}

		return false;
	}

    /**
     * Render the dynamic user info on frontend.
     *
     * Gets the current user and prints its details.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$field = $this->get_settings( 'user_field' );
		$value = '';

		$user_id = $this->resolve_user_id();

		if ( ! $user_id ) {
			return;
		}

		$user = get_userdata( $user_id );

		switch ( $field ) {

			case 'id':
				$value = $user_id;
				break;

			case 'name':
				if ( $user && ! empty( $user->display_name ) ) {
					$value = $user->display_name;
				} elseif ( $user ) {
					$value = $user->user_login;
				} else {
					$value = '';
				}
				break;

			case 'email':
				$value = $user ? $user->user_email : '';
				break;

			case 'website':
				$value = $user ? $user->user_url : '';
				break;

			case 'display_name':
				$value = $user ? $user->display_name : '';
				break;

			case 'nickname':
				$value = $user ? $user->nickname : '';
				if ( empty( $value ) && $user ) {
					$value = $user->display_name;
				}
				break;

			case 'first_name':
				$value = $user ? $user->first_name : '';
				break;

			case 'last_name':
				$value = $user ? $user->last_name : '';
				break;

			case 'description':
				$value = $user ? $user->description : '';
				break;

			case 'role':
				if ( $user && ! empty( $user->roles ) ) {
					$value = implode( ', ', $user->roles );
				} else {
					$value = '';
				}
				break;

			default:
				$value = '';
				break;
		}

		if ( is_scalar( $value ) && $value !== '' ) {
			echo esc_html( $value );
		}
	}
}
