<?php

use Elementor\Core\DynamicTags\Data_Tag;
use Elementor\Modules\DynamicTags\Module;
use Elementor\Controls_Manager;
use Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Featured Image
 *
 * Provides a dynamic tag for Elementor to output the current product featured image.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Featured_Image extends Data_Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-featured-image';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Featured Image', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Image, Media) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::IMAGE_CATEGORY,
			Module::MEDIA_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
	public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'fallback_image',
			[
				'label'       => esc_html__( 'Fallback Image', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::MEDIA,
				'ai'          => false,
                'description' => wp_kses_post(
                    sprintf(
                        '<p class="tp-controller-label-text"><i>%s</i></p>',
                        esc_html__( 'Displayed when the product has no featured image.', 'theplus' )
                    )
                ),
			]
		);
	}

	/**
	 * Resolves the current product ID in various contexts.
	 *
	 * @since 6.4.5
	 * @return int|null Product ID or null if not found.
	 */
	private function resolve_product_id() {
		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product();
			if ( $product && is_callable( [ $product, 'get_id' ] ) ) {
				return (int) $product->get_id();
			}
		}

		$post_id = get_the_ID();
		if ( $post_id && get_post_type( $post_id ) === 'product' ) {
			return (int) $post_id;
		}

		if ( isset( $_GET['product_id'] ) ) {
			$maybe = absint( $_GET['product_id'] );
			if ( $maybe ) {
				return $maybe;
			}
		}

		return null;
	}

	/**
	 * Get the dynamic tag value.
	 *
	 * @since 6.4.5
	 * @return array|false Array with 'id' and 'url' keys, or false if not found.
	 */
	public function get_value( array $options = [] ) {
		$product_id = $this->resolve_product_id();
		$settings   = $this->get_settings();

		if ( $product_id ) {
			$thumb_id = get_post_thumbnail_id( $product_id );
			if ( $thumb_id ) {
				$url = wp_get_attachment_image_url( $thumb_id, 'full' );
				if ( $url ) {
					return [
						'id'  => (int) $thumb_id,
						'url' => $url,
					];
				}
			}
		}

		if ( ! empty( $settings['fallback_image']['url'] ) ) {
			return [
				'id'  => $settings['fallback_image']['id'] ?? 0,
				'url' => $settings['fallback_image']['url'],
			];
		}

		return [
			'id'  => 0,
			'url' => Utils::get_placeholder_image_src(),
		];
	}

    /**
     * Render the dynamic post title on frontend.
     *
     * Gets the current post ID and prints its title.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {
		$value = $this->get_value();

		if ( ! empty( $value['url'] ) ) {
			echo esc_url( $value['url'] );
		}
	}
}
