<?php

use Elementor\Core\DynamicTags\Data_Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Gallery Image
 *
 * Provides a dynamic tag for Elementor to output the current product gallery image.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Gallery extends Data_Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-gallery-image';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Gallery Image', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Image, Media) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::IMAGE_CATEGORY,
			Module::MEDIA_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
	public function is_settings_required(): bool {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'gallery_index',
			[
				'label'       => esc_html__( 'Gallery Image Index', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'default'     => 0,
				'min'         => 0,
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( '0 for first gallery image, 1 for second image, etc.', 'theplus' )
					),
				),
			]
		);
		$this->add_control(
			'fallback',
			[
				'label'       => esc_html__( 'Fallback', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::MEDIA,
				'dynamic'     => [ 
					'active' => false, 
				],
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'This image will be used if the gallery image is not found', 'theplus' )
					),
				),
			]
		);
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_advanced_section(): void {}

	/**
	 * Get the dynamic tag value of the product gallery image.
	 *
	 * @since 6.4.5
	 * @return array|false Array with 'id' and 'url' keys, or false if not found.
	 */
	public function get_value( array $options = [] ) {

		$settings = $this->get_settings();
		$product_id = $this->get_product_id();
		$gallery_index = absint( $settings['gallery_index'] ?? 0 );

		if ( ! $product_id ) {
			return $this->get_settings( 'fallback' );
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return $this->get_settings( 'fallback' );
		}

		$gallery_image_ids = $product->get_gallery_image_ids();

		if ( empty( $gallery_image_ids ) ) {
			return $this->get_settings( 'fallback' );
		}

		$image_id = $gallery_image_ids[ $gallery_index ] ?? null;

		if ( $image_id ) {
			$image_data = [
				'id' => $image_id,
				'url' => wp_get_attachment_image_url( $image_id, 'full' ),
			];
		} else {
			$image_data = $this->get_settings( 'fallback' );
		}

		return $image_data;
	}
}
