<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Title
 *
 * Provides a dynamic tag for Elementor to output the current product title.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Attribute extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-attribute';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Attribute', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

	/**
	 * Register controls for this dynamic tag.
	 *
	 * @since 6.4.5
	 * @return void
	 */
	protected function register_controls(): void {

		$attributes = wc_get_attribute_taxonomies();
		$options    = [];

		$options[''] = esc_html__( 'Select Attribute', 'theplus' );

		if ( ! empty( $attributes ) ) {
			foreach ( $attributes as $attr ) {
				$options[ $attr->attribute_name ] = $attr->attribute_label;
			}
		}

		$this->add_control(
			'product_attribute_name',
			[
				'label'   => esc_html__( 'Select Attribute', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => $options,
				'default' => '',
			]
		);
		$this->add_control(
			'separator',
			[
				'label'       => esc_html__( 'Separator', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'default'     => ', ',
				'ai'          => false,
				'placeholder' => 'Add Separator',
			]
		);
	}

    /**
     * Render the dynamic product title on frontend.
     *
     * Gets the current product ID and prints its title.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product_id = get_the_ID();
		if ( ! $product_id ) {
			return;
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return;
		}

		$attr_name = $this->get_settings( 'product_attribute_name' );
		if ( empty( $attr_name ) ) {
			return;
		}

		$separator = $this->get_settings( 'separator' );
		$separator = isset( $separator ) ? $separator : ', ';

		$taxonomy = 'pa_' . $attr_name;

		$terms = wp_get_post_terms( $product_id, $taxonomy );

		if ( empty( $terms ) || is_wp_error( $terms ) ) {
			return;
		}

		$values = wp_list_pluck( $terms, 'name' );

		echo esc_html( implode( $separator, $values ) );
	}

}
