<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Category
 *
 * Provides a dynamic tag for Elementor to output the current product category.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Category extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-category';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Category', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'separator',
			[
				'label'   => esc_html__( 'Separator', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					','  => esc_html__( 'Comma (,)', 'theplus' ),
					'/'  => esc_html__( 'Slash (/)', 'theplus' ),
					'|'  => esc_html__( 'Pipe (|)', 'theplus' ),
					'custom' => esc_html__( 'Custom', 'theplus' ),
				],
				'default' => ',',
			]
		);
		$this->add_control(
			'custom_separator',
			[
				'label'     => esc_html__( 'Custom Separator', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => ', ',
				'ai'        => false,
				'condition' => [
					'separator' => 'custom',
				],
			]
		);
	}

	/**
	 * Resolves the current WooCommerce product.
	 *
	 * @since 6.4.5
	 * @return WC_Product|null
	 */
	private function resolve_product() {

		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product();
			if ( $product ) {
				return $product;
			}
		}

		$post_id = get_the_ID();

		if ( $post_id && 'product' === get_post_type( $post_id ) ) {
			return wc_get_product( $post_id );
		}

		return null;
	}

    /**
     * Render the dynamic product category on frontend.
     *
     * Gets the current product category and prints it.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product = $this->resolve_product();

		if ( ! $product ) {
			return;
		}

		$terms = get_the_terms( $product->get_id(), 'product_cat' );

		if ( empty( $terms ) || is_wp_error( $terms ) ) {
			return;
		}

		$names = wp_list_pluck( $terms, 'name' );

		$sep_type = $this->get_settings( 'separator' );

		if ( 'custom' === $sep_type ) {
			$separator = $this->get_settings( 'custom_separator' );
		} else {
			$separator = ' ' . $sep_type . ' ';
		}

		echo esc_html( implode( $separator, $names ) );
	}
}
