<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Description
 *
 * Provides a dynamic tag for Elementor to output the current product description.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Description extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-description';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Description', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text, Post Meta) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
			Module::POST_META_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

	/**
	 * Register controls for this dynamic tag.
	 *
	 * @since 6.4.5
	 * @return void
	 */
	protected function register_controls(): void {
		$this->add_control(
			'word_limit',
			[
				'label'       => esc_html__( 'Word Limit', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'default'     => 0,
				'min'         => 0,
				'description' => wp_kses_post(
                    sprintf(
                        '<p class="tp-controller-label-text"><i>%s</i></p>',
                        esc_html__( 'Set 0 to show full description.', 'theplus' )
                    )
                ),
			]
		);
	}

    /**
     * Render the dynamic product description on frontend.
     *
     * Gets the current product description and prints it.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$settings   = $this->get_settings();
		$word_limit = ! empty( $settings['word_limit'] ) ? (int) $settings['word_limit'] : 0;

		$description = '';

		$product = function_exists( 'wc_get_product' ) ? wc_get_product() : false;

		if ( $product ) {
			$description = $product->get_description();
		} else {
			$post_id = get_the_ID();

			if ( ! $post_id || get_post_type( $post_id ) !== 'product' ) {
				return;
			}

			$post = get_post( $post_id );

			if ( ! $post ) {
				return;
			}

			$description = $post->post_content;
		}

		if ( empty( $description ) ) {
			return;
		}

		$description = wp_strip_all_tags( $description );

		if ( $word_limit > 0 ) {
			$description = wp_trim_words( $description, $word_limit );
		}

		echo esc_html( $description );
	}
}
