<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Price
 *
 * Provides a dynamic tag for Elementor to output the current product prices.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Price extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-price-html';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Price', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'price_type',
			[
				'label'   => esc_html__( 'Price Type', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'auto'    => esc_html__( 'Auto (WC formatted HTML)', 'theplus' ),
					'regular' => esc_html__( 'Regular Price', 'theplus' ),
					'sale'    => esc_html__( 'Sale Price', 'theplus' ),
				],
				'default' => 'auto',
			]
		);
	}

	/**
	 * Resolve current product in various contexts
	 *
	 * @return WC_Product|null
	 */
	private function resolve_product() {
		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product();
			if ( $product ) {
				return $product;
			}
		}

		$post_id = get_the_ID();
		if ( $post_id && 'product' === get_post_type( $post_id ) ) {
			return wc_get_product( $post_id );
		}

		return null;
	}

    /**
     * Render the dynamic product price on frontend.
     *
     * Gets the current product price and prints it.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product = $this->resolve_product();

		if ( ! $product ) {
			return;
		}

		$price_type = $this->get_settings( 'price_type' );

		if ( 'auto' === $price_type ) {
			$price_html = $product->get_price_html();

			echo wp_kses_post( $price_html );
			return;
		}

		if ( 'regular' === $price_type ) {
			$regular = $product->get_regular_price();
			if ( '' === $regular || null === $regular ) {
				echo wp_kses_post( $product->get_price_html() );
				return;
			}

			echo wp_kses_post( wc_price( (float) $regular ) );
			return;
		}

		if ( 'sale' === $price_type ) {
			$sale = $product->get_sale_price();
			if ( '' === $sale || null === $sale ) {
				echo wp_kses_post( $product->get_price_html() );
				return;
			}
			echo wp_kses_post( wc_price( (float) $sale ) );
			return;
		}
	}
}
