<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Sale
 *
 * Provides a dynamic tag for Elementor to output the current product sale information.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Sale extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-sale';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Sale', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'discount_type',
			[
				'label'   => esc_html__( 'Discount Type', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => 'amount',
				'options' => [
					'amount'  => esc_html__( 'Discount Amount', 'theplus' ),
					'percent' => esc_html__( 'Discount Percentage', 'theplus' ),
					'custom'  => esc_html__( 'Custom', 'theplus' ),
				],
			]
		);
		$this->add_control(
			'custom_text',
			[
				'label'     => esc_html__( 'Custom Text', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => 'On Sale',
				'ai'        => false,
				'condition' => [
					'discount_type' => 'custom',
				],
			]
		);
		$this->add_control(
			'show_discount',
			[
				'label'     => esc_html__( 'Show Discount', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'label_on'  => esc_html__( 'Yes', 'theplus' ),
				'label_off' => esc_html__( 'No', 'theplus' ),
				'default'   => 'no',
				'condition' => [
					'discount_type' => 'custom',
				],
			]
		);
		$this->add_control(
			'discount_position',
			[
				'label'     => esc_html__( 'Discount Position', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'after',
				'options'   => [
					'before' => esc_html__( 'Before Text', 'theplus' ),
					'after'  => esc_html__( 'After Text', 'theplus' ),
				],
				'condition' => [
					'discount_type' => 'custom',
					'show_discount' => 'yes',
				],
			]
        );
	}

    /**
     * Render the dynamic product title on frontend.
     *
     * Gets the current product ID and prints its title.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product_id = get_the_ID();
		if ( ! $product_id ) {
			return;
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return;
		}

		$regular = (float) $product->get_regular_price();
		$sale    = (float) $product->get_sale_price();

		if ( $regular <= 0 || $sale <= 0 ) {
			return;
		}

		$discount_amount  = $regular - $sale;
		$discount_percent = round( ( $discount_amount / $regular ) * 100 );

		$type             = $this->get_settings( 'discount_type' );
		$custom_text      = $this->get_settings( 'custom_text' );
		$show_discount    = $this->get_settings( 'show_discount' );
		$discount_pos     = $this->get_settings( 'discount_position' );

		switch ( $type ) {

			case 'amount':
				// echo esc_html( wc_price( $discount_amount ) );
				echo esc_html( wp_strip_all_tags( wc_price( $discount_amount ) ) );
				break;

			case 'percent':
				echo esc_html( $discount_percent . '%' );
				break;

			case 'custom':

				$discount = $discount_percent . '%';

				if ( 'yes' !== $show_discount ) {
					echo esc_html( $custom_text );
					return;
				}

				if ( 'before' === $discount_pos ) {
					$output = $discount . ' ' . $custom_text;
				} else {
					$output = $custom_text . ' ' . $discount;
				}

				echo esc_html( $output );
				break;
		}
	}
}
