<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Shipping
 *
 * Provides a dynamic tag for Elementor to output the current product shipping information.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Shipping extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-shipping';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Shipping', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

	
    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'shipping_type',
			[
				'label'   => esc_html__( 'Shipping Type', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => 'weight',
				'options' => [
					'weight'     => esc_html__( 'Weight', 'theplus' ),
					'dimensions' => esc_html__( 'Dimensions', 'theplus' ),
					'class'      => esc_html__( 'Shipping Class', 'theplus' ),
					'custom'     => esc_html__( 'Custom Text', 'theplus' ),
				],
			]
		);
		$this->add_control(
			'dimension_type',
			[
				'label'     => esc_html__( 'Dimension Type', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'all',
				'options'   => [
					'all'    => esc_html__( 'All', 'theplus' ),
					'length' => esc_html__( 'Length', 'theplus' ),
					'width'  => esc_html__( 'Width', 'theplus' ),
					'height' => esc_html__( 'Height', 'theplus' ),
				],
				'condition' => [
					'shipping_type' => 'dimensions',
				],
			]
		);
		$this->add_control(
			'custom_text',
			[
				'label'       => esc_html__( 'Custom Text', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter custom shipping text', 'theplus' ),
				'ai'          => false,
				'condition'   => [
					'shipping_type' => 'custom',
				],
			]
		);
		$this->add_control(
			'show_label',
			[
				'label'        => esc_html__( 'Show Label?', 'theplus' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'theplus' ),
				'label_off'    => esc_html__( 'No', 'theplus' ),
				'return_value' => 'yes',
				'default'      => '',
			]
		);
		$this->add_control(
			'label_text',
			[
				'label'     => esc_html__( 'Label Text', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'ai'        => false,
				'default'   => esc_html__( 'Shipping', 'theplus' ),
				'condition' => [
					'show_label' => 'yes',
				],
			]
		);
		$this->add_control(
			'label_position',
			[
				'label'     => esc_html__( 'Label Position', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'before',
				'options'   => [
					'before' => esc_html__( 'Before', 'theplus' ),
					'after'  => esc_html__( 'After', 'theplus' ),
				],
				'condition' => [
					'show_label' => 'yes',
				],
			]
		);
	}

    /**
     * Render the dynamic product shipping on frontend.
     *
     * Gets the current product ID and prints its shipping information.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product_id = get_the_ID();
		if ( ! $product_id ) {
			return;
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return;
		}

		$shipping_type  = $this->get_settings( 'shipping_type' );
		$dimension_type = $this->get_settings( 'dimension_type' );
		$custom_text    = $this->get_settings( 'custom_text' );

		$show_label     = $this->get_settings( 'show_label' );
		$label_text     = $this->get_settings( 'label_text' );
		$label_position = $this->get_settings( 'label_position' );

		$value = '';

		switch ( $shipping_type ) {

			case 'weight':
				$weight = $product->get_weight();
				if ( $weight ) {
					$value = wc_format_weight( $weight );
				}
				break;

			case 'dimensions':
				$length = $product->get_length();
				$width  = $product->get_width();
				$height = $product->get_height();

				if ( $dimension_type === 'length' ) {
					$value = $length;
				} elseif ( $dimension_type === 'width' ) {
					$value = $width;
				} elseif ( $dimension_type === 'height' ) {
					$value = $height;
				} else {
					$value = wc_format_dimensions( [ $length, $width, $height ] );
				}
				break;

			case 'class':
				$class_slug = $product->get_shipping_class();
				if ( $class_slug ) {
					$term  = get_term_by( 'slug', $class_slug, 'product_shipping_class' );
					$value = $term ? $term->name : $class_slug;
				}
				break;

			case 'custom':
				$value = $custom_text;
				break;
		}

		if ( empty( $value ) ) {
			return;
		}

		if ( $show_label === 'yes' ) {
			if ( $label_position === 'before' ) {
				$value = $label_text . ' ' . $value;
			} else {
				$value = $value . ' ' . $label_text;
			}
		}

		echo wp_kses_post( $value );
	}
}
