<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Stock Quantity
 *
 * Provides a dynamic tag for Elementor to output the current product stock quantity.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Stock_Quantity extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-stock-qty';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Stock Quantity', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'qty_type',
			[
				'label'   => esc_html__( 'Quantity Type', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'stock'     => esc_html__( 'Stock Quantity', 'theplus' ),
					'backorder' => esc_html__( 'Backorder Quantity', 'theplus' ),
				],
				'default' => 'stock',
			]
		);
	}

	/**
	 * Resolve the current product object.
	 *
	 * @since 6.4.5
	 * @return WC_Product|null
	 */
	private function resolve_product() {
		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product();
			if ( $product ) {
				return $product;
			}
		}

		$post_id = get_the_ID();
		if ( $post_id && 'product' === get_post_type( $post_id ) ) {
			return wc_get_product( $post_id );
		}

		return null;
	}

    /**
     * Render the dynamic product stock quantity on frontend.
     *
     * Gets the current product ID and prints its stock quantity.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product = $this->resolve_product();

		if ( ! $product ) {
			return;
		}

		$qty_type = $this->get_settings( 'qty_type' );

		if ( 'backorder' === $qty_type ) {

			$backorders = $product->backorders_allowed() ? $product->get_backorders() : 0;

			echo esc_html( $backorders ? $backorders : '0' );
			return;
		}

		// Default: Stock Quantity
		$qty = $product->get_stock_quantity(); // may return null

		if ( $qty === null ) {
			echo esc_html__( 'N/A', 'theplus' );
			return;
		}

		echo esc_html( $qty );
	}
}
