<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Title
 *
 * Provides a dynamic tag for Elementor to output the current product title.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Stock_Status extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-stock-status';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Stock Status', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
    public function is_settings_required() {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'stock_type',
			[
				'label'   => esc_html__( 'Status Type', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'auto' => esc_html__( 'Auto (Woo Text)', 'theplus' ),
					'raw'  => esc_html__( 'Raw Status (instock/outofstock/onbackorder)', 'theplus' ),
				],
				'ai'      => false,
				'default' => 'auto',
			]
		);
	}

	/**
	 * Resolve the current product object.
	 *
	 * @since 6.4.5
	 * @return WC_Product|null
	 */
	private function resolve_product() {
		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product();
			if ( $product ) {
				return $product;
			}
		}

		$post_id = get_the_ID();
		if ( $post_id && 'product' === get_post_type( $post_id ) ) {
			return wc_get_product( $post_id );
		}

		return null;
	}

    /**
     * Render the dynamic product stock status on frontend.
     *
     * Gets the current product ID and prints its stock status.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product = $this->resolve_product();

		if ( ! $product ) {
			return;
		}

		$stock_type = $this->get_settings( 'stock_type' );
		$status_raw = $product->get_stock_status();

		if ( 'raw' === $stock_type ) {
			echo esc_html( $status_raw );
			return;
		}

		$stock_text = wc_get_stock_html( $product );

		if ( $stock_text ) {
			echo wp_kses_post( wp_strip_all_tags( $stock_text ) );
			return;
		}

		switch ( $status_raw ) {
			case 'instock':
				echo esc_html__( 'In stock', 'theplus' );
				break;
			case 'outofstock':
				echo esc_html__( 'Out of stock', 'theplus' );
				break;
			case 'onbackorder':
				echo esc_html__( 'Available on backorder', 'theplus' );
				break;
		}
	}
}
