<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Tags
 *
 * Provides a dynamic tag for Elementor to output the current product tags.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Tags extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-tags';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Tags', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (Text, Post Meta) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::TEXT_CATEGORY,
			Module::POST_META_CATEGORY,
		];
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'type',
			[
				'label'   => esc_html__( 'Tag Type', 'theplus' ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'first'   => esc_html__( 'First Tag', 'theplus' ),
					'all'     => esc_html__( 'All Tags', 'theplus' ),
				],
				'default' => 'all',
			]
		);
		$this->add_control(
			'separator',
			[
				'label'     => esc_html__( 'Separator (for multiple)', 'theplus' ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => ', ',
				'ai'        => false,
				'condition' => [
					'type' => 'all',
				],
			]
		);
	}

	/**
	 * Resolve the current product object.
	 *
	 * @since 6.4.5
	 * @return WC_Product|null
	 */
	private function resolve_product() {

		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product();
			if ( $product ) {
				return $product;
			}
		}

		$post_id = get_the_ID();
		if ( $post_id && 'product' === get_post_type( $post_id ) ) {
			return wc_get_product( $post_id );
		}

		return null;
	}

    /**
     * Render the dynamic product title on frontend.
     *
     * Gets the current product ID and prints its title.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product = $this->resolve_product();

		if ( ! $product ) {
			return;
		}

		$tags = get_the_terms( $product->get_id(), 'product_tag' );

		if ( empty( $tags ) || is_wp_error( $tags ) ) {
			return;
		}

		$type = $this->get_settings( 'type' );

		if ( 'first' === $type ) {
			$first = $tags[0]->name;
			echo esc_html( $first );
			return;
		}

		$separator = $this->get_settings( 'separator' ) ?: ', ';

		$tag_names = wp_list_pluck( $tags, 'name' );

		echo esc_html( implode( $separator, $tag_names ) );
	}
}
