<?php

use Elementor\Core\DynamicTags\Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Category URL
 *
 * Provides a dynamic tag for Elementor to output the current product category url.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Category_URL extends Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-category-url';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Category URL', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (URL) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::URL_CATEGORY,
		];
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_advanced_section() {}

	/**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {}

	/**
	 * Resolve the current product object.
	 *
	 * @since 6.4.5
	 * @return WC_Product|null
	 */
	private function resolve_product() {

		if ( function_exists( 'wc_get_product' ) ) {
			$product = wc_get_product();
			if ( $product ) {
				return $product;
			}
		}

		$post_id = get_the_ID();
		if ( $post_id && 'product' === get_post_type( $post_id ) ) {
			return wc_get_product( $post_id );
		}

		return null;
	}

    /**
     * Render the dynamic product Category URL on frontend.
     *
     * Gets the current product ID and prints its Category URL.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {

		$product = $this->resolve_product();

		if ( ! $product ) {
			return;
		}

		$terms = get_the_terms( $product->get_id(), 'product_cat' );

		if ( empty( $terms ) || is_wp_error( $terms ) ) {
			return;
		}

		$term = reset( $terms );
		$url  = get_term_link( $term );

		if ( ! is_wp_error( $url ) ) {
			echo esc_url( $url );
		}
	}
}
