<?php

use Elementor\Core\DynamicTags\Data_Tag;
use Elementor\Modules\DynamicTags\Module;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Plus Addons Dynamic Tag - Product Checkout URL
 *
 * Provides a dynamic tag for Elementor to output the current product checkout url.
 *
 * @since 6.4.5
 */
class ThePlus_Dynamic_Tag_Product_Checkout_URL extends Data_Tag {

    /**
     * Unique dynamic tag name used internally by Elementor.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_name(): string {
		return 'plus-tag-product-checkout-url';
	}

    /**
     * Label shown in Elementor Dynamic Tags list.
     *
     * @since 6.4.5
     * @return string
     */
	public function get_title(): string {
		return esc_html__( 'Product Checkout URL', 'theplus' );
	}

    /**
     * Registers the group under which this tag will appear.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_group(): array {
		return [ 'plus-opt-woocommerce' ];
	}

    /**
     * Defines the category type (URL) for this dynamic tag.
     *
     * @since 6.4.5
     * @return array
     */
	public function get_categories(): array {
		return [
			Module::URL_CATEGORY,
		];
	}

    /**
     * Indicates that settings panel should be shown (even if empty).
     *
     * @since 6.4.5
     * @return bool
     */
	public function is_settings_required(): bool {
		return true;
	}

    /**
     * Register controls for this dynamic tag.
     *
     * @since 6.4.5
     * @return void
     */
	protected function register_controls(): void {
		$this->add_control(
			'checkout_type',
			[
				'label'       => esc_html__( 'Checkout Type', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::SELECT,
				'default'     => 'direct',
				'options'     => [
					'direct'      => esc_html__( 'Direct to Checkout', 'theplus' ),
					'add_to_cart' => esc_html__( 'Add to Cart & Checkout', 'theplus' ),
				],
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Choose whether to go directly to checkout or add product to cart first', 'theplus' )
					),
				),
			]
		);
		$this->add_control(
			'quantity',
			[
				'label'       => esc_html__( 'Quantity', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'default'     => 1,
				'min'         => 1,
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Number of items to add to cart', 'theplus' )
					),
				),
				'condition'   => [
					'checkout_type' => 'add_to_cart',
				],
			]
		);
		$this->add_control(
			'fallback',
			[
				'label'       => esc_html__( 'Fallback URL', 'theplus' ),
				'type'        => \Elementor\Controls_Manager::URL,
				'dynamic'     => [
					'active' => false,
				],	
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Used if no product is selected', 'theplus' )
					),
				),
			]
		);
	}

	/**
	 * Get the dynamic tag value.
	 *
	 * @since 6.4.5
	 * @param array $options Optional. Additional options.
	 * @return string
	 */
	public function get_value( array $options = [] ) {

		$settings = $this->get_settings();

		$product_id    = get_the_ID();
		$checkout_type = $settings['checkout_type'] ?? 'direct';

		if ( $checkout_type === 'direct' ) {
			return wc_get_checkout_url();
		}

		// $product_id = $this->get_product_id();
		if ( ! $product_id ) {
			return wc_get_checkout_url();
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return wc_get_checkout_url();
		}

		$quantity = max( 1, (int) ( $settings['quantity'] ?? 1 ) );

		// For variable or grouped products, redirect to product page
		if ( $product->is_type( ['variable', 'grouped'] ) ) {
			return get_permalink( $product_id );
		}

		// Simple product: add to cart and redirect to checkout
		return add_query_arg(
			[
				'add-to-cart' => $product_id,
				'quantity'    => $quantity,
				'checkout'    => '1',
			],
			wc_get_cart_url()
		);
	}

    /**
     * Render the dynamic product Checkout URL on frontend.
     *
     * Gets the current product ID and prints its Checkout URL.
     *
     * @since 6.4.5
     * @return void
     */
	public function render(): void {
		echo esc_url( $this->get_value() );
	}
}
