<?php
/**
 * Widget Name: Scroll Sequence
 * Description: Image Scroll Sequence
 * Author: Theplus
 * Author URI: https://posimyth.com
 *
 * @package ThePlus
 */

namespace TheplusAddons\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

use TheplusAddons\Theplus_Element_Load;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class ThePlus_Scroll_Sequence
 */
class ThePlus_Scroll_Sequence extends Widget_Base {

	/**
	 * Document Link For Need help.
	 *
	 * @var tp_doc of the class.
	 */
	public $tp_doc = THEPLUS_TPDOC;

	/**
	 * Helpdesk Link For Need help.
	 *
	 * @var tp_help of the class.
	 */
	public $tp_help = THEPLUS_HELP;

	/**
	 * Get Widget Name.
	 *
	 * @since 1.0.0
	 */
	public function get_name() {
		return 'tp-scroll-sequence';
	}

	/**
	 * Get Widget Title.
	 *
	 * @since 1.0.0
	 */
	public function get_title() {
		return esc_html__( 'Scroll Sequence', 'theplus' );
	}

	/**
	 * Get Widget Icon.
	 *
	 * @since 1.0.0
	 */
	public function get_icon() {
		return 'theplus-i-scroll-sequence tpae-editor-logo';
	}

	/**
	 * Get Widget categories.
	 *
	 * @since 1.0.0
	 */
	public function get_categories() {
		return array( 'plus-advanced' );
	}

	/**
	 * Get Widget keywords.
	 *
	 * @since 1.0.0
	 */
	public function get_keywords() {
		return array( 'Scroll Animation', 'Tp Scroll Sequence', 'Image Gallery Animation', 'Dynamic Scroll Effects', 'Interactive Image Sequences', 'Image Sequence Animation', 'Sequence Scrolling' );
	}

	/**
	 * Get doc URLs.
	 *
	 * @since 1.0.0
	 * 
	 * @version 5.6.5
	 */
	public function get_custom_help_url() {
		$help_url = $this->tp_help;

		return esc_url( $help_url );
	}

	/**
	 * Check Elementor preview mode.
	 *
	 * @since 1.0.0
	 */
	public function is_reload_preview_required() {
		return true;
	}

	/**
     * It is use for widget add in catch or not.
     *
     * @since 6.1.1
     */
    public function is_dynamic_content(): bool {
        return false;
    }

	/**
	 * Disable Elementor's default inner wrapper for custom HTML control.
	 *
	 * @since 6.3.2
	 */
	public function has_widget_inner_wrapper(): bool {
		return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}

	/**
	 * Register controls.
	 *
	 * @since 1.0.0
	 * @version 5.3.4
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'scroll_sequence_tab_content',
			array(
				'label' => esc_html__( 'Scroll Sequence', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		$this->add_control(
			'applyTo',
			array(
				'label'   => esc_html__( 'Apply To', 'theplus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'default',
				'options' => array(
					'default'        => esc_html__( 'Default', 'theplus' ),
					'body'           => esc_html__( 'Body', 'theplus' ),
					'innerContainer' => esc_html__( 'Inner Column', 'theplus' ),
				),
			)
		);
		$this->add_control(
			'default_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Apply the scroll sequence to the entire container area.', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'applyTo' => 'default',
				),
			)
		);
		$this->add_control(
			'body_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Apply the scroll sequence across the full page. To ensure your content stays visible, set the z-index to 1 from the Style tab.', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'applyTo' => 'body',
				),
			)
		);
		$this->add_control(
			'innerContainer_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Apply the scroll sequence only inside the inner column of the section.', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'applyTo' => 'innerContainer',
				),
			)
		);
		$this->add_control(
			'imageUpldType',
			array(
				'label'   => esc_html__( 'Upload Type', 'theplus' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'gallery',
				'options' => array(
					'gallery' => esc_html__( 'Gallery', 'theplus' ),
					'server'  => esc_html__( 'Remote Server', 'theplus' ),
				),
			)
		);
		$this->add_control(
			'gallery_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Upload images directly from your media library to create the scroll sequence. Best for quick setup and local image control.', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'imageUpldType' => 'gallery',
				),
			)
		);
		$this->add_control(
			'server_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Load images from an external server using URLs. Useful when hosting large image sequences externally to reduce site load.', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'imageGallery',
			array(
				'label'     => esc_html__( 'Image Upload', 'theplus' ),
				'type'      => Controls_Manager::GALLERY,
				'default'   => array(),
				'condition' => array(
					'imageUpldType' => 'gallery',
				),
			)
		);
		$this->add_control(
			'imagePath',
			array(
				'label'       => esc_html__( 'Folder Path', 'theplus' ),
				'type'        => Controls_Manager::URL,
				'dynamic'     => array(
					'active' => true,
				),
				'placeholder' => esc_html__( 'https://www.demo-link.com', 'theplus' ),
				'condition'   => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'imagePath_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Enter the complete folder URL where your images are stored. Make sure each image name ends with increasing numbers (for example: image001.jpg, image002.jpg, image003.jpg) so the sequence loads correctly..', 'theplus' ),
						esc_url( $this->tp_doc . 'create-image-sequence-scroll-animation-from-a-url/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'imageUpldType' => 'server',
				),
			)
		);

		$this->add_control(
			'imagePrefix',
			array(
				'label'       => esc_html__( 'Prefix', 'theplus' ),
				'type'        => Controls_Manager::TEXT,
				'ai'          => false,
				'placeholder' => esc_html__( 'Enter your prefix', 'theplus' ),
				'condition'   => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'prefixtxtNote',
			array(
				'type'      => Controls_Manager::RAW_HTML,
				'raw'       => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Enter the name of image you have used above without digits. e.g. image.', 'theplus' ),
					)
				),
				'condition' => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'imageDigit',
			array(
				'label'     => esc_html__( 'Digit', 'theplus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '1',
				'options'   => array(
					'1' => esc_html__( '1-9', 'theplus' ),
					'2' => esc_html__( '01-99', 'theplus' ),
					'3' => esc_html__( '001-999', 'theplus' ),
					'4' => esc_html__( '0001-9999', 'theplus' ),
				),
				'condition' => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'imageDigitNote',
			array(
				'type'      => Controls_Manager::RAW_HTML,
				'raw'       => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Choose right number of digits based on your total number of images. e.g. If you are having 39 images, choose 01-99.', 'theplus' ),
					)
				),
				'condition'   => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'imageType',
			array(
				'label'     => esc_html__( 'Image Type', 'theplus' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'jpg',
				'options'   => array(
					'jpg'  => esc_html__( 'JPG', 'theplus' ),
					'png'  => esc_html__( 'PNG', 'theplus' ),
					'webp' => esc_html__( 'WebP', 'theplus' ),
				),
				'condition' => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'imageType_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Select the file extension of your images (JPG, PNG, WebP). It must match the actual files on your server.', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'totalImage',
			array(
				'label'     => esc_html__( 'Total Image', 'theplus' ),
				'type'      => Controls_Manager::NUMBER,
				'min'       => 1,
				'max'       => 5000,
				'step'      => 1,
				'default'   => 20,
				'condition' => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->add_control(
			'totalImage_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Enter the total number of images in your sequence so the widget knows how many frames to load.', 'theplus' ),
					)
				),
				'label_block' => true,
				'condition'   => array(
					'imageUpldType' => 'server',
				),
			)
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'extra_Opt_section',
			array(
				'label' => esc_html__( 'Extra Option', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		$this->add_control(
			'preloadImg',
			array(
				'label'   => esc_html__( 'Preload Image (%)', 'theplus' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 100,
				'step'    => 1,
				'default' => 20,
			)
		);
		$this->add_control(
			'preloadImg_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Set how many images should preload before the scroll sequence begins.', 'theplus' ),
						esc_url( $this->tp_doc . 'scroll-sequence-elementor-widget-settings-overview/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'label_block' => true,
			)
		);
		$this->add_control(
			'stickySec',
			array(
				'label'     => esc_html__( 'Sticky Sections', 'theplus' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => '',
				'label_on'  => esc_html__( 'Show', 'theplus' ),
				'label_off' => esc_html__( 'Hide', 'theplus' ),
			)
		);
		$this->add_control(
			'stickySec_label',
			array(
				'type'  => Controls_Manager::RAW_HTML,
				'raw'   => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i> %s <a class="tp-docs-link" href="%s" target="_blank" rel="noopener noreferrer">%s</a></i></p>',
						esc_html__( 'Enable sticky behavior for selected sections during scroll.', 'theplus' ),
						esc_url( $this->tp_doc . 'add-sticky-sections-in-image-scroll-sequence/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget' ),
						esc_html__( 'Learn More', 'theplus' ),
					)
				),
				'label_block' => true,
			)
		);
		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'sectionId',
			array(
				'label'       => esc_html__( 'Section ID', 'theplus' ),
				'type'        => Controls_Manager::TEXT,
				'ai'          => false,
				'title'       => 'Add Section Id to make content sticky',
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Enter the CSS ID of the section you want to control.', 'theplus' ),
					)
				),
			)
		);
		$repeater->add_control(
			'secStart',
			array(
				'label'       => esc_html__( 'Start', 'theplus' ),
				'type'        => Controls_Manager::NUMBER,
				'title'       => 'Enter image number where element becomes visible',
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Set when the sticky effect should start (in viewport or px).', 'theplus' ),
					)
				),
			)
		);
		$repeater->add_control(
			'secEnd',
			array(
				'label'       => esc_html__( 'End', 'theplus' ),
				'type'        => Controls_Manager::NUMBER,
				'title'       => 'Enter image number where element becomes Hidden',
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Set when the sticky effect should stop.', 'theplus' ),
					)
				),
			)
		);
		$repeater->add_control(
			'offsetop',
			array(
				'label'       => esc_html__( 'Top (%)', 'theplus' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( '%' ),
				'range'       => array(
					'%' => array(
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => '%',
					'size' => '5',
				),
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Control how far from the top the section stays while sticking.', 'theplus' ),
					)
				),
			)
		);
		$repeater->add_control(
			'secAnimationstart',
			array(
				'label'       => esc_html__( 'Start Animation', 'theplus' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'none'           => esc_html__( 'None', 'theplus' ),
					'tp-fadein'      => esc_html__( 'Fade In', 'theplus' ),
					'tp-fadeinup'    => esc_html__( 'Fade In Up', 'theplus' ),
					'tp-fadeindown'  => esc_html__( 'Fade In Down', 'theplus' ),
					'tp-fadeinleft'  => esc_html__( 'Fade In Left', 'theplus' ),
					'tp-fadeinright' => esc_html__( 'Fade In Right', 'theplus' ),
					'tp-rotatein'    => esc_html__( 'Rotate In', 'theplus' ),
				),
				'default'     => 'none',
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Choose the animation that should trigger when the section starts.', 'theplus' ),
					)
				),
			)
		);
		$repeater->add_control(
			'secAnimationend',
			array(
				'label'       => esc_html__( 'End Animation', 'theplus' ),
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'none'            => esc_html__( 'None', 'theplus' ),
					'tp-fadeout'      => esc_html__( 'Fade Out', 'theplus' ),
					'tp-fadeoutup'    => esc_html__( 'Fade Out Up', 'theplus' ),
					'tp-fadeoutdown'  => esc_html__( 'Fade Out Down', 'theplus' ),
					'tp-fadeoutleft'  => esc_html__( 'Fade Out Left', 'theplus' ),
					'tp-fadeoutright' => esc_html__( 'Fade Out Right', 'theplus' ),
					'tp-rotateout'    => esc_html__( 'Rotate Out', 'theplus' ),
				),
				'default'     => 'none',
				'description' => wp_kses_post(
					sprintf(
						'<p class="tp-controller-label-text"><i>%s</i></p>',
						esc_html__( 'Choose the animation that should trigger when the section ends.', 'theplus' ),
					)
				),
			)
		);
		$this->add_control(
			'seclist',
			array(
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'condition'   => array(
					'stickySec' => 'yes',
				),
				'default'     => array(
					array(
						'secAnimationstart' => esc_html__( 'none', 'theplus' ),
						'secAnimationend'   => esc_html__( 'none', 'theplus' ),
					),
				),
				'title_field' => '{{{ sectionId }}}',
			)
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'theplus_section_needhelp',
			array(
				'label' => esc_html__( 'Need Help?', 'theplus' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);
		$this->add_control(
			'theplus_help_requests',
			array(
				'label'   => __( 'Need Help', 'theplus' ),
				'type'    => 'tpae_need_help',
				'default' => array(
					array(
						'label' => __( 'Read Docs', 'theplus' ),
						'url'   => 'https://theplusaddons.com/help/scroll-sequence/?utm_source=wpbackend&utm_medium=elementoreditor&utm_campaign=widget',
					),
				),
			)
		);
		$this->end_controls_section();
		/**Scroll Sequence Style Start*/
		$this->start_controls_section(
			'scroll_seq_section_style',
			array(
				'label' => esc_html__( 'Scroll Sequence', 'theplus' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);
		$this->add_control(
			'canVidPosition',
			array(
				'label'        => __( 'Position', 'theplus' ),
				'type'         => Controls_Manager::POPOVER_TOGGLE,
				'label_off'    => __( 'Default', 'theplus' ),
				'label_on'     => __( 'Custom', 'theplus' ),
				'return_value' => 'yes',
				'default'      => 'no',
			)
		);
		$this->start_popover();
		$this->add_responsive_control(
			'posTop',
			array(
				'label'       => esc_html__( 'Top (%)', 'theplus' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( '%' ),
				'range'       => array(
					'%' => array(
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => '%',
					'size' => '',
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'.tp-scroll-seq-inner.elementor-element-{{ID}}-canvas' => 'top: {{SIZE}}% !important;',
				),
				'condition'   => array(
					'canVidPosition' => 'yes',
				),
			)
		);
		$this->add_responsive_control(
			'posLeft',
			array(
				'label'       => esc_html__( 'Left (%)', 'theplus' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( '%' ),
				'range'       => array(
					'%' => array(
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					),
				),
				'default'     => array(
					'unit' => '%',
					'size' => '',
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'.tp-scroll-seq-inner.elementor-element-{{ID}}-canvas' => 'left: {{SIZE}}% !important;',
				),
				'condition'   => array(
					'canVidPosition' => 'yes',
				),
			)
		);
		$this->end_popover();
		$this->add_control(
			'canVidZIndex',
			array(
				'label'     => esc_html__( 'Z-Index', 'theplus' ),
				'type'      => Controls_Manager::NUMBER,
				'min'       => -10,
				'max'       => 10,
				'step'      => 1,
				'default'   => 0,
				'selectors' => array(
					'.tp-scroll-seq-inner.elementor-element-{{ID}}-canvas' => 'z-index: {{VALUE}} !important;',
				),
			)
		);
		$this->add_responsive_control(
			'canVidWidth',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Width', 'theplus' ),
				'size_units'  => array( 'px', '%' ),
				'range'       => array(
					'%'  => array(
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					),
					'px' => array(
						'min'  => 0,
						'max'  => 1000,
						'step' => 2,
					),

				),
				'default'     => array(
					'unit' => 'px',
					'size' => '',
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'.tp-scroll-seq-inner.elementor-element-{{ID}}-canvas' => 'width: {{SIZE}}{{UNIT}} !important;',

				),
			)
		);
		$this->add_responsive_control(
			'canVidHeight',
			array(
				'type'        => Controls_Manager::SLIDER,
				'label'       => esc_html__( 'Height', 'theplus' ),
				'size_units'  => array( 'px', '%' ),
				'range'       => array(
					'%'  => array(
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					),
					'px' => array(
						'min'  => 0,
						'max'  => 1000,
						'step' => 2,
					),

				),
				'default'     => array(
					'unit' => 'px',
					'size' => '',
				),
				'render_type' => 'ui',
				'selectors'   => array(
					'.tp-scroll-seq-inner.elementor-element-{{ID}}-canvas' => 'height: {{SIZE}}{{UNIT}} !important;',
				),
			)
		);
		$this->add_responsive_control(
			'canStartOffset',
			array(
				'label'       => esc_html__( 'Start Offset (px)', 'theplus' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => -1000,
						'max'  => 1000,
						'step' => 10,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => '',
				),
				'render_type' => 'ui',
			)
		);
		$this->add_responsive_control(
			'canEndOffset',
			array(
				'label'       => esc_html__( 'End Offset (px)', 'theplus' ),
				'type'        => Controls_Manager::SLIDER,
				'size_units'  => array( 'px' ),
				'range'       => array(
					'px' => array(
						'min'  => -1000,
						'max'  => 1000,
						'step' => 10,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => '',
				),
				'render_type' => 'ui',
			)
		);
		$this->end_controls_section();
		/**Scroll Sequence Style End*/

	}

	/**
	 * Render Progress Bar
	 *
	 * Written in PHP and HTML.
	 *
	 * @since 1.0.0
	 * @version 5.3.4
	 */
	protected function render() {
		$settings   = $this->get_settings_for_display();
		$uid_scroll = uniqid( 'tp-scs' );

		$image_upld_type = ! empty( $settings['imageUpldType'] ) ? $settings['imageUpldType'] : 'gallery';

		$image_gallery = ! empty( $settings['imageGallery'] ) ? $settings['imageGallery'] : array();
		$image_path    = ! empty( $settings['imagePath']['url'] ) ? $settings['imagePath']['url'] : '';
		$image_prefix  = ! empty( $settings['imagePrefix'] ) ? $settings['imagePrefix'] : '';
		$image_digit   = ! empty( $settings['imageDigit'] ) ? (int) $settings['imageDigit'] : '1';
		$image_type    = ! empty( $settings['imageType'] ) ? $settings['imageType'] : 'jpg';
		$total_image   = ! empty( $settings['totalImage'] ) ? $settings['totalImage'] : 20;
		$apply_to      = ! empty( $settings['applyTo'] ) ? $settings['applyTo'] : 'default';
		$preload_img   = ! empty( $settings['preloadImg'] ) ? $settings['preloadImg'] : 20;

		$can_start_offset = ! empty( $settings['canStartOffset']['size'] ) ? $settings['canStartOffset']['size'] : 0;
		$can_end_offset   = isset( $settings['canEndOffset']['size'] ) ? $settings['canEndOffset']['size'] : 0;

		$sticky_sec = ! empty( $settings['stickySec'] ) ? $settings['stickySec'] : 0;
		$seclist    = ! empty( $settings['seclist'] ) ? $settings['seclist'] : '';

		$data_attr = array();
		$img_glr   = array();

		if ( 'gallery' === $image_upld_type && ! empty( $image_gallery ) ) {
			$img_glr = array_column( $image_gallery, 'url' );
		} elseif ( 'server' === $image_upld_type ) {

			if ( is_numeric( $image_digit ) && ! empty( $image_path ) && ! empty( $total_image ) && in_array( $image_type, array( 'jpg', 'png', 'webp' ) ) ) {
				for ( $i = 1; $i <= $total_image; $i++ ) {
					$immm    = str_pad( $i, $image_digit, '0', STR_PAD_LEFT );
					$img_url = $image_path . '/' . $image_prefix . $immm . '.' . $image_type;

					$u_r_lexists = wp_remote_get( $img_url );
					if ( ! empty( $u_r_lexists ) ) {
						$img_glr[] = $img_url;
					}
				}
			}
		}

		$massage = '';
		$icon    = '';
		if ( ! empty( $img_glr ) ) {
			$data_attr = array(
				'widget_id'   => $this->get_id(),
				'imgGellary'  => $img_glr,
				'applyto'     => esc_attr( $apply_to ),
				'imgUpdType'  => esc_attr( $image_upld_type ),
				'preloadImg'  => esc_attr( $preload_img ),
				'startOffset' => esc_attr( $can_start_offset ),
				'endOffset'   => esc_attr( $can_end_offset ),
				'stickySec'   => esc_attr( $sticky_sec ),
				'seclist'     => $seclist,
			);
		} else {
			$error_title   = 'No Image Selected!';
			$error_massage = 'Please Select Image To Get The Desired Result';

			$massage = theplus_get_widgetError( $error_title, $error_massage );
		}

		$data_attr = 'data-attr="' . htmlspecialchars( wp_json_encode( $data_attr, true ), ENT_QUOTES, 'UTF-8' ) . '"';

		$output = '';

		$output     .= '<div class="tp-scroll-sequence tp-widget-' . esc_attr( $uid_scroll ) . '" ' . $data_attr . '>';
			$output .= $massage;
		$output     .= '</div>';

		echo $output;
	}
}