<?php
/**
 * PayTabs - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_PayTabs' ) ) {

    class TC_Gateway_PayTabs extends TC_Gateway_API {

        var $plugin_name = 'paytabs';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl = false;
        var $ipn_url;
        var $site_url;
        var $permanently_active = false;
        var $skip_payment_screen = false;
        var $currency = '';
        var $profile_id = '';
        var $server_key = '';
        var $manual_capture = '';
        var $gateway_url = 'https://secure-global.paytabs.com/payment/request';

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        /**
         * Initialize Variables
         */
        function init() {
            global $tc;
            $this->admin_name = __( 'PayTabs', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'PayTabs', 'tickera-event-ticketing-system' );
            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/paytabs.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-paytabs.png', $this->plugin_name );
            $this->profile_id = $this->get_option( 'profile_id' );
            $this->server_key = $this->get_option( 'server_key' );
            $this->manual_capture = $this->get_option( 'manual_capture', 'no' );
            $this->currency = $tc->get_store_currency();
            $this->site_url = get_site_url();
        }

        /**
         * Generate Payment Form
         * @param $cart
         */
        function payment_form( $cart ) {}

        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();
            $payment_info = $this->save_payment_info();

            $order_id = $tc->generate_order_id();
            $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, false );

            $fields = [
                'profile_id' => $this->profile_id,
                'tran_type' => 'sale',
                'tran_class' => 'ecom',
                'cart_description' => $this->cart_items(),
                'cart_id' => $order_id,
                'cart_currency' => $this->currency,
                'cart_amount' => $this->total(),
                'callback' => $this->ipn_url,
                'return' => $tc->get_confirmation_slug( true, $order_id ),
                'hide_shipping' => true
            ];

            if ( 'yes' == $this->manual_capture ) {
                $fields[ 'tran_type' ] = 'auth';
            }

            $response = self::request( $fields );
            $response = tickera_sanitize_array( $response, false, true );

            if ( isset( $response[ 'error' ] ) ) {
                $tc->session->set( 'tc_gateway_error', sprintf(
                /* translators: %s: A link to Tickera cart page */
                    __( 'Sorry, something went wrong. <a href="%s">Please try again</a>.', 'tickera-event-ticketing-system' ),
                    esc_url( $tc->get_cart_slug( true ) )
                ), true );
                tickera_redirect( $tc->get_payment_slug( true ), true );

            } else {

                if ( isset( $response[ 'redirect_url' ] ) ) {
                    tickera_redirect( $response[ 'redirect_url' ], true );

                } elseif ( isset( $response[ 'code' ] ) && isset( $response[ 'message' ] ) ) {
                    $tc->session->set( 'tc_gateway_error', $response[ 'message' ] );
                    tickera_redirect( $tc->get_payment_slug( true ), true );

                } else {

                    $tc->session->set( 'tc_gateway_error', sprintf(
                    /* translators: %s: A link to Tickera cart page */
                        __( 'Sorry, something went wrong. <a href="%s">Please try again</a>.', 'tickera-event-ticketing-system' ),
                        esc_url( $tc->get_cart_slug( true ) )
                    ), true );

                    tickera_redirect( $tc->get_payment_slug( true ), true );
                }
            }
        }

        /**
         * Verify Paytabs payment.
         * Set Order statuses base on response result
         *
         * @param $order
         * @param string $payment_info
         * @param string $cart_info
         */
        function order_confirmation( $order, $payment_info = '', $cart_info = '' ) {

            global $tc;

            // Request body include a signature post Form URL encoded field
            $signature_fields = filter_input_array( INPUT_POST );

            // Missing signature fields.
            if ( ! $signature_fields ) {
                $order = tickera_get_order_id_by_name( $order );
                $tc->update_order_status( $order->ID, 'order_cancelled' );
                \Tickera\TC_Order::add_order_note( $order->ID, __( 'Invalid redirect.', 'tickera-event-ticketing-system' ) );
                return;
            }

            $order = tickera_get_order_id_by_name( $order );

            // 'signature' (hexadecimal encoding for hmac of sorted post form fields)
            $requestSignature = $signature_fields[ "signature" ];
            unset( $signature_fields[ "signature" ] );

            /*
             * Step 1: Ignore empty values fields.
             * Step 2: Sort form fields
             * Step 3: Generate URL-encoded query string of Post fields except signature field.
             */
            $signature_fields = array_filter( $signature_fields );
            ksort( $signature_fields );
            $query = http_build_query( $signature_fields );

            $signature = hash_hmac( 'sha256', $query, $this->server_key );
            if ( hash_equals( $signature, $requestSignature ) === TRUE ) {

                // Store transaction reference for later use.
                update_post_meta( $order->ID, 'transaction_reference', sanitize_key( $signature_fields[ 'tranRef' ] ) );

                // Update order status and add some notes.
                self::process_order_status( [
                    'order_id' => $order->ID,
                    'transaction_reference' => $signature_fields[ 'tranRef' ],
                    'response_status' => $signature_fields[ 'respStatus' ],
                    'response_message' => $signature_fields[ 'respMessage' ]
                ]);

            } else {
                $tc->update_order_status( $order->ID, 'order_cancelled' );
                \Tickera\TC_Order::add_order_note( $order->ID, __( 'Invalid redirect.', 'tickera-event-ticketing-system' ) );
            }
        }

        /**
         * Update order status base on the response status.
         * Add order notes for admin reference.
         *
         * @param $data
         */
        function process_order_status( $data ) {

            global $tc;
            $order_id = (int) $data[ 'order_id' ];

            switch ( $data[ 'response_status' ] ) {

                // Authorized
                case 'A':
                    $tc->update_order_payment_status( $order_id, true );
                    break;

                case 'V': // Voided
                case 'D': // Declined
                case 'C': // Cancelled
                    $tc->update_order_status( $order_id, 'order_cancelled' );
                    break;
            }

            $notes = [
                __( 'Transaction Reference: ', 'tickera-event-ticketing-system' ) . $data[ 'transaction_reference' ],
                __( 'Response Message: ', 'tickera-event-ticketing-system' ) . $data[ 'response_message' ],
            ];
            \Tickera\TC_Order::add_order_note( $order_id, implode( '<br>', $notes ));
        }

        /**
         * Send Request
         * @param $data
         * @return bool|string
         */
        function request( $data ) {

            $header = [
                'authorization: ' . $this->server_key,
                'content-type: application/json'
            ];

            $ch = curl_init();
            curl_setopt( $ch, CURLOPT_URL, $this->gateway_url );
            curl_setopt( $ch, CURLOPT_HTTPHEADER, $header );
            curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true );
            curl_setopt( $ch, CURLOPT_POST, true );
            curl_setopt( $ch, CURLOPT_POSTFIELDS,  json_encode( $data ) );
            $response = curl_exec( $ch );

            if ( curl_errno( $ch ) ) {
                $response = [];
                $response[ 'error' ] = curl_error( $ch );

            } else {
                $response = json_decode( $response, true );
            }

            curl_close( $ch );
            return $response;
        }


        /**
         * Generate view for Admin Setting
         *
         * @param $settings
         * @param $visible
         */
        function gateway_admin_settings( $settings, $visible ) { ?>

            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Paytabs Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"> <?php esc_html_e( 'Sell your tickets via PayTabs payment gateway.', 'tickera-event-ticketing-system' ); ?> </span>
                </h3>
                <div class="inside">

                    <?php
                    $fields = array(
                        'profile_id' => array(
                            'title' => __( 'Profile ID', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'server_key' => array(
                            'title' => __( 'Server key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'manual_capture' => array(
                            'title' => __( 'Capture payments manually', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array( 'yes' => __( 'Yes', 'tickera-event-ticketing-system' ), 'no' => __( 'No', 'tickera-event-ticketing-system' ) ),
                            'default' => 'no',
                            'description' => __( 'Manually capture payments in PayTabs account dashboard.', 'tickera-event-ticketing-system' )
                        )
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        /**
         * Callback
         */
        function ipn() {

            $payload = file_get_contents( 'php://input' );
            $payload_arr = json_decode( $payload, true );
            $cart_id = isset( $payload_arr[ 'cart_id' ] ) ? sanitize_key( $payload_arr[ 'cart_id' ] ) : '';
            $tran_ref = isset( $payload_arr[ 'tran_ref' ] ) ? sanitize_key( $payload_arr[ 'tran_ref' ] ) : '';

            // Required meta for validation
            if ( ! $cart_id || ! $tran_ref ) {
                return;
            }

            // Check for an existing order
            $order = tickera_get_order_id_by_name( $cart_id );
            if ( ! $order ) {
                return;
            }

            // Validate order transaction reference.
            $order_tran_ref = get_post_meta( $order->ID, 'transaction_reference', true );
            if ( ! $order_tran_ref || ( $tran_ref != $order_tran_ref ) ) {
                return;
            }

            // Required payment result values.
            $payment_result = isset( $payload_arr[ 'payment_result' ] ) ? $payload_arr[ 'payment_result' ] : '';
            if ( ! $payment_result ) {
                return;
            }

            self::process_order_status([
                'order_id' => $order->ID,
                'transaction_reference' => $tran_ref,
                'response_status' => $payload_arr[ 'payment_result' ][ 'response_status' ],
                'response_message' => $payload_arr[ 'payment_result' ][ 'response_message' ]
            ]);
        }
    }

    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_PayTabs', 'paytabs', __( 'PayTabs', 'tickera-event-ticketing-system' ) );
}