<?php
/**
 * PIN Payment Gateway (www.pin.net.au)
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_PIN' ) ) {

    class TC_Gateway_PIN extends TC_Gateway_API {

        var $plugin_name = 'pin';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl;
        var $ipn_url;
        var $publishable_key, $private_key, $currency;
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = false;
        var $public_key;

        /**
         * Support for older payment gateway API
         */
        function on_creation() {
            $this->init();
        }

        function init() {
            global $tc;

        $settings = get_option( 'tickera_settings' );

            $this->admin_name = __( 'PIN', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'PIN', 'tickera-event-ticketing-system' );
            $this->public_key = $this->get_option( 'public_key' );
            $this->private_key = $this->get_option( 'private_key' );
            $this->force_ssl = $this->get_option( 'is_ssl', '0' ) == '1' ? true : false;
            $this->currency = $this->get_option( 'currency', 'AUD' );

            $this->method_img_url = sanitize_text_field( apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/pin.png', $this->plugin_name ) );
            $this->admin_img_url = sanitize_text_field( apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-pin.png', $this->plugin_name ) );

            add_action( 'wp_enqueue_scripts', array( &$this, 'enqueue_scripts' ) );

            $currencies = array(
                "AUD" => __( 'AUD - Australian Dollar', 'tickera-event-ticketing-system' ),
                "USD" => __( 'USD - United States Dollar', 'tickera-event-ticketing-system' ),
                "NZD" => __( 'NZD - New Zealand Dollar', 'tickera-event-ticketing-system' ),
                "SGD" => __( 'SGD - Singaporean Dollar', 'tickera-event-ticketing-system' ),
                "EUR" => __( 'EUR - Euro', 'tickera-event-ticketing-system' ),
                "GBP" => __( 'GBP - British Pound', 'tickera-event-ticketing-system' ),
                "HKD" => __( 'HKD - Hong Kong Dollar', 'tickera-event-ticketing-system' ),
                "JPY" => __( 'JPY - Japanese Yen', 'tickera-event-ticketing-system' ),
            );

            $this->currencies = $currencies;
        }

        function enqueue_scripts() {
            global $tc, $wp;

            if ( $this->is_active() && $this->is_payment_page() ) {
                if ( $this->force_ssl ) {
                    wp_enqueue_script( 'js-pin', 'https://api.pin.net.au/pin.js', array( 'jquery' ) );
                } else {
                    wp_enqueue_script( 'js-pin', 'https://test-api.pin.net.au/pin.js', array( 'jquery' ) );
                }

                wp_enqueue_script( 'pin-handler', $tc->plugin_url . '/includes/gateways/pin/pin-handler.js', array( 'js-pin', 'jquery' ) );
                wp_localize_script( 'pin-handler', 'pin_vars', array(
                        'publishable_api_key' => $this->public_key,
                    )
                );
            }
        }

        function payment_form( $cart ) {
            global $tc;
            $content = '';
            $content .= '<div id="pin_checkout_errors"><ul></ul></div>';
            $content .= '<table class="tc_cart_billing"><thead><tr><th colspan="2">' . esc_html__( 'Enter Your Credit Card Information:', 'tickera-event-ticketing-system' ) . '</th></tr></thead><tbody><tr><td>' . esc_html__( 'Cardholder Name:', 'tickera-event-ticketing-system' ) . '</td><td><input id="cc-name" type="text" value="' . esc_attr( $this->buyer_info( 'full_name' ) ) . '" /> </td></tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'Card Number', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input type="text" autocomplete="off" id="cc-number"/>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'Expiration:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<select id="cc-expiry-month">';
            $content .= tickera_months_dropdown();
            $content .= '</select>';
            $content .= '<span> / </span>';
            $content .= '<select id="cc-expiry-year">';
            $content .= tickera_years_dropdown( '', true );
            $content .= '</select>';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '<tr>';
            $content .= '<td>';
            $content .= esc_html__( 'CVC:', 'tickera-event-ticketing-system' );
            $content .= '</td>';
            $content .= '<td>';
            $content .= '<input type="text" size="4" autocomplete="off" id="cc-cvc" />';
            $content .= '</td>';
            $content .= '</tr>';
            $content .= '</table>';
            $content .= '<span id="pin_processing" style="display: none;float: right;"><img src="' . esc_url( $tc->plugin_url ) . 'images/loading.gif" /> ' . esc_html__( 'Processing...', 'tickera-event-ticketing-system' ) . '</span>';

            return $content;
        }

        /**
         * Generate Order Confirmation Page upon success checkout
         * @param $order
         * @param string $cart_info
         * @return string
         */
        function order_confirmation_message( $order, $cart_info = '' ) {

            global $tc;
            $order = tickera_get_order_id_by_name( $order );
            $order = new \Tickera\TC_Order( $order->ID );
            $content = '';

            switch ( $order->details->post_status ) {

                case 'order_received':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: PIN Payment Gateway order total amount */ __( 'Your payment via PIN for this order totaling <strong>%s</strong> is not yet complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    $content .= '<p>' . wp_kses_post( __( 'Current order status: <strong>Pending Payment</strong>', 'tickera-event-ticketing-system' ) ) . '</p>';
                    break;

                case 'order_fraud':
                    $content .= '<p>' . esc_html__( 'Your payment is under review. We will back to you soon.', 'tickera-event-ticketing-system' ) . '</p>';
                    break;

                case 'order_paid':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: PIN Payment Gateway order total amount */ __( 'Your payment via PIN for this order totaling <strong>%s</strong> is complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_cancelled':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: PIN Payment Gateway order total amount */ __( 'Your payment via PIN for this order totaling <strong>%s</strong> is cancelled.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_refunded':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: %s: PIN Payment Gateway order total amount */ __( 'Your payment via PIN for this order totaling <strong>%s</strong> is refunded.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ). '</p>';
                    break;

            }

            $content = wp_kses_post( apply_filters( 'tc_order_confirmation_message_content', $content, $order, $this->plugin_name ) );
            $tc->remove_order_session_data();
            $tc->session->drop( 'card_token' );

            $tc->maybe_skip_confirmation_screen( $this, $order );
            return $content;
        }

        function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();

            if ( isset( $_POST[ 'card_token' ] ) ) {
                $tc->session->set( 'card_token', sanitize_text_field( $_POST[ 'card_token' ] ) );
            }

            $session = $tc->session->get();

            if ( isset( $session[ 'card_token' ] ) ) {
                $tc->session->set( 'tc_gateway_error', __( 'The PIN Token was not generated correctly.', 'tickera-event-ticketing-system' ) );
                tickera_redirect( $tc->get_payment_slug( true ), true );
            }

            if ( $this->force_ssl ) {
                define( 'TC_PIN_API_CHARGE_URL', 'https://api.pin.net.au/1/charges' );

            } else {
                define( 'TC_PIN_API_CHARGE_URL', 'https://test-api.pin.net.au/1/charges' );
            }

            define( 'TC_PIN_API_KEY', $this->private_key );
            $token = sanitize_text_field( $session[ 'card_token' ] );

            if ( $token ) {

                $order_id = $tc->generate_order_id();

                try {

                    $args = array(
                        'method' => 'POST',
                        'httpversion' => '1.1',
                        'timeout' => apply_filters( 'tc_http_request_timeout', 30 ),
                        'blocking' => true,
                        'compress' => true,
                        'headers' => array( 'Authorization' => 'Basic ' . base64_encode( TC_PIN_API_KEY . ':' . '' ) ),
                        'body' => array(
                            'amount' => (int) ( $this->total() * 100 ),
                            'currency' => strtolower( $this->currency ),
                            'description' => $this->cart_items(),
                            'email' => $this->buyer_info( 'email' ),
                            'ip_address' => sanitize_text_field( $session[ 'ip_address' ] ),
                            'card_token' => sanitize_text_field( $session[ 'card_token' ] )
                        ),
                        'cookies' => array()
                    );

                    $charge = wp_remote_post( TC_PIN_API_CHARGE_URL, $args );
                    $charge = $charge[ 'response' ];

                    if ( $charge[ 'success' ] == true ) {
                        $payment_info = array();
                        $payment_info[ 'method' ] = sprintf(
                        /* translators: 1: PIN card scheme 2: PIN card display number */
                            __( '%1$s Card %2$s', 'tickera-event-ticketing-system' ),
                            ucfirst( $charge[ 'card' ][ 'scheme' ] ),
                            $charge[ 'card' ][ 'display_number' ]
                        );
                        $payment_info[ 'transaction_id' ] = $charge[ 'token' ];
                        $payment_info = $this->save_payment_info();

                        $paid = true;
                        $order = $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
                        tickera_redirect( $tc->get_confirmation_slug( true, $order_id ), true );

                    } else {

                        $tc->session->drop( 'card_token' );
                        $tc->session->set( 'tc_gateway_error', __( 'There was an error processing your card.', 'tickera-event-ticketing-system' ) );
                        tickera_redirect( $tc->get_payment_slug( true ), true );
                    }
                } catch ( Exception $e ) {

                    $tc->session->drop( 'card_token' );
                    $tc->session->set( 'tc_gateway_error', sprintf( /* translators: %s: PIN Payment Gateway card processing error */ __( 'There was an error processing your card: "%s".', 'tickera-event-ticketing-system' ), $e->getMessage() ) );
                    tickera_redirect( $tc->get_payment_slug( true ), true );
                }
            }
        }

        function gateway_admin_settings( $settings, $visible ) {
            global $tc; ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: PIN Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( "Accept all major credit cards directly on your site. Your sales revenue is deposited to any Australian bank account, no merchant account required.", 'tickera-event-ticketing-system' ) ?></span>
                </h3>
                <div class="inside">

                    <?php
                    $fields = array(
                        'is_ssl' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '0' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                '1' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => '0',
                        ),
                        'private_key' => array(
                            'title' => __( 'Secret API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'description' => __( 'You must login to PIN to <a target="_blank" href="https://dashboard.pin.net.au/account">get your API credentials</a>. You can use test keys while sandbox testing but change them to live ones when you switch the mode to Live.', 'tickera-event-ticketing-system' ),
                        ),
                        'public_key' => array(
                            'title' => __( 'Publishable API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),
                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'AUD',
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }
    }

    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_PIN', 'pin', __( 'PIN', 'tickera-event-ticketing-system' ) );
}