<?php
/**
 * Stripe - Payment Gateway
 */

namespace Tickera\Gateway;
use Tickera\TC_Gateway_API;

use TCStripe\Event;
use TCStripe\StripeClient;
use TCStripe\PaymentIntent;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( '\Tickera\Gateway\TC_Gateway_Stripe' ) ) {

    class TC_Gateway_Stripe extends TC_Gateway_API {

        var $plugin_name = 'stripe';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl;
        var $ipn_url;
        var $publishable_key, $private_key, $currency;
        var $currencies = array();
        var $permanently_active = false;
        var $skip_payment_screen = false;
        var $send_receipt = 0;
        var $default_api_version = '2023-08-16'; // Prev: 2020-08-27
        var $api_version = '';
        var $enable_webhook;
        var $zero_decimal_currencies;

        /**
         * Support for older payment gateway API
         */
        public function on_creation() {
            $this->init();
            add_action( 'tc_save_tc_gateway_settings', array( $this, 'save_settings' ) );
        }

        /**
         * Register/Remove Stripe Webhook
         *
         * @return void
         * @since 3.5.5.6
         */
        function save_settings() {

            try {

                $webhook_exists = false;
                $enabled_webhook = $this->get_option( 'enable_webhook', 0 );

                // Retrieve all stripe webhooks and check if Tickera Stripe Exists
                $stripe_webhook = new StripeClient( $this->private_key );
                $stripe_webhook_obj = $stripe_webhook->webhookEndpoints->all();
                $stripe_webhook_obj = tickera_sanitize_array( $stripe_webhook_obj, false, true );

                foreach ( $stripe_webhook_obj[ 'data' ] as $key => $value ) {
                    if ( $value[ 'url' ] == $this->ipn_url ) {

                        // Remove webhook
                        if ( ! $enabled_webhook ) {
                            $stripe_webhook->webhookEndpoints->delete( $value[ 'id' ] );
                        }

                        $webhook_exists = true;
                        break;
                    }
                }

                // Create webhook
                if ( $enabled_webhook && ! $webhook_exists ) {
                    $stripe_webhook->webhookEndpoints->create([
                        'url' => $this->ipn_url,
                        'description' => 'Tickera: Charge Captured Event',
                        'enabled_events' => [ 'charge.captured', 'charge.refunded', 'payment_intent.succeeded', 'payment_intent.payment_failed' ]
                    ]);
                }

            } Catch ( \Exception $e ) {
                // Nothing to do here for now
            }
        }

        public function init_stripe() {
            global $tc;

            if ( ! class_exists( 'TCStripe\Stripe' ) && ! class_exists( 'Stripe' ) && ! class_exists( 'TCStripe\\Stripe' ) ) {//Load Stripe classes only if it doesn't exist already
                require_once $tc->plugin_dir . '/includes/gateways/stripe/init.php';
            }

            try {
                \TCStripe\Stripe::setApiKey( $this->private_key );
                \TCStripe\Stripe::setApiVersion( $this->api_version );

            } Catch( \Exception $e ) {
                $tc->session->set( 'tc_gateway_error', sanitize_text_field( $e->getMessage() ) . ' ' . sprintf( /* translators: 1: A link to Tickera cart page */ __( '<a href="%s">Please try again</a>.', 'tickera-event-ticketing-system' ), esc_url( $tc->get_cart_slug( true ) ) ) );
            }
        }

        public function init() {

            global $tc;
            get_option( 'tickera_settings' );

            $this->admin_name = __( 'Stripe', 'tickera-event-ticketing-system' );
            $this->public_name = __( 'Stripe', 'tickera-event-ticketing-system' );

            $this->method_img_url = apply_filters( 'tc_gateway_method_img_url', $tc->plugin_url . 'images/gateways/stripe.png', $this->plugin_name );
            $this->admin_img_url = apply_filters( 'tc_gateway_admin_img_url', $tc->plugin_url . 'images/gateways/small-stripe.png', $this->plugin_name );

            $this->publishable_key = $this->get_option( 'publishable_key' );
            $this->private_key = $this->get_option( 'private_key' );

            $api_version = $this->get_option( 'api_version' );
            $this->api_version = $api_version ? $api_version : $this->default_api_version;

            $this->force_ssl = $this->get_option( 'is_ssl', '0' ) === '1';
            $this->currency = $this->get_option( 'currency', 'USD' );
            $this->zero_decimal_currencies = array( 'MGA', 'BIF', 'CLP', 'PYG', 'DJF', 'RWF', 'GNF', 'UGX', 'JPY', 'VND', 'VUV', 'XAF', 'KMF', 'KRW', 'XOF', 'XPF' );

            $this->send_receipt = $this->get_option( 'send_receipt', '0' );
            $this->enable_webhook = $this->get_option( 'enable_webhook', 0 );

            $currencies = array(
                'AED' => __( 'AED - United Arab Emirates Dirham', 'tickera-event-ticketing-system' ),
                'AFN' => __( 'AFN - Afghan Afghani', 'tickera-event-ticketing-system' ),
                'ALL' => __( 'ALL - Albanian Lek', 'tickera-event-ticketing-system' ),
                'AMD' => __( 'AMD - Armenian Dram', 'tickera-event-ticketing-system' ),
                'ANG' => __( 'ANG - Netherlands Antillean Gulden', 'tickera-event-ticketing-system' ),
                'AOA' => __( 'AOA - Angolan Kwanza', 'tickera-event-ticketing-system' ),
                'ARS' => __( 'ARS - Argentine Peso', 'tickera-event-ticketing-system' ),
                'AUD' => __( 'AUD - Australian Dollar', 'tickera-event-ticketing-system' ),
                'AWG' => __( 'AWG - Aruban Florin', 'tickera-event-ticketing-system' ),
                'AZN' => __( 'AZN - Azerbaijani Manat', 'tickera-event-ticketing-system' ),
                'BAM' => __( 'BAM - Bosnia & Herzegovina Convertible Mark', 'tickera-event-ticketing-system' ),
                'BBD' => __( 'BBD - Barbadian Dollar', 'tickera-event-ticketing-system' ),
                'BDT' => __( 'BDT - Bangladeshi Taka', 'tickera-event-ticketing-system' ),
                'BGN' => __( 'BGN - Bulgarian Lev', 'tickera-event-ticketing-system' ),
                'BIF' => __( 'BIF - Burundian Franc', 'tickera-event-ticketing-system' ),
                'BMD' => __( 'BMD - Bermudian Dollar', 'tickera-event-ticketing-system' ),
                'BND' => __( 'BND - Brunei Dollar', 'tickera-event-ticketing-system' ),
                'BOB' => __( 'BOB - Bolivian Boliviano', 'tickera-event-ticketing-system' ),
                'BRL' => __( 'BRL - Brazilian Real', 'tickera-event-ticketing-system' ),
                'BSD' => __( 'BSD - Bahamian Dollar', 'tickera-event-ticketing-system' ),
                'BWP' => __( 'BWP - Botswana Pula', 'tickera-event-ticketing-system' ),
                'BZD' => __( 'BZD - Belize Dollar', 'tickera-event-ticketing-system' ),
                'CAD' => __( 'CAD - Canadian Dollar', 'tickera-event-ticketing-system' ),
                'CDF' => __( 'CDF - Congolese Franc', 'tickera-event-ticketing-system' ),
                'CHF' => __( 'CHF - Swiss Franc', 'tickera-event-ticketing-system' ),
                'CLP' => __( 'CLP - Chilean Peso', 'tickera-event-ticketing-system' ),
                'CNY' => __( 'CNY - Chinese Renminbi Yuan', 'tickera-event-ticketing-system' ),
                'COP' => __( 'COP - Colombian Peso', 'tickera-event-ticketing-system' ),
                'CRC' => __( 'CRC - Costa Rican Colon', 'tickera-event-ticketing-system' ),
                'CVE' => __( 'CVE - Cape Verdean Escudo', 'tickera-event-ticketing-system' ),
                'CZK' => __( 'CZK - Czech Koruna', 'tickera-event-ticketing-system' ),
                'DJF' => __( 'DJF - Djiboutian Franc', 'tickera-event-ticketing-system' ),
                'DKK' => __( 'DKK - Danish Krone', 'tickera-event-ticketing-system' ),
                'DOP' => __( 'DOP - Dominican Peso', 'tickera-event-ticketing-system' ),
                'DZD' => __( 'DZD - Algerian Dinar', 'tickera-event-ticketing-system' ),
                'EEK' => __( 'EEK - Estonian Kroon', 'tickera-event-ticketing-system' ),
                'EGP' => __( 'EGP - Egyptian Pound', 'tickera-event-ticketing-system' ),
                'ETB' => __( 'ETB - Ethiopian Birr', 'tickera-event-ticketing-system' ),
                'EUR' => __( 'EUR - Euro', 'tickera-event-ticketing-system' ),
                'FJD' => __( 'FJD - Fijian Dollar', 'tickera-event-ticketing-system' ),
                'FKP' => __( 'FKP - Falkland Islands Pound', 'tickera-event-ticketing-system' ),
                'GBP' => __( 'GBP - British Pound', 'tickera-event-ticketing-system' ),
                'GEL' => __( 'GEL - Georgian Lari', 'tickera-event-ticketing-system' ),
                'GIP' => __( 'GIP - Gibraltar Pound', 'tickera-event-ticketing-system' ),
                'GMD' => __( 'GMD - Gambian Dalasi', 'tickera-event-ticketing-system' ),
                'GNF' => __( 'GNF - Guinean Franc', 'tickera-event-ticketing-system' ),
                'GTQ' => __( 'GTQ - Guatemalan Quetzal', 'tickera-event-ticketing-system' ),
                'GYD' => __( 'GYD - Guyanese Dollar', 'tickera-event-ticketing-system' ),
                'HKD' => __( 'HKD - Hong Kong Dollar', 'tickera-event-ticketing-system' ),
                'HNL' => __( 'HNL - Honduran Lempira', 'tickera-event-ticketing-system' ),
                'HRK' => __( 'HRK - Croatian Kuna', 'tickera-event-ticketing-system' ),
                'HTG' => __( 'HTG - Haitian Gourde', 'tickera-event-ticketing-system' ),
                'HUF' => __( 'HUF - Hungarian Forint', 'tickera-event-ticketing-system' ),
                'IDR' => __( 'IDR - Indonesian Rupiah', 'tickera-event-ticketing-system' ),
                'ILS' => __( 'ILS - Israeli New Sheqel', 'tickera-event-ticketing-system' ),
                'INR' => __( 'INR - Indian Rupee', 'tickera-event-ticketing-system' ),
                'ISK' => __( 'ISK - Icelandic Krona', 'tickera-event-ticketing-system' ),
                'JMD' => __( 'JMD - Jamaican Dollar', 'tickera-event-ticketing-system' ),
                'JPY' => __( 'JPY - Japanese Yen', 'tickera-event-ticketing-system' ),
                'KES' => __( 'KES - Kenyan Shilling', 'tickera-event-ticketing-system' ),
                'KGS' => __( 'KGS - Kyrgyzstani Som', 'tickera-event-ticketing-system' ),
                'KHR' => __( 'KHR - Cambodian Riel', 'tickera-event-ticketing-system' ),
                'KMF' => __( 'KMF - Comorian Franc', 'tickera-event-ticketing-system' ),
                'KRW' => __( 'KRW - South Korean Won', 'tickera-event-ticketing-system' ),
                'KYD' => __( 'KYD - Cayman Islands Dollar', 'tickera-event-ticketing-system' ),
                'KZT' => __( 'KZT - Kazakhstani Tenge', 'tickera-event-ticketing-system' ),
                'LAK' => __( 'LAK - Lao Kip', 'tickera-event-ticketing-system' ),
                'LBP' => __( 'LBP - Lebanese Pound', 'tickera-event-ticketing-system' ),
                'LKR' => __( 'LKR - Sri Lankan Rupee', 'tickera-event-ticketing-system' ),
                'LRD' => __( 'LRD - Liberian Dollar', 'tickera-event-ticketing-system' ),
                'LSL' => __( 'LSL - Lesotho Loti', 'tickera-event-ticketing-system' ),
                'LTL' => __( 'LTL - Lithuanian Litas', 'tickera-event-ticketing-system' ),
                'LVL' => __( 'LVL - Latvian Lats', 'tickera-event-ticketing-system' ),
                'MAD' => __( 'MAD - Moroccan Dirham', 'tickera-event-ticketing-system' ),
                'MDL' => __( 'MDL - Moldovan Leu', 'tickera-event-ticketing-system' ),
                'MGA' => __( 'MGA - Malagasy Ariary', 'tickera-event-ticketing-system' ),
                'MKD' => __( 'MKD - Macedonian Denar', 'tickera-event-ticketing-system' ),
                'MNT' => __( 'MNT - Mongolian TÃ¶grÃ¶g', 'tickera-event-ticketing-system' ),
                'MOP' => __( 'MOP - Macanese Pataca', 'tickera-event-ticketing-system' ),
                'MRO' => __( 'MRO - Mauritanian Ouguiya', 'tickera-event-ticketing-system' ),
                'MUR' => __( 'MUR - Mauritian Rupee', 'tickera-event-ticketing-system' ),
                'MVR' => __( 'MVR - Maldivian Rufiyaa', 'tickera-event-ticketing-system' ),
                'MWK' => __( 'MWK - Malawian Kwacha', 'tickera-event-ticketing-system' ),
                'MXN' => __( 'MXN - Mexican Peso', 'tickera-event-ticketing-system' ),
                'MYR' => __( 'MYR - Malaysian Ringgit', 'tickera-event-ticketing-system' ),
                'MZN' => __( 'MZN - Mozambican Metical', 'tickera-event-ticketing-system' ),
                'NAD' => __( 'NAD - Namibian Dollar', 'tickera-event-ticketing-system' ),
                'NGN' => __( 'NGN - Nigerian Naira', 'tickera-event-ticketing-system' ),
                'NIO' => __( 'NIO - Nicaraguan Cordoba', 'tickera-event-ticketing-system' ),
                'NOK' => __( 'NOK - Norwegian Krone', 'tickera-event-ticketing-system' ),
                'NPR' => __( 'NPR - Nepalese Rupee', 'tickera-event-ticketing-system' ),
                'NZD' => __( 'NZD - New Zealand Dollar', 'tickera-event-ticketing-system' ),
                'PAB' => __( 'PAB - Panamanian Balboa', 'tickera-event-ticketing-system' ),
                'PEN' => __( 'PEN - Peruvian Nuevo Sol', 'tickera-event-ticketing-system' ),
                'PGK' => __( 'PGK - Papua New Guinean Kina', 'tickera-event-ticketing-system' ),
                'PHP' => __( 'PHP - Philippine Peso', 'tickera-event-ticketing-system' ),
                'PKR' => __( 'PKR - Pakistani Rupee', 'tickera-event-ticketing-system' ),
                'PLN' => __( 'PLN - Polish Zloty', 'tickera-event-ticketing-system' ),
                'PYG' => __( 'PYG - Paraguayan GuaranÃ­', 'tickera-event-ticketing-system' ),
                'QAR' => __( 'QAR - Qatari Riyal', 'tickera-event-ticketing-system' ),
                'RON' => __( 'RON - Romanian Leu', 'tickera-event-ticketing-system' ),
                'RSD' => __( 'RSD - Serbian Dinar', 'tickera-event-ticketing-system' ),
                'RUB' => __( 'RUB - Russian Ruble', 'tickera-event-ticketing-system' ),
                'RWF' => __( 'RWF - Rwandan Franc', 'tickera-event-ticketing-system' ),
                'SAR' => __( 'SAR - Saudi Riyal', 'tickera-event-ticketing-system' ),
                'SBD' => __( 'SBD - Solomon Islands Dollar', 'tickera-event-ticketing-system' ),
                'SCR' => __( 'SCR - Seychellois Rupee', 'tickera-event-ticketing-system' ),
                'SEK' => __( 'SEK - Swedish Krona', 'tickera-event-ticketing-system' ),
                'SGD' => __( 'SGD - Singapore Dollar', 'tickera-event-ticketing-system' ),
                'SHP' => __( 'SHP - Saint Helenian Pound', 'tickera-event-ticketing-system' ),
                'SLL' => __( 'SLL - Sierra Leonean Leone', 'tickera-event-ticketing-system' ),
                'SOS' => __( 'SOS - Somali Shilling', 'tickera-event-ticketing-system' ),
                'SRD' => __( 'SRD - Surinamese Dollar', 'tickera-event-ticketing-system' ),
                'STD' => __( 'STD - SÃ£o TomÃ© and PrÃ­ncipe Dobra', 'tickera-event-ticketing-system' ),
                'SVC' => __( 'SVC - Salvadoran Colon', 'tickera-event-ticketing-system' ),
                'SZL' => __( 'SZL - Swazi Lilangeni', 'tickera-event-ticketing-system' ),
                'THB' => __( 'THB - Thai Baht', 'tickera-event-ticketing-system' ),
                'TJS' => __( 'TJS - Tajikistani Somoni', 'tickera-event-ticketing-system' ),
                'TOP' => __( 'TOP - Tonga Pa\'anga', 'tickera-event-ticketing-system' ),
                'TRY' => __( 'TRY - Turkish Lira', 'tickera-event-ticketing-system' ),
                'TTD' => __( 'TTD - Trinidad and Tobago Dollar', 'tickera-event-ticketing-system' ),
                'TWD' => __( 'TWD - New Taiwan Dollar', 'tickera-event-ticketing-system' ),
                'TZS' => __( 'TZS - Tanzanian Shilling', 'tickera-event-ticketing-system' ),
                'UAH' => __( 'UAH - Ukrainian Hryvnia', 'tickera-event-ticketing-system' ),
                'UGX' => __( 'UGX - Ugandan Shilling', 'tickera-event-ticketing-system' ),
                'USD' => __( 'USD - United States Dollar', 'tickera-event-ticketing-system' ),
                'UYI' => __( 'UYI - Uruguayan Peso', 'tickera-event-ticketing-system' ),
                'UZS' => __( 'UZS - Uzbekistani Som', 'tickera-event-ticketing-system' ),
                'VEF' => __( 'VEF - Venezuelan Bolivar', 'tickera-event-ticketing-system' ),
                'VND' => __( 'VND - Vietnamese Dong ', 'tickera-event-ticketing-system' ),
                'VUV' => __( 'VUV - Vanuatu Vatu', 'tickera-event-ticketing-system' ),
                'WST' => __( 'WST - Samoan Tala', 'tickera-event-ticketing-system' ),
                'XAF' => __( 'XAF - Central African Cfa Franc', 'tickera-event-ticketing-system' ),
                'XCD' => __( 'XCD - East Caribbean Dollar', 'tickera-event-ticketing-system' ),
                'XOF' => __( 'XOF - West African Cfa Franc', 'tickera-event-ticketing-system' ),
                'XPF' => __( 'XPF - Cfp Franc', 'tickera-event-ticketing-system' ),
                'YER' => __( 'YER - Yemeni Rial', 'tickera-event-ticketing-system' ),
                'ZAR' => __( 'ZAR - South African Rand', 'tickera-event-ticketing-system' ),
                'ZMW' => __( 'ZMW - Zambian Kwacha', 'tickera-event-ticketing-system' ),
            );

            $this->currencies = $currencies;
        }

        public function get_stripe_session( $order_id ) {

            global $tc;
            $this->init_stripe();

            $checkout_session_args = [
                'client_reference_id' => $order_id,
                'customer_email' => $this->buyer_info( 'email' ),
                'success_url' => add_query_arg( 'stripe_session_id', '{CHECKOUT_SESSION_ID}', $tc->get_confirmation_slug( true, $order_id ) ),//
                'cancel_url' => $tc->get_cancel_url( $order_id ),
                // 'payment_method_types' => [ 'card' ],
                'payment_intent_data' => [
                    'description' => $this->cart_items(),
                    'metadata' => apply_filters( 'tc_stripe_checkout_metadata', [ 'order_id' => $order_id ], $this->cart_info(), $order_id )
                ]
            ];

            if ( $this->send_receipt ) {
                $checkout_session_args[ 'payment_intent_data' ][ 'receipt_email' ] = $this->buyer_info( 'email' );
            }

            /**
             * Parse session arguments base on the configured API Version.
             * There's no current way to get the configured stripe api version.
             * The api version can be collected only at Stripe Dashboard
             * @since 3.5.1.6
             *
             * API Version from 2022-08-01 onwards will process data under price_data array
             * '1692144000' is equivalent to 2022-08-01
             * @since 3.5.2.7
             */
            if ( strtotime( $this->api_version ) >= 1692144000 ) {
                $checkout_session_args[ 'line_items' ] = [[
                    'price_data' => [
                        'currency' => $this->currency,
                        'unit_amount' => $this->maybe_fix_total( $this->total() ),
                        'product_data' => [
                            'name' => $this->cart_items()
                        ]
                    ],
                    'quantity' => 1,
                ]];
                $checkout_session_args[ 'mode' ] = 'payment';

            } else {
                $checkout_session_args[ 'line_items' ] = [[
                    'currency' => $this->currency,
                    'quantity' => 1,
                    'amount' => $this->maybe_fix_total( $this->total() ),
                    'name' => $this->cart_items()
                ]];
            }

            try {
                $stripe_session = \TCStripe\Checkout\Session::create( $checkout_session_args );

                // Converting to associative array
                $stripe_session = json_encode( $stripe_session );
                $stripe_session = json_decode( $stripe_session, true );

                // Sanitize key values
                $stripe_session = tickera_sanitize_array( $stripe_session, false, true );

                if ( ! $stripe_session ) {
                    throw new \TCStripe\Exception\InvalidRequestException( __( 'Unable to create stripe session.', 'tickera-event-ticketing-system' ) );

                } else {
                    return $stripe_session;
                }

            } catch ( \Exception $e ) {
                $tc->session->set( 'tc_gateway_error', sanitize_text_field( $e->getMessage() ) . ' ' . sprintf( /* translators: 1: A link to Tickera cart page */ __( '<a href="%s">Please try again</a>.', 'tickera-event-ticketing-system' ), esc_url( $tc->get_cart_slug( true ) ) ) );
                tickera_redirect( $tc->get_payment_slug( true ), true );
            }
        }

        public function payment_form( $cart ) {}

        /**
         * Create Stripe Session data.
         * Create Order
         *
         * @param $cart
         */
        public function process_payment( $cart ) {

            global $tc;
            tickera_final_cart_check( $cart );
            $this->save_cart_info();

            $order_id = $tc->generate_order_id();
            $stripe_session = $this->get_stripe_session( $order_id );

            $paid = false;
            $payment_info = $this->save_payment_info();
            $tc->create_order( $order_id, $this->cart_contents(), $this->cart_info(), $payment_info, $paid );
            $order_id = self::parse_order_id( $order_id );

            update_post_meta( $order_id, 'stripe_session_id', isset( $stripe_session[ 'id' ] ) ? sanitize_text_field( $stripe_session[ 'id' ] ) : 'N/A' );
            update_post_meta( $order_id, 'stripe_payment_intent', isset( $stripe_session[ 'payment_intent' ] ) ? sanitize_text_field( $stripe_session[ 'payment_intent' ] ) : 'N/A' );

            @header( 'Content-Type: text/html' );
            ?>
            <div class="frame" style="white-space: nowrap;text-align: center;">
                <span class="helper" style="display: inline-block; height: 100%;vertical-align: middle;"></span>
            </div>
            <script src="https://js.stripe.com/v3/"></script>
            <script>
                const stripe = Stripe( '<?php echo esc_attr( $this->publishable_key ); ?>' );
                stripe.redirectToCheckout( {
                    sessionId: '<?php echo esc_attr( $stripe_session[ 'id' ] ); ?>'
                } ).then( function( result ) {
                } );
            </script>
            <?php
            exit( 0 );
        }

        public function order_confirmation( $order, $payment_info = '', $cart_info = '' ) {

            global $tc;
            $this->init_stripe();

            $order = tickera_get_order_id_by_name( $order );
            $order_id = $order->ID;

            $stripe_session_id = get_post_meta( $order_id, 'stripe_session_id', true );
            $payment_intent_id = get_post_meta( $order_id, 'stripe_payment_intent', true );

            // Make sure the stripe session id is the latest from stripe returned values.
            if ( isset( $_GET[ 'stripe_session_id' ] ) && ! empty( $_GET[ 'stripe_session_id' ] ) ) {
                $stripe_session_id = sanitize_text_field( $_GET[ 'stripe_session_id' ] );
                update_post_meta( $order_id, 'stripe_session_id', sanitize_text_field( $stripe_session_id ) );
            }

            // Make sure the stripe payment intent is updated from stripe session values.
            $stripe_session = \TCStripe\Checkout\Session::retrieve( $stripe_session_id );
            if ( isset( $stripe_session[ 'payment_intent' ] ) && ! empty( $stripe_session[ 'payment_intent' ] ) ) {
                $payment_intent_id = sanitize_text_field( $stripe_session[ 'payment_intent' ] );
                update_post_meta( $order_id, 'stripe_payment_intent', sanitize_text_field( $payment_intent_id ) );
            }

            try {

                $payment_intent = \TCStripe\PaymentIntent::retrieve( $payment_intent_id );
                \Tickera\TC_Order::add_order_note( $order_id, sprintf( /* translators: 1: Either /test or ot 2: Payment Intent */ __( '<a href="https://dashboard.stripe.com%1$s/payments/%2$s" target="_blank">View Payment Intent</a>', 'tickera-event-ticketing-system' ), esc_attr( ( ( ! $this->force_ssl ) ? '/test' : '' ) ), esc_attr( $stripe_session[ 'payment_intent' ] ) ) );

                if ( $payment_intent[ 'status' ] === 'succeeded' ) {
                    $tc->update_order_payment_status( $order_id, true );

                } else {
                    \Tickera\TC_Order::add_order_note( $order_id, __( 'Payment intent status: ', 'tickera-event-ticketing-system' ) . $payment_intent[ 'status' ] );
                }

            } Catch ( \Exception $e ) {
                \Tickera\TC_Order::add_order_note( $order_id, $e->getMessage() );
            }
        }

        /**
         * Generate Order Confirmation Page upon success checkout
         * @param $order
         * @param string $cart_info
         * @return string
         */
        public function order_confirmation_message( $order, $cart_info = '' ) {
            global $tc;

            $content = '';
            $order = tickera_get_order_id_by_name( $order );
            $order = new \Tickera\TC_Order( $order->ID );

            switch ( $order->details->post_status ) {

                case 'order_received':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: 1: Stripe Payment order total amount */ __( 'Your payment via Stripe for this order totaling <strong>%s</strong> is not yet complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    $content .= '<p>' . wp_kses_post( __( 'Current order status: <strong>Pending Payment</strong>', 'tickera-event-ticketing-system' ) ) . '</p>';
                    break;

                case 'order_fraud':
                    $content .= '<p>' . esc_html__( 'Your payment is under review. We will back to you soon.', 'tickera-event-ticketing-system' ) . '</p>';
                    break;

                case 'order_paid':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: 1: Stripe Payment order total amount */ __( 'Your payment via Stripe for this order totaling <strong>%s</strong> is complete.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_cancelled':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: 1: Stripe Payment order total amount */ __( 'Your payment via Stripe for this order totaling <strong>%s</strong> is cancelled.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

                case 'order_refunded':
                    $content .= '<p>' . wp_kses_post( sprintf( /* translators: 1: Stripe Payment order total amount */ __( 'Your payment via Stripe for this order totaling <strong>%s</strong> is refunded.', 'tickera-event-ticketing-system' ), esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) ) . '</p>';
                    break;

            }

            $content = wp_kses_post( apply_filters( 'tc_order_confirmation_message_content', $content, $order, $this->plugin_name ) );
            $tc->remove_order_session_data();
            $tc->maybe_skip_confirmation_screen( $this, $order );
            return $content;
        }

        public function gateway_admin_settings( $settings, $visible ) {
            global $tc;

            // Notify admin if there's a similar/updated version of Stripe Gateway installed
            $tc->tc_payment_gateway_alternative( $this->plugin_name );
            ?>
            <div id="<?php echo esc_attr( $this->plugin_name ); ?>" class="postbox" <?php echo wp_kses_post( ! $visible ? 'style="display:none;"' : '' ); ?>>
                <h3>
                    <span><?php echo esc_html( sprintf( /* translators: %s: Payment Gateway admin name */ __( '%s Settings', 'tickera-event-ticketing-system' ), esc_html( $this->admin_name ) ) ); ?></span>
                    <span class="description"><?php esc_html_e( "Sell your tickets via Stripe and accept Visa, MasterCard, American Express, Discover, JCB, and Diners Club cards", 'tickera-event-ticketing-system' ) ?></span>
                </h3>
                <div class="inside">
                    <?php
                    $fields = array(
                        'is_ssl' => array(
                            'title' => __( 'Mode', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '0' => __( 'Sandbox / Test', 'tickera-event-ticketing-system' ),
                                '1' => __( 'Live', 'tickera-event-ticketing-system' )
                            ),
                            'default' => '0',
                        ),

                        'publishable_key' => array(
                            'title' => __( 'Publishable API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                        ),

                        'private_key' => array(
                            'title' => __( 'Secret API key', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'description' => __( 'You must log in to your Stripe merchant account to <a target="_blank" href="https://manage.stripe.com/#account/apikeys">get your API credentials</a>.', 'tickera-event-ticketing-system' ),
                        ),

                        'api_version' => array(
                            'title' => __( 'API Version', 'tickera-event-ticketing-system' ),
                            'type' => 'text',
                            'placeholder' => $this->default_api_version,
                            'description' => __( 'You must log in to your Stripe merchant account to <a target="_blank" href="https://dashboard.stripe.com/developers">identify your account API version</a>.', 'tickera-event-ticketing-system' ),
                        ),

                        'currency' => array(
                            'title' => __( 'Currency', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => $this->currencies,
                            'default' => 'AUD',
                        ),

                        'send_receipt' => array(
                            'title' => __( 'Send receipt', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '1' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                '0' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 0,
                            'description' => __( 'Allow Stripe to automatically send a receipt to the customer after their payment has been made.', 'tickera-event-ticketing-system' )
                        ),

                        'enable_webhook' => array(
                            'title' => __( 'Enable webhook', 'tickera-event-ticketing-system' ),
                            'type' => 'select',
                            'options' => array(
                                '1' => __( 'Yes', 'tickera-event-ticketing-system' ),
                                '0' => __( 'No', 'tickera-event-ticketing-system' )
                            ),
                            'default' => 0,
                            'description' => __( 'Allow Stripe webhook to stay connected with your website. This will automatically update order statuses based on Stripe\'s end payment intent status', 'tickera-event-ticketing-system' )
                        ),
                    );
                    $form = new \Tickera\TC_Form_Fields_API( $fields, 'tc', 'gateways', $this->plugin_name );
                    ?>
                    <table class="form-table">
                        <?php $form->admin_options(); ?>
                    </table>
                </div>
            </div>
            <?php
        }

        /**
         * Validate decimal data type.
         *
         * @param $val
         * @return bool
         */
        function is_decimal( $val ) {
            return is_numeric( $val ) && floor( $val ) != $val;
        }

        /**
         * Get the actual post_id from order_id
         *
         * @param $order_id
         * @return int
         *
         * @since 3.5.1.6
         */
        function parse_order_id( $order_id ) {

            if ( is_numeric( $order_id ) ) {
                // DO nothing

            } else {
                $order = tickera_get_order_id_by_name( $order_id ); // Get order post from order's slug id
                $order_id = $order->ID;
            }

            return $order_id;
        }

        /**
         * Parse total values with decimals.
         *
         * @param $total
         * @return float|int
         */
        function maybe_fix_total( $total ) {

            if ( apply_filters( 'tc_round_cart_total_value', true ) ) {
                $total = ( in_array( $this->currency, $this->zero_decimal_currencies ) && ! $this->is_decimal( $total ) ) ? $total : round( $total * 100 );

            } else {
                $total = ( in_array( $this->currency, $this->zero_decimal_currencies ) && ! $this->is_decimal( $total ) ) ? $total : ( floor( $total * 100 ) / 100 ) * 100;
            }

            return $total;
        }

        /**
         * Capture Charge via Stripe account and mark order as paid
         *
         * @since 3.5.3.0
         */
        function ipn() {

            global $tc;
            self::on_creation();
            self::init_stripe();

            $payload = @file_get_contents('php://input');

            // Convert to array
            $payload = json_decode( $payload, true );

            // Sanitization
            $request_body = tickera_sanitize_array( $payload, false, true );

            $event_id = $request_body[ 'id' ];
            $event_obj = self::tc_validate_stripe_event( $event_id );

            if ( $event_obj[ 'tc_validated' ] ) {

                $order_statuses = [ 'charge.refunded' => 'order_refunded' ];

                $payment_intent = $event_obj[ 'payment_intent' ];
                $order_id = $payment_intent[ 'metadata' ][ 'order_id' ];

                /*
                 * Check if the cancelled/refund reason is fraud.
                 * Update order status to fraud
                 */
                if ( 'charge.refunded' == $event_obj[ 'type' ] ) {

                    $is_fraud = false;
                    $refund_obj = @$event_obj[ 'data' ][ 'object' ][ 'refunds' ][ 'data' ];

                    foreach ( $refund_obj as $key => $val ) {
                        if ( 'fraudulent' == $val[ 'reason' ] ) {
                            $is_fraud = true;
                            break;
                        }
                    }

                    if ( $is_fraud )
                        $order_statuses[ 'charge.refunded' ] = 'order_fraud';
                }

                if ( 'charge.captured' == $event_obj[ 'type' ] || 'payment_intent.succeeded' == $event_obj[ 'type' ] ) {
                    $tc->update_order_payment_status( $order_id, true );

                } else {
                    wp_update_post( [ 'ID' => $order_id, 'post_status' => sanitize_key( $order_statuses[ $event_obj[ 'type' ] ] ) ] );
                }

                \Tickera\TC_Order::add_order_note( $order_id, $event_obj[ 'type' ] );
            }
        }

        /**
         * Validate Stripe Event.
         * If the event is not found, it is an indication that the event is invalid.
         *
         * @param $event_id
         * @return Event
         *
         * @since 3.5.3.0
         */
        function tc_validate_stripe_event( $event_id ) {

            try {

                $event_obj = Event::retrieve( $event_id );

                $event_obj = tickera_sanitize_array( $event_obj, false, true );

                // Set as default
                $event_obj[ 'tc_validated' ] = false;

                // Note: payment_intent.payment_failed - No action needed
                $enabled_events = [ 'charge.captured', 'charge.refunded', 'payment_intent.succeeded' ];

                if ( $event_obj ) {

                    if ( in_array( $event_obj[ 'type' ], $enabled_events ) ) {

                        // Retrieve Payment Intent ID
                        if ( isset( $event_obj[ 'data' ][ 'object' ][ 'payment_intent' ] ) && $event_obj[ 'data' ][ 'object' ][ 'payment_intent' ] ) {
                            $payment_intent_id = $event_obj[ 'data' ][ 'object' ][ 'payment_intent' ];

                        } else {
                            $payment_intent_id = $event_obj[ 'data' ][ 'object' ][ 'id' ];
                        }

                        $payment_intent = PaymentIntent::retrieve( $payment_intent_id );
                        $payment_intent = tickera_sanitize_array( $payment_intent, false, true );

                        $event_obj[ 'payment_intent' ] = $payment_intent;
                        $meta_order_id = $payment_intent[ 'metadata' ][ 'order_id' ]; // It could be post ID or custom title

                        $order_id = (int) @tickera_get_order_id_by_name( $meta_order_id )->ID;
                        $order_id = ( $order_id && 'tc_orders' == get_post_type( $order_id ) ) ? $order_id : null;

                        if ( $order_id ) {
                            $event_obj[ 'payment_intent' ][ 'metadata' ][ 'order_id' ] = $order_id;
                            $event_obj[ 'tc_validated' ] = true;
                        }
                    }
                }

                return $event_obj;

            } Catch ( \Exception $e ) {
                return [ 'tc_validate' => false ];
            }
        }
    }

    \Tickera\tickera_register_gateway_plugin( '\Tickera\Gateway\TC_Gateway_Stripe', 'stripe', __( 'Stripe', 'tickera-event-ticketing-system' ) );
}