import { stringConvention } from '../functions';
import { useCallback, useEffect, useState } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { debounce } from "lodash";
const blockNamespace = 'tickera/checkout-fields';

var fieldValue,
    confirmFieldValue;

const validateEmail = ( email ) => {
    return ( email.includes( '@' ) ) ? true : false;
};

export const ConfirmEmailFieldControl = ( prop ) => {

    let pageLoaded = prop.loaded,
        item = prop.item,
        i = prop.i;

    prop = prop.data.prop;

    /*
     * setExtensionData will update the wc/store/checkout data store with the values supplied.
     * It can be used to pass data from the client to the server when submitting the checkout form.
     */
    const { setExtensionData } = prop.checkoutExtensionData;

    /*
     * Debounce the setExtensionData function to avoid
     * multiple calls to the API when rapidly changing options.
     */
    const debouncedSetExtensionData = useCallback(
        debounce( ( namespace, key, value ) => {
            setExtensionData( namespace, key, value );
        }, 1000 ),
        [ setExtensionData ]
    );

    const { setValidationErrors, clearValidationError } = useDispatch( 'wc/store/validation' );

    /**
     * Email Field
     *
     * @type {string}
     */
    const emailKey = item.id + '_' + i + '_owner_email_post_meta';
    const [ emailFieldValue, setEmailFieldValue ] = useState( '' );

    // Handles email validation
    const emailValidationErrorId = emailKey + '-validator';
    const emailValidationError = useSelect( ( select ) => {
        const store = select( 'wc/store/validation' );
        return store.getValidationError( emailValidationErrorId );
    } );

    // Listen to email field
    useEffect(() => {

        fieldValue = emailFieldValue;

        // Set Validation Errors. Initially hidden until page has been fully loaded
        let hideError = ! pageLoaded ? true : false;
        setValidationErrors( {
            [ emailValidationErrorId ]: {
                message: tc_woobridge_block.please_enter_a_valid + ' ' + tc_woobridge_block.email_address.toLowerCase(),
                hidden: hideError,
            }
        } );

        setValidationErrors( {
            [ confirmEmailValidationErrorId ]: {
                message: tc_woobridge_block.please_enter_a_valid + ' ' + tc_woobridge_block.confirmation_email_address.toLowerCase(),
                hidden: hideError,
            },
        });

        // Clear error with the following conditions
        if ( pageLoaded && emailFieldValue.trim() && validateEmail( emailFieldValue ) ) {
            clearValidationError( emailValidationErrorId );
        }

        if ( pageLoaded && ( fieldValue == confirmFieldValue ) ) {
            clearValidationError( confirmEmailValidationErrorId );
        }

        // Update in tickera/checkout-fields extended data
        setExtensionData( blockNamespace, emailKey, emailFieldValue );

        // Ensures setExtensionData is being called one at a time.
        debouncedSetExtensionData( blockNamespace, emailKey, emailFieldValue );

    }, [ setExtensionData, emailFieldValue, setValidationErrors, clearValidationError ] );


    /**
     * Email Confirmation Field
     *
     * @type {string}
     */
    const confirmEmailKey = item.id + '_' + i + '_owner_confirm_email_post_meta';
    const [ confirmEmailFieldValue, setConfirmEmailFieldValue ] = useState('');

    // Handles email confirmation validation
    const confirmEmailValidationErrorId = confirmEmailKey + '-validator';
    const confirmEmailValidationError = useSelect((select) => {
        const store = select('wc/store/validation');
        return store.getValidationError( confirmEmailValidationErrorId );
    });

    // Listen to email confirmation field
    useEffect(() => {

        confirmFieldValue = confirmEmailFieldValue;

        // Set Validation Errors. Initially hidden until page has been fully loaded
        let hideError = ! pageLoaded ? true : false;
        setValidationErrors( {
            [ confirmEmailValidationErrorId ]: {
                message: tc_woobridge_block.please_enter_a_valid + ' ' + tc_woobridge_block.confirmation_email_address.toLowerCase(),
                hidden: hideError,
            },
        } );

        // Clear error with the following conditions
        if ( pageLoaded && ( fieldValue == confirmFieldValue ) ) {
            clearValidationError( confirmEmailValidationErrorId );
        }

        // Update in tickera/checkout-fields extended data
        setExtensionData( blockNamespace, confirmEmailKey, confirmEmailFieldValue );

        // Ensures setExtensionData is being called one at a time.
        debouncedSetExtensionData( blockNamespace, confirmEmailKey, confirmEmailFieldValue );

    }, [ setExtensionData, confirmEmailFieldValue, setValidationErrors, clearValidationError ] );

    let emailError = ( ! emailValidationError?.hidden ) ? emailValidationError?.message : false,
        confirmEmailError = ( ! confirmEmailValidationError?.hidden ) ? confirmEmailValidationError?.message : false,
        content = [];

    content.push(
        <div className={ 'tc-wb-block-component tc-wb-block-component-owner_email' + ( emailError ? ' has-error' : '' ) }>
            <div className='wc-block-components-text-input is-active'>
                <input
                    type="email"
                    id='owner_email'
                    onChange={ ( e ) => setEmailFieldValue( e.target.value ) }
                />
                <label htmlFor='owner_email'>{ stringConvention( tc_woobridge_block.email_address ) }</label>
                { emailError ? <div className='wc-block-components-validation-error'><p>{ emailError }</p></div> : '' }
            </div>
        </div>
    );

    content.push(
        <div className={ 'tc-wb-block-component tc-wb-block-component-owner_confirm_email' + ( confirmEmailError ? ' has-error' : '' ) }>
            <div className='wc-block-components-text-input is-active'>
            <input
                type="email"
                id='owner_confirm_email'
                onChange={ ( e ) => setConfirmEmailFieldValue( e.target.value ) }
            />
            <label htmlFor='owner_confirm_email'>{ stringConvention( tc_woobridge_block.confirm_email_address ) }</label>
                { confirmEmailError ? <div className='wc-block-components-validation-error'><p>{ confirmEmailError }</p></div> : '' }
            </div>
        </div>
    );

    return content;
}
