import { useCallback, useEffect, useState } from "@wordpress/element";
import { CheckboxControl } from '@woocommerce/blocks-checkout';
import { useSelect, useDispatch } from '@wordpress/data';
import { debounce } from "lodash";

const blockNamespace = 'tickera/checkout-fields';

export const ConfirmAgeFieldControl = ( prop ) => {

    let field = prop.field,
        pageLoaded = prop.loaded;

    prop = prop.data.prop;

    /*
     * setExtensionData will update the wc/store/checkout data store with the values supplied.
     * It can be used to pass data from the client to the server when submitting the checkout form.
     */
    const { setExtensionData } = prop.checkoutExtensionData;

    /*
     * Debounce the setExtensionData function to avoid
     * multiple calls to the API when rapidly changing options.
     */
    const debouncedSetExtensionData = useCallback(
        debounce((namespace, key, value) => {
            setExtensionData(namespace, key, value);
        }, 1000),
        [setExtensionData]
    );

    var checkboxes = [],
        contents = [];

    let postMeta = ( typeof field.post_meta !== 'undefined' ) ? field.post_meta : true,
        key = field.field_name + ( postMeta == true ? '_post_meta' : '' );
    /**
     * Handles fields validation
     * Applies on checkbox last option
     */
    const validationKey = key;

    const validationErrorId = validationKey + '-validator';
    const { setValidationErrors, clearValidationError } = useDispatch( 'wc/store/validation' );
    const validationError = useSelect( ( select ) => {
        const store = select( 'wc/store/validation' );
        return store.getValidationError( validationErrorId );
    } );

    /**
     * Handles changes in fields value.
     * Exclude Label field.
     */
    let [ fieldValue, setFieldValue ] = useState( '' );

    useEffect( () => {

        // Set Validation Errors. Initially hidden until page has been fully loaded
        let hideError = !pageLoaded ? true : false;
        setValidationErrors( {
            [ validationErrorId ]: {
                message: field.error_message,
                hidden: hideError,
            },
        } );

        // Clear error with the following conditions
        if ( pageLoaded && fieldValue === true ) {
            clearValidationError( validationErrorId );
        }

        // Update in tickera/checkout-fields extended data
        setExtensionData( blockNamespace, key, fieldValue );

        // Ensures setExtensionData is being called one at a time.
        debouncedSetExtensionData( blockNamespace, key, fieldValue );

    }, [ setExtensionData, fieldValue, setValidationErrors, clearValidationError ] );

    let error = ( !validationError?.hidden ) ? validationError?.message : false;
    checkboxes.push( <CheckboxControl id={key} label={field.field_title} checked={ fieldValue } onChange={ setFieldValue }/> );

    contents.push(
        <div className={ 'tc-wb-block-component tc-wb-block-components-' + field.field_name + ( error ? ' has-error' : '' ) }>
            <div className='wc-block-components-checkbox'>
                { checkboxes }
            </div>
            { error ? <div className='wc-block-components-validation-error'><p>{ error }</p></div> : '' }
        </div>
    );

    return contents;
}