import { stringConvention } from '../functions';
import { useCallback, useEffect, useState } from '@wordpress/element';
import { useSelect, useDispatch } from '@wordpress/data';
import { debounce } from "lodash";

const blockNamespace = 'tickera/checkout-fields';

export const NumberFieldControl = ( prop ) => {

    let field = prop.field,
        pageLoaded = prop.loaded,
        category = prop.category,
        item = prop.item,
        i = prop.i;

    prop = prop.data.prop;

    /*
     * setExtensionData will update the wc/store/checkout data store with the values supplied.
     * It can be used to pass data from the client to the server when submitting the checkout form.
     */
    const { setExtensionData } = prop.checkoutExtensionData;

    /*
     * Debounce the setExtensionData function to avoid
     * multiple calls to the API when rapidly changing options.
     */
    const debouncedSetExtensionData = useCallback(
        debounce((namespace, key, value) => {
            setExtensionData(namespace, key, value);
        }, 1000),
        [setExtensionData]
    );

    const key = ( 'owner' == category )
        ? item.id + '_' + i + '_' + field.field_name + '_post_meta'
        : 'buyer_data_' + field.field_name + '_post_meta';

    /**
     * Handles fields validation
     */
    const validationErrorId = key + '-validator';
    const { setValidationErrors, clearValidationError } = useDispatch( 'wc/store/validation' );
    const validationError = useSelect((select) => {
        const store = select('wc/store/validation');
        return store.getValidationError( validationErrorId );
    });

    /**
     * Handles changes in fields value.
     * Exclude Label field.
     */
    const [ fieldValue, setFieldValue ] = useState('');

    useEffect(() => {

        // Set Validation Errors. Initially hidden until page has been fully loaded
        let hideError = ! pageLoaded ? true : false;

        // Required input error handler
        if ( ( field.required && ! fieldValue.trim() ) || isNaN( parseInt( fieldValue ) ) == true ) {

            setValidationErrors( {
                [ validationErrorId ]: {
                    message: tc_woobridge_block.please_enter_a_valid + ' ' + field.field_title.toLowerCase(),
                    hidden: hideError,
                },
            } );

        } else if ( parseInt( field.field_min ) > 0 && ( parseInt( fieldValue ) < parseInt( field.field_min ) ) ) {

            setValidationErrors( {
                [ validationErrorId ]: {
                    message: tc_woobridge_block.please_enter_number + ' ' + parseInt( field.field_min ) + ' ' + tc_woobridge_block.and_above,
                    hidden: hideError,
                },
            } );

        } else if ( parseInt( field.field_max ) > 0 && ( parseInt( fieldValue ) > parseInt( field.field_max ) ) ) {

            setValidationErrors( {
                [ validationErrorId ]: {
                    message: tc_woobridge_block.please_enter_number + ' ' + parseInt( field.field_max ) + ' ' + tc_woobridge_block.and_below,
                    hidden: hideError,
                },
            } );

        } else if ( parseInt( field.field_step ) > 0 && ( parseInt( fieldValue ) % parseInt( field.field_step ) ) > 0 ) {

            setValidationErrors( {
                [ validationErrorId ]: {
                    message: tc_woobridge_block.enter_divisible_by + ' ' + parseInt( field.field_step ),
                    hidden: hideError,
                },
            } );

        } else {

            // Clear error with the following conditions
            if ( pageLoaded ) {
                clearValidationError( validationErrorId );
            }
        }

        // Update in tickera/checkout-fields extended data
        setExtensionData( blockNamespace, key, fieldValue );

        // Ensures setExtensionData is being called one at a time.
        debouncedSetExtensionData( blockNamespace, key, fieldValue );

    }, [ setExtensionData, fieldValue, setValidationErrors, clearValidationError ] );

    let error = ( ! validationError?.hidden ) ? validationError?.message : false;

    return (
        <div className={ 'tc-wb-block-component tc-wb-block-component-' + field.field_name + ( error ? ' has-error' : '' )}>
            <div className='wc-block-components-text-input is-active'>
                <input
                    type="number"
                    id={ field.field_name }
                    min={ field.field_min }
                    max={ field.field_max }
                    step={ field.field_step }
                    onChange={ ( e ) => setFieldValue( e.target.value ) }
                />
                <label htmlFor={ field.field_name }>{ ( field.field_placeholder ) ? stringConvention( field.field_placeholder ) : stringConvention( field.field_title ) } </label>
                { error ? <div className='wc-block-components-validation-error'><p>{ error }</p></div> : '' }
                { ( field.field_description ) ? <div className='wc-block-components-checkout-step__description description'>{ field.field_description }</div> : '' }
            </div>
        </div>
    );
}
