<?php

/**
 * Plugin Name: Tickera CSV Export
 * Plugin URI: http://tickera.com/
 * Description: Export attendees data in CSV file format
 * Author: Tickera.com
 * Author URI: http://tickera.com/
 * Version: 1.3.2
 * Update URI: https://api.freemius.com
 * Text Domain: tccsv
 * Domain Path: /languages/
 * Copyright 2017 Tickera (http://tickera.com/)
 */
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
// Exit if accessed directly
if ( !function_exists( 'csv_export_fs' ) ) {
    // Create a helper function for easy SDK access.
    function csv_export_fs() {
        global $csv_export_fs;
        if ( !isset( $csv_export_fs ) ) {
            // Activate multisite network integration.
            if ( !defined( 'WP_FS__PRODUCT_3168_MULTISITE' ) ) {
                define( 'WP_FS__PRODUCT_3168_MULTISITE', true );
            }
            // Include Freemius SDK.
            if ( file_exists( dirname( dirname( __FILE__ ) ) . '/tickera-event-ticketing-system/freemius/start.php' ) ) {
                /* Try to load SDK from parent plugin folder. */
                require_once dirname( dirname( __FILE__ ) ) . '/tickera-event-ticketing-system/freemius/start.php';
            } elseif ( file_exists( dirname( dirname( __FILE__ ) ) . '/tickera/freemius/start.php' ) ) {
                /* Try to load SDK from premium parent plugin folder. */
                require_once dirname( dirname( __FILE__ ) ) . '/tickera/freemius/start.php';
            } else {
                require_once dirname( __FILE__ ) . '/freemius/start.php';
            }
            $csv_export_fs = fs_dynamic_init( array(
                'id'               => '3168',
                'slug'             => 'csv-export',
                'premium_slug'     => 'csv-export',
                'type'             => 'plugin',
                'public_key'       => 'pk_13a7fbe9592e5e651369b0ef62d7e',
                'is_premium'       => true,
                'is_premium_only'  => true,
                'has_paid_plans'   => true,
                'is_org_compliant' => false,
                'parent'           => array(
                    'id'         => '3102',
                    'slug'       => 'tickera-event-ticketing-system',
                    'public_key' => 'pk_7a38a2a075ec34d6221fe925bdc65',
                    'name'       => 'Tickera',
                ),
                'menu'             => array(
                    'first-path' => 'plugins.php',
                    'support'    => false,
                ),
                'is_live'          => true,
            ) );
        }
        return $csv_export_fs;
    }

}
/**
 * Check if the parent's init SDK method exists.
 *
 * @return bool
 */
function csv_export_fs_is_parent_active_and_loaded() {
    return function_exists( '\\Tickera\\tets_fs' );
}

function csv_export_fs_is_parent_active() {
    $active_plugins = get_option( 'active_plugins', [] );
    if ( is_multisite() ) {
        $network_active_plugins = get_site_option( 'active_sitewide_plugins', [] );
        $active_plugins = array_merge( $active_plugins, array_keys( $network_active_plugins ) );
    }
    foreach ( $active_plugins as $basename ) {
        if ( 0 === strpos( $basename, 'tickera-event-ticketing-system/' ) || 0 === strpos( $basename, 'tickera/' ) ) {
            return true;
        }
    }
    return false;
}

function csv_export_fs_init() {
    if ( csv_export_fs_is_parent_active_and_loaded() ) {
        /*
         * Parent is active, add your init code here.
         * Init Freemius.
         */
        csv_export_fs();
        if ( !csv_export_fs()->can_use_premium_code() ) {
            return;
        }
    } else {
        /* Parent is inactive, add your error handling here. */
    }
}

if ( csv_export_fs_is_parent_active_and_loaded() ) {
    // If parent already included, init add-on.
    csv_export_fs_init();
} else {
    if ( csv_export_fs_is_parent_active() ) {
        // Init add-on only after the parent is loaded.
        add_action( 'tets_fs_loaded', 'csv_export_fs_init' );
    } else {
        // Even though the parent is not activated, execute add-on for activation / uninstall hooks.
        csv_export_fs_init();
    }
}
if ( !class_exists( 'TC_Export_Csv_Mix' ) ) {
    class TC_Export_Csv_Mix {
        var $version = '1.3.0';

        var $tc_version_required = '3.5.2.7';

        var $title = 'CSV Export';

        var $name = 'tc_export_csv_mix';

        var $dir_name = 'csv-export';

        var $location = 'plugins';

        var $plugin_dir = '';

        var $plugin_url = '';

        var $language;

        function __construct() {
            $this->init_vars();
            require_once $this->plugin_dir . 'includes/classes/class.bridge.php';
            require_once $this->plugin_dir . 'includes/classes/class.resource.php';
            require_once $this->plugin_dir . 'includes/classes/class.totals.php';
            add_action( 'admin_notices', array($this, 'admin_notices') );
            add_action( 'activated_plugin', array($this, 'plugin_dependencies'), 99 );
            add_action( 'tc_add_menu_items_after_ticket_templates', array($this, 'add_admin_menu_csv_export_to_tc') );
            add_filter( 'tc_admin_capabilities', array($this, 'append_capabilities') );
            add_action( 'plugins_loaded', array(&$this, 'localization'), 9 );
            if ( isset( $_GET['page'] ) && 'tc_export_csv_mix' == $_GET['page'] && isset( $_GET['post_type'] ) && 'tc_events' == $_GET['post_type'] ) {
                add_action( 'admin_enqueue_scripts', array(&$this, 'enqueue_scripts') );
            }
            add_action( 'wp_ajax_tc_export_attendee_list', array($this, 'tc_export_attendee_list') );
            add_action( 'wp_ajax_tc_export_csv', array($this, 'tc_export') );
            add_action( 'wp_ajax_tc_get_ticket_type', array($this, 'tc_get_ticket_type') );
        }

        /**
         * Load this plugin the very last to ensure all the dependencies loaded first.
         */
        function plugin_dependencies() {
            $plugin_slug = basename( __DIR__ ) . '/' . basename( __FILE__ );
            $active_plugins = get_option( 'active_plugins', [] );
            $index = array_search( $plugin_slug, $active_plugins );
            if ( $index !== false ) {
                unset($active_plugins[$index]);
                $active_plugins[] = $plugin_slug;
                update_option( 'active_plugins', array_values( $active_plugins ) );
            }
        }

        /**
         * On page load get ticket type
         */
        function tc_get_ticket_type() {
            if ( isset( $_POST ) ) {
                $event_name = ( apply_filters( 'tc_bridge_for_woocommerce_is_active', false ) == true ? '_event_name' : 'event_name' );
                $post_type = ( apply_filters( 'tc_bridge_for_woocommerce_is_active', false ) == true ? 'product' : 'tc_tickets' );
                $event_ids = ( isset( $_POST['ids'] ) ? tickera_sanitize_array( $_POST['ids'], false, true ) : [] );
                if ( in_array( 'all', $event_ids ) ) {
                    $events_search = new Tickera\TC_Events_Search('', '', -1);
                    $event_ids = wp_list_pluck( $events_search->get_results(), 'ID' );
                }
                $ticket_type = ( $event_ids ? get_posts( array(
                    'posts_per_page'         => -1,
                    'post_type'              => $post_type,
                    'post_status'            => ['publish', 'private', 'draft'],
                    'meta_key'               => $event_name,
                    'meta_value'             => $event_ids,
                    'update_post_term_cache' => false,
                    'update_post_meta_cache' => false,
                    'cache_results'          => false,
                    'fields'                 => 'ids',
                    'order'                  => 'DESC',
                    'orderby'                => 'ID',
                ) ) : [] );
                $response = [];
                foreach ( $ticket_type as $ticket_type_id ) {
                    // Marking 'TC' to avoid auto json sort in the frontend
                    $response['ticket_types']['TC' . $ticket_type_id] = get_the_title( $ticket_type_id );
                }
                $response['error'] = ( !$response ? true : false );
                wp_send_json( $response );
            }
        }

        /**
         * Append menu capabilities.
         * Add additional capabilities to staff and admins
         *
         * @param $capabilities
         * @return mixed
         */
        function append_capabilities( $capabilities ) {
            $capabilities['manage_' . $this->name . '_cap'] = 1;
            return $capabilities;
        }

        /**
         * Add admin csv menu.
         * Add additional menu item under Tickera admin menu
         *
         */
        function add_admin_menu_csv_export_to_tc() {
            global $first_tc_menu_handler;
            $handler = 'csv_export';
            add_submenu_page(
                $first_tc_menu_handler,
                __( $this->title, 'tccsv' ),
                __( $this->title, 'tccsv' ),
                'manage_' . $this->name . '_cap',
                $this->name,
                $this->name . '_admin'
            );
            eval( "function " . $this->name . "_admin() {require_once( '" . $this->plugin_dir . "includes/admin-pages/settings-tickera_export_csv_mixed_data.php');}" );
            do_action( $this->name . '_add_menu_items_after_' . $handler );
        }

        /**
         * Plugin localization function
         */
        function localization() {
            /*
             * Load up the localization file if we're using WordPress in a different language
             * Place it in this plugin's "languages" folder and name it "tccsv-[value in wp-config].mo"
             */
            if ( 'mu-plugins' == $this->location ) {
                load_muplugin_textdomain( 'tccsv', 'languages/' );
            } elseif ( 'subfolder-plugins' == $this->location ) {
                load_plugin_textdomain( 'tccsv', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
            } elseif ( 'plugins' == $this->location ) {
                load_plugin_textdomain( 'tccsv', false, 'languages/' );
            }
            $temp_locales = explode( '_', get_locale() );
            $this->language = ( $temp_locales[0] ? $temp_locales[0] : 'en' );
        }

        /**
         * Initialize plugin variables
         */
        function init_vars() {
            // Setup proper directories
            if ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . $this->dir_name . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'subfolder-plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/' . $this->dir_name . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );
            } elseif ( defined( 'WP_PLUGIN_URL' ) && defined( 'WP_PLUGIN_DIR' ) && file_exists( WP_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'plugins';
                $this->plugin_dir = WP_PLUGIN_DIR . '/';
                $this->plugin_url = plugins_url( '/', __FILE__ );
            } elseif ( is_multisite() && defined( 'WPMU_PLUGIN_URL' ) && defined( 'WPMU_PLUGIN_DIR' ) && file_exists( WPMU_PLUGIN_DIR . '/' . basename( __FILE__ ) ) ) {
                $this->location = 'mu-plugins';
                $this->plugin_dir = WPMU_PLUGIN_DIR;
                $this->plugin_url = WPMU_PLUGIN_URL;
            } else {
                wp_die( sprintf( __( 'There was an issue determining where %s is installed. Please reinstall it.', 'tccsv' ), $this->title ) );
            }
        }

        /**
         * Include CSS and JS Files
         */
        function enqueue_scripts() {
            wp_enqueue_style( $this->name . '-jquery-ui', $this->plugin_url . 'assets/css/lib/jquery-ui.css' );
            wp_enqueue_style( $this->name . '-admin', $this->plugin_url . 'assets/css/admin.css' );
            wp_enqueue_script( 'jquery-ui-progressbar' );
            wp_enqueue_script(
                $this->name . '-admin',
                $this->plugin_url . 'assets/js/admin.js',
                array('jquery'),
                false,
                false
            );
            $admin_url = strtok( admin_url( 'admin-ajax.php', ( is_ssl() ? 'https' : 'http' ) ), '?' );
            wp_localize_script( $this->name . '-admin', 'tc_csv_vars', array(
                'ajaxUrl'             => $admin_url,
                'select_all'          => __( 'All', 'tccsv' ),
                'ticket_type_message' => __( 'There are no ticket types assigned to this event', 'tccsv' ),
                'attendee_list_error' => __( 'There is no data to export for selected date range', 'tccsv' ),
            ) );
        }

        /**
         * Validate required Tickera version
         */
        function admin_notices() {
            global $tc;
            if ( current_user_can( 'manage_options' ) ) {
                if ( isset( $tc->version ) && version_compare( $tc->version, $this->tc_version_required, '<' ) ) {
                    ?>
                    <div class="notice notice-error">
                        <p><?php 
                    printf(
                        __( '%s add-on requires at least %s version of %s plugin. Your current version of %s is %s. Please update it.', 'tc' ),
                        $this->title,
                        $this->tc_version_required,
                        $tc->title,
                        $tc->title,
                        $tc->version
                    );
                    ?></p>
                    </div>
                    <?php 
                }
            }
        }

        /**
         * Check Session Status and Start
         */
        function tc_session_start() {
            if ( version_compare( PHP_VERSION, '7.0.0', '>=' ) ) {
                if ( function_exists( 'session_status' ) && session_status() == PHP_SESSION_NONE ) {
                    session_start( [
                        'cache_limiter'  => 'private_no_expire',
                        'read_and_close' => false,
                    ] );
                }
            } elseif ( version_compare( PHP_VERSION, '5.4.0', '>=' ) && version_compare( PHP_VERSION, '7.0.0', '<' ) ) {
                if ( function_exists( 'session_status' ) && session_status() == PHP_SESSION_NONE ) {
                    session_cache_limiter( 'private_no_expire' );
                    session_start();
                }
            } else {
                if ( !session_id() ) {
                    if ( version_compare( PHP_VERSION, '4.0.0', '>=' ) ) {
                        session_cache_limiter( 'private_no_expire' );
                    }
                    session_start();
                }
            }
        }

        /**
         * Create virtual CSV.
         * CSV will be processed via Method tc_export
         *
         * @param array $array
         * @return false|string|null
         */
        function array2csv( array $array ) {
            if ( 0 == count( $array ) ) {
                return null;
            }
            ob_start();
            $enclosure = apply_filters( 'tc_export_csv_enable_string_enclosures', true );
            $df = fopen( "php://output", 'w' );
            // CSV columns
            if ( $enclosure ) {
                fputcsv(
                    $df,
                    array_keys( reset( $array ) ),
                    ",",
                    '"',
                    "\\"
                );
            } else {
                fwrite( $df, implode( ',', array_keys( reset( $array ) ) ) . "\n" );
            }
            // CSV row values
            foreach ( $array as $row ) {
                if ( $enclosure ) {
                    fputcsv(
                        $df,
                        $row,
                        ",",
                        '"',
                        "\\"
                    );
                } else {
                    fwrite( $df, implode( ',', $row ) . "\n" );
                }
            }
            fclose( $df );
            return ob_get_clean();
        }

        function set_max( $value ) {
            if ( $value > 100 ) {
                $value = 100;
            }
            return round( $value, 0 );
        }

        function get_events_api_keys() {
        }

        /**
         * Collect and prepare attendees data.
         * Collection will be used in Method tc_export
         */
        function tc_export_attendee_list() {
            error_reporting( E_ERROR );
            self::tc_session_start();
            $resource = new TC_Export_Resource();
            $events_api_keys = $resource->get_events_api_keys();
            $time_start = microtime( true );
            $order_statuses_filter = ( isset( $_POST['tc_limit_order_type'] ) ? (array) $_POST['tc_limit_order_type'] : ['any'] );
            ini_set( 'max_input_time', 3600 * 3 );
            ini_set( 'max_execution_time', 3600 * 3 );
            if ( function_exists( 'set_time_limit' ) ) {
                set_time_limit( 0 );
            }
            $page = max( 1, $_POST['page_num'] );
            $per_page = apply_filters( 'tc_csv_export_per_page_limit', 66 );
            $date_from = date( 'Y-m-d H:i:s', strtotime( $_POST['tc_list_from'] ) );
            // From list
            $date_to = date( 'Y-m-d H:i:s', strtotime( $_POST['tc_list_to'] ) );
            // To list
            $meta_event_id_value = ( isset( $_POST['tc_export_csv_event_data'] ) ? $_POST['tc_export_csv_event_data'] : [] );
            $meta_ticket_type_value = ( isset( $_POST['tc_export_csv_ticket_type_data'] ) ? $_POST['tc_export_csv_ticket_type_data'] : [] );
            /**
             * Condition for all ticket_type or specific 1.
             * Collect all product variation IDs.
             *
             * Skip ticket type filter.
             * @since 1.2.6.9
             *
             * Collect all ticket types and events if filter(s) is set to 'all'
             * @since 1.2.8.3
             */
            if ( !apply_filters( 'tc_export_skip_ticket_type_filter', false ) ) {
                $meta_key = 'ticket_type_id';
                $meta_value = ( in_array( 'all', $meta_ticket_type_value ) ? $resource->get_events_ticket_type_ids( $meta_event_id_value ) : $resource->validate_events_ticket_type_ids( $meta_event_id_value, $meta_ticket_type_value ) );
            } else {
                $meta_key = 'event_id';
                $meta_value = $resource->maybe_get_all_events( $meta_event_id_value );
            }
            $meta_value = apply_filters( 'tc_csv_export_filtered_ticket_type_ids', $meta_value );
            if ( !$meta_value ) {
                wp_send_json( [
                    'exported'       => 0,
                    'page'           => 1,
                    'done'           => true,
                    'execution_time' => 0,
                    'found_posts'    => 0,
                ] );
            }
            /*
             * Tickera alongside Bridge for Woocommerce
             * Cancelled orders will move the tickets to trash
             */
            $post_statuses = 'publish';
            foreach ( $order_statuses_filter as $status ) {
                switch ( $status ) {
                    case 'any':
                        $post_statuses = ['trash', 'publish', 'draft'];
                        break;
                    case 'wc-cancelled':
                        $post_statuses = ['trash'];
                        break;
                }
            }
            $query = new WP_Query(array(
                'date_query'             => array(
                    '0' => array(
                        'after'     => $date_from,
                        'before'    => $date_to,
                        'compare'   => '>=',
                        'column'    => 'post_date',
                        'inclusive' => true,
                    ),
                ),
                'post_type'              => 'tc_tickets_instances',
                'post_status'            => $post_statuses,
                'posts_per_page'         => $per_page,
                'paged'                  => $page,
                'meta_key'               => $meta_key,
                'meta_value'             => $meta_value,
                'update_post_term_cache' => false,
                'update_post_meta_cache' => false,
                'cache_results'          => false,
                'fields'                 => array('ID'),
                'orderby'                => 'ID',
            ));
            if ( 1 == $page ) {
                unset($_SESSION['tc_csv_array']);
                $tc_csv_array = [];
                $processed_orders = [];
                $_SESSION['tc_csv_array']['data'] = $tc_csv_array;
                $_SESSION['tc_csv_array']['orders'] = $processed_orders;
            } else {
                $tc_csv_array = ( isset( $_SESSION['tc_csv_array'] ) && isset( $_SESSION['tc_csv_array']['data'] ) ? $_SESSION['tc_csv_array']['data'] : [] );
                $processed_orders = ( isset( $_SESSION['tc_csv_array'] ) && isset( $_SESSION['tc_csv_array']['orders'] ) ? $_SESSION['tc_csv_array']['orders'] : [] );
            }
            $i = 0;
            while ( $query->have_posts() ) {
                $query->the_post();
                $post_id = get_the_ID();
                // Search all the tickets from the event that are confirmed
                $instance = new Tickera\TC_Ticket_Instance($post_id);
                $order_id = $instance->details->post_parent;
                $order = $resource->get_order( $order_id );
                $order_status = $order->details->post_status;
                if ( in_array( 'any', $order_statuses_filter ) || in_array( $order_status, $order_statuses_filter ) || in_array( 'wc-' . $order_status, $order_statuses_filter ) ) {
                    $totals = new TC_Export_Totals($order);
                    // Check if Order has multiple instances
                    $multiple_instances = ( in_array( $order_id, $processed_orders ) ? true : false );
                    $processed_orders[] = $order_id;
                    // Order Data
                    $order_details = ( isset( $order->details ) ? $order->details : '' );
                    $cart_info = ( isset( $order_details->tc_cart_info ) ? $order_details->tc_cart_info : [] );
                    // Ticket Instance Data
                    $instance_details = ( isset( $instance->details ) ? $instance->details : '' );
                    $instance_id = ( isset( $instance_details->ID ) ? $instance_details->ID : '' );
                    // Event Data
                    $event = new Tickera\TC_Event($instance->details->event_id);
                    // Buyer data
                    $buyer_data = ( isset( $cart_info['buyer_data'] ) ? $cart_info['buyer_data'] : '' );
                    // Collect event name
                    $tc_event_name_array = ( isset( $_POST['col_event_name'] ) ? array(
                        __( 'Event Name', 'tccsv' ) => ( isset( $event->details->post_title ) ? $event->details->post_title : '' ),
                    ) : [] );
                    do_action( 'tc_export_csv_after_event_name', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect owner first names
                    $owner_first_name = ( isset( $instance_details->first_name ) ? $instance_details->first_name : '' );
                    $tc_first_name_array = ( isset( $_POST['col_owner_first_name'] ) ? array(
                        __( 'First Name', 'tccsv' ) => $owner_first_name,
                    ) : [] );
                    do_action( 'tc_export_csv_after_owner_first_name', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect owner last names
                    $owner_last_name = ( isset( $instance_details->last_name ) ? $instance_details->last_name : '' );
                    $tc_last_name_array = ( isset( $_POST['col_owner_last_name'] ) ? array(
                        __( 'Last Name', 'tccsv' ) => $owner_last_name,
                    ) : [] );
                    do_action( 'tc_export_csv_after_owner_last_name', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect owner full names
                    $owner_full_name = $owner_first_name . ' ' . $owner_last_name;
                    $tc_name_array = ( isset( $_POST['col_owner_name'] ) ? array(
                        __( 'Name', 'tccsv' ) => $owner_full_name,
                    ) : [] );
                    do_action( 'tc_export_csv_after_owner_name', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect owner emails
                    $owner_email = ( isset( $instance_details->owner_email ) ? $instance_details->owner_email : '' );
                    $tc_owner_email_array = ( isset( $_POST['col_owner_email'] ) ? array(
                        __( 'Attendee E-mail', 'tccsv' ) => $owner_email,
                    ) : [] );
                    do_action( 'tc_export_csv_after_owner_email', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect order payment dates
                    $order_post_date_gmt = ( isset( $order_details->post_date_gmt ) ? tickera_format_date( strtotime( $order_details->post_date_gmt ) ) : '' );
                    $tc_payment_array = ( isset( $_POST['col_payment_date'] ) ? array(
                        __( 'Payment Date', 'tccsv' ) => $order_post_date_gmt,
                    ) : [] );
                    // Collect order numbers
                    $order_post_title = ( isset( $order_details->post_title ) ? $order_details->post_title : '' );
                    $tc_order_number_array = ( isset( $_POST['col_order_number'] ) ? array(
                        __( 'Order Number', 'tccsv' ) => apply_filters( 'tc_export_order_number_column_value', $order_post_title, $order_id ),
                    ) : [] );
                    do_action( 'tc_export_csv_after_order_number', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect order payment gateways
                    $gateway_admin_name = ( isset( $cart_info['gateway_admin_name'] ) ? $cart_info['gateway_admin_name'] : '' );
                    $tc_payment_gateway_array = ( isset( $_POST['col_payment_gateway'] ) ? array(
                        __( 'Payment Gateway', 'tccsv' ) => apply_filters( 'tc_order_payment_gateway_name', $gateway_admin_name, $order_id ),
                    ) : [] );
                    do_action( 'tc_export_csv_after_payment_gateway', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect order discount codes
                    $discount_code = ( isset( $order_details->tc_discount_code ) ? $order_details->tc_discount_code : '' );
                    $tc_discount_array = ( isset( $_POST['col_discount_code'] ) ? array(
                        __( 'Discount Code', 'tccsv' ) => $discount_code,
                    ) : [] );
                    do_action( 'tc_export_csv_after_discount_value', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect order statuses
                    $tc_order_status_values = [
                        'order_paid'      => __( 'Paid', 'tccsv' ),
                        'order_received'  => __( 'Received / Pending', 'tccsv' ),
                        'order_fraud'     => __( 'Fraud Detected', 'tccsv' ),
                        'order_cancelled' => __( 'Cancelled', 'tccsv' ),
                        'order_refunded'  => __( 'Refunded', 'tccsv' ),
                    ];
                    $order_st = ( isset( $tc_order_status_values[$order_status] ) ? __( $tc_order_status_values[$order_status], 'tccsv' ) : '' );
                    $order_st = apply_filters(
                        'tc_order_status_title',
                        $order_st,
                        $order_id,
                        $order_status
                    );
                    $order_st = ( $order_st ? $order_st : $order_status );
                    $tc_order_status_array = ( isset( $_POST['col_order_status'] ) ? array(
                        __( 'Order Status', 'tccsv' ) => $order_st,
                    ) : [] );
                    do_action( 'tc_export_csv_after_order_status', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect order totals
                    $tc_order_total_array = ( isset( $_POST['col_order_total'] ) ? array(
                        __( 'Order Total', 'tccsv' ) => $totals->get_order_totals()->total,
                    ) : [] );
                    do_action( 'tc_export_csv_after_order_total', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect order totals (once)
                    $tc_order_total_once_array = [];
                    if ( isset( $_POST['col_order_total_once'] ) ) {
                        $tc_order_total_once_array = ( !$multiple_instances ? array(
                            __( 'Order Total (Shown Once)', 'tccsv' ) => $totals->get_order_totals()->total,
                        ) : array(
                            __( 'Order Total (Shown Once)', 'tccsv' ) => '',
                        ) );
                    }
                    do_action( 'tc_export_csv_after_order_total_once', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect Ticket Order Total Values
                    $owner_totals_column = [
                        'col_ticket_subtotal',
                        'col_ticket_discount',
                        'col_ticket_fee',
                        'col_ticket_tax',
                        'col_ticket_total'
                    ];
                    foreach ( $owner_totals_column as $column_key ) {
                        $meta_name = str_replace( 'col_', '', $column_key );
                        $column_name = str_replace( '_', ' ', $meta_name );
                        $column_value = $totals->get_ticket_totals( $instance_id )->{str_replace( 'ticket_', '', $meta_name )};
                        ${$meta_name} = ( isset( $_POST[$column_key] ) ? array(
                            __( ucwords( $column_name ), 'tccsv' ) => $column_value,
                        ) : [] );
                        do_action( 'tc_export_csv_after_' . $meta_name, ( isset( $_POST ) ? $_POST : '' ) );
                    }
                    // Collect ticket codes
                    $ticket_code = ( isset( $instance_details->ticket_code ) ? $instance_details->ticket_code : '' );
                    $tc_ticket_id_array = ( isset( $_POST['col_ticket_id'] ) ? array(
                        __( 'Ticket Code', 'tccsv' ) => $ticket_code,
                    ) : [] );
                    do_action( 'tc_export_csv_after_ticket_id', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect ticket instance ids
                    $tc_ticket_type_instance_id = ( isset( $_POST['col_ticket_instance_id'] ) ? array(
                        __( 'Ticket ID', 'tccsv' ) => $instance_id,
                    ) : [] );
                    do_action( 'tc_export_csv_after_ticket_instance_id', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect ticket type ids
                    $tc_ticket_type_array = [];
                    $ticket_type_id = ( isset( $instance_details->ticket_type_id ) ? $instance_details->ticket_type_id : '' );
                    if ( isset( $_POST['col_ticket_type'] ) ) {
                        $tc_ticket_type_array = array(
                            __( 'Ticket Type', 'tccsv' ) => apply_filters(
                                'tc_checkout_owner_info_ticket_title',
                                html_entity_decode( get_the_title( $ticket_type_id ) ),
                                $ticket_type_id,
                                [],
                                $instance_id
                            ),
                        );
                        // html_entity_decode to ensure string/character is printed properly.
                    }
                    do_action( 'tc_export_csv_after_ticket_type', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect buyer first names
                    $tc_buyer_first_name_info_array = [];
                    $buyer_first_name = ( isset( $buyer_data['first_name_post_meta'] ) ? $buyer_data['first_name_post_meta'] : '' );
                    if ( isset( $_POST['col_buyer_first_name'] ) ) {
                        $tc_buyer_first_name_info_array = array(
                            __( 'Buyer First Name', 'tccsv' ) => apply_filters( 'tc_ticket_checkin_buyer_first_name', $buyer_first_name, $order_id ),
                        );
                    }
                    do_action( 'tc_export_csv_after_buyer_first_name', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect buyer last names
                    $tc_buyer_last_name_info_array = [];
                    $buyer_last_name = ( isset( $buyer_data['last_name_post_meta'] ) ? $buyer_data['last_name_post_meta'] : '' );
                    if ( isset( $_POST['col_buyer_last_name'] ) ) {
                        $tc_buyer_last_name_info_array = array(
                            __( 'Buyer Last Name', 'tccsv' ) => apply_filters( 'tc_ticket_checkin_buyer_last_name', $buyer_last_name, $order_id ),
                        );
                    }
                    do_action( 'tc_export_csv_after_buyer_last_name', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect buyer full names
                    $tc_buyer_info_array = [];
                    if ( isset( $_POST['col_buyer_name'] ) ) {
                        $buyer_full_name = $buyer_first_name . ' ' . $buyer_last_name;
                        $tc_buyer_info_array = array(
                            __( 'Buyer Name', 'tccsv' ) => apply_filters( 'tc_ticket_checkin_buyer_full_name', $buyer_full_name, $order_id ),
                        );
                    }
                    do_action( 'tc_export_csv_after_buyer_name', ( isset( $_POST ) ? $_POST : '' ) );
                    // Collect buyer emails
                    $tc_buyer_email_array = [];
                    if ( isset( $_POST['col_buyer_email'] ) ) {
                        $buyer_email = ( isset( $buyer_data['email_post_meta'] ) ? $buyer_data['email_post_meta'] : '' );
                        $tc_buyer_email_array = array(
                            __( 'Buyer E-Mail', 'tccsv' ) => apply_filters( 'tc_ticket_checkin_buyer_email', $buyer_email, $order_id ),
                        );
                    }
                    do_action( 'tc_export_csv_after_email', ( isset( $_POST ) ? $_POST : '' ) );
                    /*
                     * Prepare col_checked_in and col_checkins variables
                     * @since 1.2.8
                     */
                    if ( isset( $_POST['col_checked_in'] ) || isset( $_POST['col_checkins'] ) ) {
                        $checkins = get_post_meta( $instance_id, 'tc_checkins', true );
                        $checkins = ( $checkins ? $checkins : [] );
                    }
                    // Check to see if attendee is checked-in
                    $tc_checked_in_array = [];
                    if ( isset( $_POST['col_checked_in'] ) ) {
                        $checked_in = __( 'No', 'tccsv' );
                        foreach ( $checkins as $checkin ) {
                            if ( 'Pass' == $checkin['status'] ) {
                                $checked_in = ( is_array( $checkins ) && $checkins ? __( 'Yes', 'tccsv' ) : __( 'No', 'tccsv' ) );
                                break;
                            }
                        }
                        $tc_checked_in_array = array(
                            __( 'Checked-in', 'tccsv' ) => $checked_in,
                        );
                    }
                    do_action( 'tc_export_csv_after_checked_in', ( isset( $_POST ) ? $_POST : '' ) );
                    // Check-ins list for an attendee
                    $tc_checkins_array = [];
                    if ( isset( $_POST['col_checkins'] ) ) {
                        $checkins_list = [];
                        foreach ( $checkins as $checkin ) {
                            $api_key = ( isset( $checkin['api_key_id'] ) ? $checkin['api_key_id'] : '' );
                            $api_key_obj = new Tickera\TC_API_Key((int) $api_key);
                            $api_key_name = ( isset( $api_key_obj->details ) && isset( $api_key_obj->details->api_key_name ) ? $api_key_obj->details->api_key_name : '' );
                            $api_key_title = '';
                            if ( apply_filters( 'tc_show_checkins_api_key_names', true ) ) {
                                $api_key_name = ( !empty( $api_key_name ) ? $api_key_name : $api_key );
                                $api_key_title = ( $api_key_name ? ' (' . $api_key_name . ')' : '' );
                            }
                            if ( apply_filters( 'tc_csv_export_show_checkins_status', true ) ) {
                                $api_key_title .= '(' . $checkin['status'] . ')';
                            }
                            $date_checked_in = ( isset( $checkin['date_checked'] ) ? tickera_format_date( $checkin['date_checked'] ) . $api_key_title : '' );
                            if ( $date_checked_in ) {
                                $checkins_list[] = $date_checked_in;
                            }
                        }
                        $tc_checkins_array = array(
                            __( 'Check-ins', 'tccsv' ) => implode( "\n", $checkins_list ),
                        );
                    }
                    do_action( 'tc_export_csv_after_checkins', ( isset( $_POST ) ? $_POST : '' ) );
                    // Check-outs list for an attendee
                    $tc_checkouts_array = [];
                    if ( isset( $_POST['col_checkouts'] ) ) {
                        $checkouts_list = [];
                        $checkouts = get_post_meta( $instance_id, 'tc_checkouts', true );
                        $checkouts = ( $checkouts ? $checkouts : [] );
                        foreach ( $checkouts as $checkout ) {
                            $api_key = ( isset( $checkout['api_key_id'] ) ? $checkout['api_key_id'] : '' );
                            $api_key_obj = new Tickera\TC_API_Key((int) $api_key);
                            $api_key_name = ( isset( $api_key_obj->details ) && isset( $api_key_obj->details->api_key_name ) ? $api_key_obj->details->api_key_name : '' );
                            $api_key_title = '';
                            if ( apply_filters( 'tc_show_checkins_api_key_names', true ) == true ) {
                                $api_key_name = ( !empty( $api_key_name ) ? $api_key_name : $api_key );
                                $api_key_title = ( $api_key_name ? ' (' . $api_key_name . ')' : '' );
                            }
                            if ( apply_filters( 'tc_csv_export_show_checkouts_status', true ) ) {
                                $api_key_title .= '(' . $checkout['status'] . ')';
                            }
                            $date_checked_out = ( isset( $checkout['date_checked'] ) ? tickera_format_date( $checkout['date_checked'] ) . $api_key_title : '' );
                            if ( $date_checked_out ) {
                                $checkouts_list[] = $date_checked_out;
                            }
                        }
                        $tc_checkouts_array = array(
                            __( 'Check-outs', 'tccsv' ) => implode( "\r\n", $checkouts_list ),
                        );
                    }
                    do_action( 'tc_export_csv_after_checkouts', ( isset( $_POST ) ? $_POST : '' ) );
                    // API Keys list for an attendee
                    $tc_api_key_array = [];
                    if ( isset( $_POST['col_owner_api_key'] ) ) {
                        $api_keys = $events_api_keys[$instance->details->event_id];
                        $tc_api_key_array = ( $api_keys ? array(
                            __( 'API Key', 'tccsv' ) => implode( ",", $api_keys ),
                        ) : [] );
                    }
                    do_action( 'tc_export_csv_after_api_key', ( isset( $_POST ) ? $_POST : '' ) );
                    // Price list for an attendee
                    $tc_order_price_array = [];
                    if ( isset( $_POST['col_order_price'] ) ) {
                        if ( false == apply_filters( 'tc_bridge_for_woocommerce_is_active', false ) ) {
                            // Check bridge is active or not
                            $tc_order_price = get_post_meta( $instance_id, 'ticket_subtotal', true );
                            $tc_order_price_array = array(
                                __( 'Price', 'tccsv' ) => $tc_order_price,
                            );
                        } else {
                            $tc_order_price = get_post_meta( $ticket_type_id, '_price', true );
                            $tc_order_price_array = array(
                                __( 'Price', 'tccsv' ) => $tc_order_price,
                            );
                        }
                    }
                    // Order Notes
                    $tc_order_notes_array = [];
                    if ( isset( $_POST['col_order_notes'] ) ) {
                        $order_notes = \Tickera\TC_Order::get_order_notes( $order_id );
                        if ( isset( $order_notes ) && isset( $order_notes['tc_order_notes'] ) && count( $order_notes['tc_order_notes'] ) > 0 ) {
                            $notes = [];
                            foreach ( $order_notes['tc_order_notes'] as $order_note ) {
                                $notes[] = $order_note['note'] . ' - ' . $order_note['note_author'] . ', ' . $order_note['created_at'];
                            }
                            $tc_order_notes_array = array(
                                __( 'Order Notes', 'tccsv' ) => implode( "\n", $notes ),
                            );
                        }
                    }
                    // Ticket Download Url
                    $ticket_download_urls = [];
                    if ( isset( $_POST['col_ticket_download_url'] ) ) {
                        $ticket_download_urls = array(
                            __( 'Ticket Download Url', 'tccsv' ) => tickera_get_raw_ticket_download_link(
                                '',
                                '',
                                $instance->details->ID,
                                true
                            ),
                        );
                        do_action( 'tc_export_csv_after_ticket_downloa_url', ( isset( $_post ) ? $_post : '' ) );
                    }
                    $tc_csv_array[] = apply_filters(
                        'tc_csv_array',
                        array_merge(
                            $tc_event_name_array,
                            $tc_first_name_array,
                            $tc_last_name_array,
                            $tc_name_array,
                            $tc_owner_email_array,
                            $tc_payment_array,
                            $tc_order_number_array,
                            $tc_payment_gateway_array,
                            $tc_order_status_array,
                            $tc_order_total_array,
                            $tc_order_total_once_array,
                            @$ticket_subtotal,
                            @$ticket_discount,
                            @$ticket_fee,
                            @$ticket_tax,
                            @$ticket_total,
                            $tc_discount_array,
                            $tc_ticket_id_array,
                            $ticket_download_urls,
                            $tc_ticket_type_instance_id,
                            $tc_ticket_type_array,
                            $tc_buyer_first_name_info_array,
                            $tc_buyer_last_name_info_array,
                            $tc_buyer_info_array,
                            $tc_buyer_email_array,
                            $tc_checked_in_array,
                            $tc_checkins_array,
                            $tc_checkouts_array,
                            $tc_api_key_array,
                            $tc_order_price_array,
                            $tc_order_notes_array
                        ),
                        $order,
                        $instance,
                        $_POST,
                        $i
                    );
                    $_SESSION['tc_csv_array']['data'] = $tc_csv_array;
                }
                // Record processed order ids. Used for Total (Shown Once)
                $_SESSION['tc_csv_array']['orders'] = $processed_orders;
                $i++;
            }
            $exported = $page * $per_page;
            $time_end = microtime( true );
            $execution_time = $time_end - $time_start;
            $response = array(
                'exported'       => ( $query->found_posts > 0 ? $this->set_max( ceil( $exported / ($query->found_posts / 100) ) ) : 0 ),
                'page'           => $page + 1,
                'done'           => ( $exported >= $query->found_posts ? true : false ),
                'execution_time' => $execution_time,
                'found_posts'    => $query->found_posts,
            );
            if ( $response['done'] ) {
                // Keep fields selection
                if ( isset( $_POST['tc_keep_selection_fields'] ) ) {
                    update_option( 'tc_atteende_keep_selection', tickera_sanitize_array( $_POST, false, true ) );
                } else {
                    delete_option( 'tc_atteende_keep_selection' );
                }
            }
            wp_send_json( $response );
        }

        /**
         * Process CSV Export and force download onto the browser
         */
        function tc_export() {
            self::tc_session_start();
            if ( defined( 'TC_DEBUG' ) ) {
                error_reporting( E_ALL );
                ini_set( 'display_errors', 1 );
            }
            $this->download_send_headers( sanitize_text_field( $_GET['document_title'] ) . ".csv" );
            if ( isset( $_SESSION['tc_csv_array'] ) ) {
                echo $this->array2csv( $_SESSION['tc_csv_array']['data'] );
                unset($_SESSION['tc_csv_array']);
            }
            exit;
        }

        /**
         * Initialise download headers
         *
         * @param $filename
         */
        function download_send_headers( $filename ) {
            // Disable caching
            if ( $filename ) {
                $filename = str_replace( ',', '', $filename );
                $now = gmdate( "D, d M Y H:i:s" );
                header( "Expires: Tue, 03 Jul 2001 06:00:00 GMT" );
                header( "Cache-Control: max-age=0, no-cache, must-revalidate, proxy-revalidate" );
                header( "Last-Modified: {$now} GMT" );
                // Force download
                header( "Content-Type: application/force-download" );
                header( "Content-Type: application/octet-stream" );
                header( "Content-Type: application/download" );
                // Disposition / Encoding on response body
                header( "Content-Disposition: attachment;filename={$filename}" );
                header( "Content-Transfer-Encoding: binary" );
            }
        }

    }

}
if ( !function_exists( 'is_plugin_active_for_network' ) ) {
    require_once ABSPATH . '/wp-admin/includes/plugin.php';
}
//if( csv_export_fs_is_parent_active_and_loaded() && csv_export_fs()->can_use_premium_code() ) {
//    if ( is_multisite() && is_plugin_active_for_network( plugin_basename( __FILE__ ) ) ) {
//
//        add_action( 'tets_fs_loaded', 'tc_export_csv_mix_load', 20 );
//        function tc_export_csv_mix_load() {
//            global $tc_export_csv_mix;
//            $tc_export_csv_mix = new TC_Export_Csv_Mix();
//        }
//
//    } else {
//        $tc_export_csv_mix = new TC_Export_Csv_Mix();
//    }
//}
$tc_export_csv_mix = new TC_Export_Csv_Mix();