<?php
/*
 * TC_Firebase
 */

if ( !defined( 'ABSPATH' ) )
    exit; // Exit if accessed directly

class TC_Firebase {

    var $firebase_path = '';
    var $firebase = '';
    var $firebase_secret = '';
    var $firebase_enabled = false;

    function __construct() {

        $tc_seat_charts_settings = TC_Seat_Chart::get_settings();
        $this->firebase_enabled = isset( $tc_seat_charts_settings['user_firebase_integration'] ) ? $tc_seat_charts_settings['user_firebase_integration'] : false;
        $this->firebase_secret = isset( $tc_seat_charts_settings['secret'] ) ? $tc_seat_charts_settings['secret'] : '';
        $this->firebase_path = isset( $tc_seat_charts_settings['databaseURL'] ) ? $tc_seat_charts_settings['databaseURL'] : '';

        $this->init_firebase();

        add_action( 'wp_ajax_nopriv_tc_add_seat_to_firebase_cart', array( &$this, 'tc_add_seat_to_firebase_cart' ) );
        add_action( 'wp_ajax_tc_add_seat_to_firebase_cart', array( &$this, 'tc_add_seat_to_firebase_cart' ) );
        add_action( 'wp_ajax_nopriv_tc_remove_expired_firebase_seats', array( &$this, 'tc_remove_expired_firebase_seats' ) );
        add_action( 'wp_ajax_tc_remove_expired_firebase_seats', array( &$this, 'tc_remove_expired_firebase_seats' ) );
        add_action( 'wp_ajax_nopriv_tc_initialize_firebase_reserved_seats', array( &$this, 'tc_initialize_firebase_reserved_seats' ) );
        add_action( 'wp_ajax_tc_initialize_firebase_reserved_seats', array( &$this, 'tc_initialize_firebase_reserved_seats' ) );
        add_action( 'wp_ajax_nopriv_tc_remove_seat_from_firebase_cart', array( &$this, 'tc_remove_seat_from_firebase_cart' ) );
        add_action( 'wp_ajax_tc_remove_seat_from_firebase_cart', array( &$this, 'tc_remove_seat_from_firebase_cart' ) );
        add_action( 'tc_seat_chart_woo_cart_item_remove_seat', array( &$this, 'tc_remove_seat_from_firebase_cart' ), 10, 2 );
        add_action( 'tc_check_in_notification', array( &$this, 'tc_check_in_added' ), 99, 1 );
        add_action( 'tc_check_in_deleted', array( &$this, 'tc_check_in_deleted' ), 10, 2 );
        add_action( 'delete_post', array( &$this, 'tc_delete_chart_from_firebase' ), 10, 1 );
        add_action( 'admin_enqueue_scripts', array( $this, 'admin_firebase_scripts' ) );
        add_action( 'tc_remove_order_session_data', array( $this, 'remove_in_cart_from_firebase' ) );

        // Handles removal of firebase reserved entries when post is trashed/deleted. Also when there's an order cancellation
        add_action( 'save_post', array( $this, 'tc_remove_firebase_reserved_seats' ) );
        add_action( 'trashed_post', array( $this, 'tc_remove_firebase_reserved_seats' ) );
        add_action( 'deleted_post', array( $this, 'tc_remove_firebase_reserved_seats' ) );

        // Handles addition of firebase reserved entries when order is created
        add_action( 'save_post', array( $this, 'tc_insert_firebase_reserved_seats' ) );
        add_action( 'tc_created_order_ticket_instance', array( $this, 'tc_insert_firebase_reserved_seats' ), 10, 3 );

        add_action( 'init', array( $this, 'capture_update_cart_actions' ) );
    }

    /**
     * Load Scripts and Styles
     */
    function admin_firebase_scripts() {

        global $TC_Seat_Chart;

        if ( $this->firebase_enabled ) {

            $tc_seat_charts_settings = TC_Seat_Chart::get_settings();

            if (!session_id()) {
                @session_start();
            }

            // Include only if base uri and Secret Key exists
            if ( $this->firebase_path && $this->firebase_secret ) {
                wp_enqueue_script( 'tc-firebase-app', 'https://www.gstatic.com/firebasejs/8.4.1/firebase-app.js', [], $TC_Seat_Chart->version );
                wp_enqueue_script( 'tc-firebase-database', 'https://www.gstatic.com/firebasejs/8.4.1/firebase-database.js', [], $TC_Seat_Chart->version );
            }

            wp_enqueue_script( 'tc-seat-charts-firebase-admin', plugins_url( '../js/firebase/admin.js', __FILE__ ), array( 'jquery', 'tc-firebase-app', 'tc-firebase-database' ), $TC_Seat_Chart->version, false );
            wp_localize_script( 'tc-seat-charts-firebase-admin', 'tc_firebase_vars', array (
                    'apiKey' => isset( $tc_seat_charts_settings['apiKey'] ) ? $tc_seat_charts_settings['apiKey'] : '',
                    'authDomain' => isset( $tc_seat_charts_settings['authDomain'] ) ? $tc_seat_charts_settings['authDomain'] : '',
                    'databaseURL' => isset( $tc_seat_charts_settings['databaseURL'] ) ? $tc_seat_charts_settings['databaseURL'] : '',
                    'session_id' => session_id(),
                    'chart_id' => isset( $_GET['post'] ) ? $_GET['post'] : false,
                    'checkedin_seat_color' => isset( $tc_seat_charts_settings['checkedin_seat_color'] ) ? $tc_seat_charts_settings['checkedin_seat_color'] : '#000',
                )
            );
        }
    }

    /**
     * Initialize Firebase
     *
     * @global type $TC_Seat_Chart
     */
    function init_firebase() {

        // Firebase Lib
        require_once 'firebase/firebaseInterface.php';
        require_once 'firebase/firebaseStub.php';
        require_once 'firebase/firebaseLib.php';

        // Executes only if base URI and Secret Key are provided
        if ( $this->firebase_path && $this->firebase_secret ) {
            $this->firebase = new \Firebase\FirebaseLib( $this->firebase_path, $this->firebase_secret );
        }
    }

    /**
     * Capture Cart Update Action.
     * Remove seating chart's firebase entries when cart is emptied.
     */
    function capture_update_cart_actions() {

        if ( isset( $_POST['cart_action'] ) && 'empty_cart' == $_POST['cart_action'] ) {

            $in_cart_seats = TC_Seat_Chart::get_cart_seats_cookie();

            foreach ( $in_cart_seats as $seat_ticket_type_id => $ticket_type_seats_in_carts ) {

                foreach ( $ticket_type_seats_in_carts as $ticket_type_seats_in_cart ) {
                    $seat_id = $ticket_type_seats_in_cart[0];
                    $chart_id = (int) $ticket_type_seats_in_cart[2];
                    $this->firebase->delete( '/in-cart/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent') );
                }
            }
        }
    }

    /**
     * This method will execute after completing a checkout process.
     * Remove in cart seats from the firebase.
     * Insert reserved seats onto firebase.
     *
     * @param $order
     */
    function remove_in_cart_from_firebase( $order ) {

        if ( !session_id() ) {
            @session_start();
        }

        $in_cart_seats = TC_Seat_Chart::get_cart_seats_cookie();

        $data = [];
        $data['timestamp'] = ( time() * 1000 );
        $data['session_id'] = session_id();

        foreach ( $in_cart_seats as $seat_ticket_type_id => $ticket_type_seats_in_carts ) {

            foreach ( $ticket_type_seats_in_carts as $ticket_type_seats_in_cart ) {
                $seat_id = $ticket_type_seats_in_cart[0];
                $chart_id = (int) $ticket_type_seats_in_cart[2];

                $this->firebase->delete( '/in-cart/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent') );
                $this->firebase->update( '/reserved/' . $chart_id . '/' . $seat_id, $data, array( 'print' => 'silent' ) );
            }
        }
    }

    /**
     * If an order and it's ticket instances are created, mark those seats as reserved in firebase.
     * This is to ensure any seats have been recorded onto firebase after an order is placed.
     * Issues occur when a payment gateway requires redirection.
     *
     * @param $meta_id
     * @param $object_id
     * @param $meta_key
     * @param $meta_value
     */
    function tc_insert_firebase_reserved_seats( $post_id, $order_id = false, $blocks = false ) {

        $order_types = apply_filters( 'tc_order_types', [ 'tc_orders' ] );

        // If the order is coming from woocommerce blocks, include draft status
        $reserved_statuses = TC_Seat_Chart::get_reserved_order_statuses( $blocks );

        if (
            ( isset( $_POST['action'] ) && 'editpost' == $_POST['action'] )
            && ( isset( $_POST['save'] ) && 'Update' == $_POST['save'] )
            && in_array( $_POST[ 'post_type' ], $order_types )
        ) {

            // Executes when an order status is changes from the admin.
            if ( in_array( $_POST[ 'order_status' ], $reserved_statuses ) ) {

                $ticket_ids = \Tickera\TC_Orders::get_tickets_ids( $post_id );
                foreach ( $ticket_ids as $ticket_id ) {

                    $chart_id = get_post_meta( $ticket_id, 'chart_id', true );
                    $seat_id = get_post_meta( $ticket_id, 'seat_id', true );

                    if ( $chart_id && $seat_id ) {
                        $this->firebase->delete( '/in-cart/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent') );
                        $this->firebase->update( '/reserved/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent') );
                    }
                }
            }

        } elseif (
            'tc_tickets_instances' == get_post_type( $post_id )
            && ( $order_id && in_array( get_post_type( $order_id ), $order_types ) )
        ) {

            $chart_id = get_post_meta( $post_id, 'chart_id', true );
            $seat_id = get_post_meta( $post_id, 'seat_id', true );
            $order_status = apply_filters( 'tc_order_status', get_post_status( $order_id ), $order_id );

            if ( in_array( $order_status, $reserved_statuses ) && $chart_id && $seat_id ) {
                $this->firebase->delete( '/in-cart/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent') );
                $this->firebase->update( '/reserved/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent') );
            }
        }
    }

    /**
     * Remove seats from the firebase database if the order is cancelled or a ticket instances is deleted.
     *
     * @param $post_id
     */
    function tc_remove_firebase_reserved_seats( $post_id ) {

        $order_types = apply_filters( 'tc_order_types', [ 'tc_orders' ] );
        $order_status = apply_filters( 'tc_order_status', get_post_status( $post_id ), $post_id );
        $reserved_statuses = TC_Seat_Chart::get_reserved_order_statuses();

        if ( isset( $_POST['order_status_change'] ) && in_array( get_post_type( $post_id ), $order_types ) ) {

            /**
             * Executes when an order status is changes from the admin.
             */
            $post_status = sanitize_key( $_POST['order_status_change'] );

            if ( in_array( $post_status, ['order_cancelled', 'wc-cancelled', 'order_refunded', 'wc-refunded'] ) ) {

                foreach ( (array) self::get_ticket_instances( $post_id ) as $ticket_instance ) {

                    $metas = get_post_meta( $ticket_instance->ID );
                    $chart_id = isset( $metas['chart_id']) ? $metas['chart_id'][0] : '';
                    $seat_id = isset( $metas['seat_id'] ) ? $metas['seat_id'][0] : '';

                    if ($chart_id && $seat_id) {
                        $this->firebase->delete( '/reserved/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent' ) );
                    }
                }
            }

        } elseif ( in_array( get_post_type( $post_id ), $order_types )
            && ! in_array( $order_status, $reserved_statuses ) ) {

            /**
             * Executes when order status is changed via payment gateways
             */
            foreach ( (array) self::get_ticket_instances( $post_id ) as $ticket_instance ) {

                $metas = get_post_meta( $ticket_instance->ID );
                $chart_id = isset( $metas['chart_id']) ? $metas['chart_id'][0] : '';
                $seat_id = isset( $metas['seat_id']) ? $metas['seat_id'][0] : '';

                if ( $chart_id && $seat_id ) {
                    $this->firebase->delete( '/reserved/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent' ) );
                }
            }

        } elseif ( 'tc_tickets_instances' == get_post_type( $post_id )
            &&  ( did_action( 'trashed_post' ) || did_action( 'deleted_post' ) ) ) {

            /**
             * Executes when an order or ticket is deleted
             */
            $metas = get_post_meta( $post_id );
            $chart_id = isset( $metas['chart_id'] ) ? $metas['chart_id'][0] : '';
            $seat_id = isset( $metas['seat_id'] ) ? $metas['seat_id'][0] : '';

            if ( $chart_id && $seat_id ) {
                $this->firebase->delete( '/reserved/' . $chart_id . '/' . $seat_id, array( 'print' => 'silent' ) );
            }
        }
    }

    /**
     * Collect ticket instances
     *
     * @param $post_id
     * @return int[]|WP_Post[]
     */
    function get_ticket_instances( $post_id ) {

        $id_post_type = ( 'tc_seat_charts' == get_post_type( $post_id ) ) ? 'seating_charts' : 'tc_wc_order';

        $ticket_instances_args = [
            'posts_per_page' => -1,
            'post_type'      => 'tc_tickets_instances',
            'post_status'    => [ 'publish' ],
        ];

        switch ( $id_post_type ) {

            case 'tc_wc_order':
                $ticket_instances_args = array_merge( $ticket_instances_args, [ 'post_parent'    => $post_id ] );
                break;

            case 'seating_charts':
                $ticket_instances_args = array_merge( $ticket_instances_args, [
                    'meta_key'       => 'chart_id',
                    'meta_value'     => $post_id,
                ]);
                break;
        }

        return get_posts( $ticket_instances_args );
    }

    /**
     * Delete whole chart from Firebase
     *
     * @param type $post_id
     */
    function tc_delete_chart_from_firebase( $post_id ) {
        if ( 'tc_seat_charts' == get_post_type( $post_id ) ) {
            $this->firebase->delete( '/in-cart/' . $post_id, array( 'print' => 'silent' ) );
        }
    }

    /**
     * Add in-cart seat to Firebase
     *
     * @global type $tc
     */
    function tc_add_seat_to_firebase_cart() {

        if ( !session_id() ) {
            @session_start();
        }

        if ( isset( $_POST['tc_seat_cart_items'] ) && $_POST['tc_seat_cart_items'] ) {

            $charts = [];
            $tc_seat_cart_items = $_POST['tc_seat_cart_items'];

            foreach ( $tc_seat_cart_items as $tc_seat_cart_item ) {

                $tc_seat_cart_item = explode( '-', $tc_seat_cart_item );
                $chart_id = $tc_seat_cart_item[0];
                $seat_id = $tc_seat_cart_item[1];

                $seat_data =  $this->firebase->get( '/in-cart/' . $chart_id . '/' . $seat_id );
                $seat_data = json_decode( $seat_data, true );

                $charts[$chart_id][$seat_id] = $seat_data
                    ? [ 'timestamp' => $seat_data['timestamp'], 'session_id' => $seat_data['session_id'], 'expires' => $seat_data['expires'] ]
                    : [ 'timestamp' => self::tc_get_current_time(), 'expires' => self::tc_get_expiration(), 'session_id' => session_id() ];
            }

            foreach ( $charts as $chart_id => $data ) {
                $this->firebase->update( '/in-cart/' . $chart_id . '/', $data, array( 'print' => 'silent' ) );
            }

            exit;
        }
    }

    /**
     * Remove expired seats from Firebase Realtime Database.
     * Execute a background process to check for an expired seats. Once a seat is expired, then remove it from firebase.
     */
    function tc_remove_expired_firebase_seats() {

        if ( isset( $_POST['tc_seating_chart_id'] ) && $_POST['tc_seating_chart_id'] ) {

            if ( !session_id() || !isset( $_SESSION ) ) {
                @session_start();
            }

            $chart_id = (int) $_POST['tc_seating_chart_id'];
            $fb_seats_data = json_decode( $this->firebase->get( '/in-cart/' . $chart_id ), true );

            /*
             * Compare current date vs expiration date time.
             * Remove seat that doesn't has expiration.
             * Remove expired seat from firebase.
             */
            foreach ( (array) $fb_seats_data as $fb_seat_id => $fb_seat_value ) {

                if ( !isset( $fb_seat_value['expires'] ) || ( ( self::tc_get_current_time() / 1000 ) >= ( $fb_seat_value['expires'] / 1000 ) ) ) {
                    self::tc_remove_seat_from_firebase_cart( $fb_seat_id, $chart_id );
                    do_action( 'tc_sc_after_expired_seat_removal', $fb_seat_id, $chart_id );
                }
            }
        }
    }

    /**
     * Update Firebase reserved seats data.
     * Initialize a firebase update after loading the seating chart map. Insert reserved seats onto firebase
     */
    function tc_initialize_firebase_reserved_seats() {

        if ( isset( $_POST ) && isset( $_POST['tc_seating_chart_id'] ) && $_POST['tc_seating_chart_id'] ) {

            $chart_id = (int) $_POST['tc_seating_chart_id'];
            $fb_reserved_seats_data = json_decode( $this->firebase->get( '/reserved/' . $chart_id ), true );

            // Collect existing ticket instances as reserved seats
            $reserved_seat_ids = [];
            foreach ( (array) self::get_ticket_instances( $chart_id ) as $ticket_instance ) {

                $order_id = $ticket_instance->post_parent;
                $order_status = apply_filters( 'tc_order_status', get_post_status( $order_id ), $order_id );

                if ( in_array( $order_status, TC_Seat_Chart::get_reserved_order_statuses() ) ) {

                    $seat_id = get_post_meta($ticket_instance->ID, 'seat_id', true);
                    if ($seat_id) {
                        $reserved_seat_ids[] = $seat_id;
                    }
                }
            }
            $reserved_seat_ids = array_values( array_unique( $reserved_seat_ids ) );

            // Restructure Firebase Seats ID for later use
            $_fb_reserved_seats_data = [];
            foreach ( (array) $fb_reserved_seats_data as $fb_seat_id ) {
                $_fb_reserved_seats_data[] = $fb_seat_id;
            }

            // Reinitialize firebase seats. Making sure firebase reserved data is up to date
            foreach ( $reserved_seat_ids as $seat_id ) {

                if ( in_array( $seat_id, $_fb_reserved_seats_data ) ) {
                    $this->firebase->delete( '/reserved/' . $chart_id . '/' . $seat_id, array( 'print' => 'pretty' ) );

                } else {
                    $this->firebase->update( '/reserved/' . $chart_id . '/' . $seat_id, array( 'print' => 'pretty' ) );
                }
            }

            // Remove expired reserved seats
            $fb_reserved_seats_data = json_decode( $this->firebase->get( '/reserved/' . $chart_id ), true );
            foreach ( (array) $fb_reserved_seats_data as $fb_seat_id => $fb_seat_values ) {
                if ( !in_array( $fb_seat_id, $reserved_seat_ids ) ) {
                    $this->firebase->delete( '/reserved/' . $chart_id . '/' . $fb_seat_id, array( 'print' => 'pretty' ) );
                }
            }
        }
    }

    /**
     * Remove seat from Firebase
     *
     * @param bool $seat_id
     * @param bool $chart_id
     */
    function tc_remove_seat_from_firebase_cart( $seat_id = false, $chart_id = false ) {
        $chart_id = $chart_id ? $chart_id : $_POST['chart_id'];
        $seat_id = $seat_id ? $seat_id : $_POST['seat_id'];
        $this->firebase->delete( '/in-cart/' . $chart_id . '/' . $seat_id, array( 'print' => 'pretty' ) );
    }

    /**
     * Delete check-in record from Firebase.
     *
     * @param type $ticket_id
     * @param type $checkins
     */
    function tc_check_in_deleted( $ticket_id, $checkins ) {

        $checkins = ( is_array( $checkins ) && count( $checkins ) > 0 ) ? count( $checkins ) : 0;

        if ( ! $checkins ) {

            // Delete a record from a firebase
            $chart_id = get_post_meta( $ticket_id, 'chart_id', true );

            if ( is_numeric( $chart_id ) ) {

                // Add a record to the firebase
                $seat = get_post_meta( $ticket_id, 'seat_id', true );
                $this->firebase->delete( '/check-ins/' . $chart_id . '/' . $seat . '/', array( 'print' => 'silent' ) );
            }
        }
    }

    /**
     * Add check-in record to Firebase.
     *
     * @param type $ticket_id
     */
    function tc_check_in_added( $ticket_id ) {

        $chart_id = get_post_meta( (int) $ticket_id, 'chart_id', true );

        if ( is_numeric( $chart_id ) ) {

            // Add a record to the firebase
            $seat = get_post_meta( $ticket_id, 'seat_id', true );
            $this->firebase->update('/check-ins/' . $chart_id . '/' . $seat . '/', array('check_in_time' => time()), array('print' => 'silent'));
        }
    }

    /**
     * Generate Expiration Time.
     * Using server time as the basis of seat expiration. This value will be used to compare current time via javascript
     * @var duration +1 = 1 min; default: 5 minutes
     *
     * @return float|int
     */
    function tc_get_expiration() {
        $tc_seat_charts_settings = TC_Seat_Chart::get_settings();
        $duration = ( isset( $tc_seat_charts_settings['expiration'] ) ) ? (int) $tc_seat_charts_settings['expiration'] : 5;
        $duration = apply_filters( 'tc_sc_seat_expiration_time_minutes', $duration );
        $duration = ( $duration < 5 ) ? 5 : $duration;
        $offset = 1000;
        return ( time() + ( $duration * 60 ) ) * $offset;
    }

    /**
     * Generate Server Time.
     * Instead of using Wordpress current_time, server time will be used.
     * This is to compare current and expiry time via javascript.
     *
     * @return float|int
     */
    function tc_get_current_time() {
        $offset = 1000;
        return ( time() * $offset );
    }
}

/**
 * Make sure that admin wants to use Firebase
 */
$tc_seat_charts_settings = TC_Seat_Chart::get_settings();
$use_firebase_integration = isset( $tc_seat_charts_settings['user_firebase_integration'] ) ? $tc_seat_charts_settings['user_firebase_integration'] : false;

if ( $use_firebase_integration ) {
    $tc_firebase = new TC_Firebase();
}
