(function ($) {
  "use strict";

  // Keep legacy widget hooks unchanged for backward compatibility.
  const WIDGET_HOOKS = [
    "frontend/element_ready/timeline-widget-addon.default",
    "frontend/element_ready/twae-post-timeline-widget.default",
    "frontend/element_ready/timeline-process-steps-widget.default",
  ];

  class ProcessStepsClass extends elementorModules.frontend.handlers.Base {
    getDefaultSettings() {
      return {
        selectors: {
          stepTitle: ".cps-process-details",
          stepFooter: ".cps-footer",
          accordion: ".cps-accordion",
          desktopImageWrap: ".cps-show-events .cps-right.cps_desktop_img",
          mobileImageWrap: ".cps-footer .cps-right.cps_mobile_image",
          image: ".cps-img",
        },
        mobileMediaQuery: "(max-width: 433px)",
        slideDurationMs: 1000,
        rotateIntervalMs: 5000,
      };
    }
    onInit() {
      super.onInit();
      this.ns = `processSteps.${this.getID?.() || this.$element?.data("id") || "x"}`;

      this.mediaQueryList = window.matchMedia(this.getDefaultSettings().mobileMediaQuery);

      this.animationTimeoutId = null;
      this.currentIndex = 0;

      this.mount();
    }

    /** Lifecycle: cleanup on destroy */
    onDestroy() {
      this.stopAnimation();
      $(window).off(`resize.${this.ns}`);
      this.$element?.off(`click.${this.ns}`);
    }

    /* --------------------------- Mount / Remount --------------------------- */

    /** First-time mount: cache, bind, paint, start animation */
    mount() {
      this.cacheDom();
      this.bindEvents();
      this.updateImageVisibility(this.currentIndex);
      this.startAnimation(this.currentIndex);
    }

    /* ------------------------------- Caching ------------------------------- */

    /** Cache frequently used DOM collections (refresh after editor re-render) */
    cacheDom() {
      const sel = this.getDefaultSettings().selectors;

      // All accordion bodies
      this.$stepFooters = this.$element.find(sel.stepFooter);

      // Choose correct image set for current viewport
      const $desktopWrap = this.$element.find(sel.desktopImageWrap);
      const $mobileWrap = this.$element.find(sel.mobileImageWrap);
      const useMobile = this.mediaQueryList.matches;

      this.$images = (useMobile ? $mobileWrap : $desktopWrap).find(sel.image);
    }

    /* ------------------------------- Binding ------------------------------- */

    /** Bind namespaced, delegated listeners (safe for dynamic editor changes) */
    bindEvents() {
      const sel = this.getDefaultSettings().selectors;

      // Resize: re-cache and repaint images (namespaced; cleaned on destroy)
      $(window)
        .off(`resize.${this.ns}`)
        .on(`resize.${this.ns}`, () => {
          this.cacheDom();
          this.updateImageVisibility(this.currentIndex);
        });

      // Delegated click: works for existing and newly added repeater items
      this.$element
        .off(`click.${this.ns}`, sel.stepTitle)
        .on(`click.${this.ns}`, sel.stepTitle, (evt) => {
          const $allTitles = this.$element.find(sel.stepTitle);
          const clickedIndex = $allTitles.index(evt.currentTarget);
          if (clickedIndex >= 0) {
            this.stopAnimation();
            this.showStep(clickedIndex);
            this.startAnimation(clickedIndex);
          }
        });
    }

    /* ----------------------------- Animation ------------------------------ */

    /** Stop auto-rotation (clears timeout safely) */
    stopAnimation() {
      if (this.animationTimeoutId) {
        clearTimeout(this.animationTimeoutId);
        this.animationTimeoutId = null;
      }
    }

    /**
     * Start/continue the auto-rotation from a given index.
     * Re-caches at start to always work on the latest editor DOM.
     */
    startAnimation(startIndex = 0) {
      const { rotateIntervalMs } = this.getDefaultSettings();

      // Ensure we operate on fresh nodes (important in editor)
      this.cacheDom();

      const totalSteps = this.$stepFooters.length;
      if (!totalSteps) return;

      let index = Math.max(0, Math.min(startIndex, totalSteps - 1));
      this.showStep(index);

      this.animationTimeoutId = setTimeout(() => {
        index = (index + 1) % totalSteps;
        this.showStep(index);
        this.startAnimation(index);
      }, rotateIntervalMs);
    }

    /* ------------------------------- UI Logic ------------------------------ */

    /** Show only the image at the given index */
    updateImageVisibility(index = 0) {
      if (!this.$images || !this.$images.length) return;

      this.$images.css("display", "none");
      const $img = this.$images.eq(index);
      if ($img && $img.length) {
        $img.css("display", "block");
      }
    }

    /**
     * Display the step at `index` and hide the others:
     * - Slide down the active footer; slide up the rest
     * - Toggle .cps-selected / .active on accordion wrappers
     * - Update the corresponding image
     */
    showStep(index) {
        const sel = this.getDefaultSettings().selectors;
        const { slideDurationMs } = this.getDefaultSettings();

        this.currentIndex = index;
        this.updateImageVisibility(index);
        this.$stepFooters.removeClass('animate');
        this.$stepFooters.each((i, footerEl) => {
            const $footer = jQuery(footerEl);
            const $accordion = $footer.closest(sel.accordion);
            const dateEl = footerEl.parentElement && footerEl.parentElement.previousElementSibling;

            if (i === index) {
                setTimeout(() => $footer.addClass('animate'), 0);

                $accordion.addClass('cps-selected');
                $footer.stop(true, false).slideDown(slideDurationMs, () => {
                    $accordion.addClass('active');
                });
                if (dateEl) dateEl.classList.add('cps-selected');
            } else {
                $accordion.removeClass('cps-selected active');
                $footer.stop(true, false).slideUp(slideDurationMs);
                if (dateEl) dateEl.classList.remove('cps-selected');
            }
        });
    }

  }

  /* ---------------------- Register handler for widgets --------------------- */

  $(window).on("elementor/frontend/init", () => {
    const addHandler = ($scope) => {
      elementorFrontend.elementsHandler.addHandler(ProcessStepsClass, { $element: $scope });
    };
    WIDGET_HOOKS.forEach((hook) => {
      elementorFrontend.hooks.addAction(hook, addHandler);
    });
  });
})(jQuery);
