<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster Pro DailyMotion support
 * https://developer.dailymotion.com/
 */
if ( ! class_exists( 'Video_Blogster_DailyMotion' )) {
    class Video_Blogster_DailyMotion {

	private $vbp = null;			// will point to main Video_Blogster instance
        private $query_fields = array();        // options for current query

	private $APIs = array(
		'channel.data'		=> 'https://api.dailymotion.com/channel/',
		'group.data'		=> 'https://api.dailymotion.com/group/',
		'playlist.data'		=> 'https://api.dailymotion.com/playlist/',
		'user.data'		=> 'https://api.dailymotion.com/user/',
		'videos.list'		=> 'https://api.dailymotion.com/videos',
		'video.data'		=> 'https://api.dailymotion.com/video/'
		);

	private $batch_limit = 100;		// max amount we can request per query

	private $total = 0;			// number of total results in query
        private $num_skipped = 0;               // a sum of all videos skipped for this request
        private $num_updated = 0;               // a sum of all videos updated for this request
        private $num_imported = 0;              // a sum of all videos imported for this request

	/**
	 * Create DailyMotion video source
	 * Point back to Video Blogster Pro object to use common functions
	 * Save the query fields for easy access
	 */
	public function __construct( $vbp, $query_fields ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

	/**
	 * Shortcut to proper API url
	 */
	private function getApi( $type ) {
		return $this->APIs[$type];
	}

	/**
	 * Make the query and check for errors.
	 */
	private function queryApi( $url ) {
		$this->vbp->info_message( sprintf( '%s : %s%s%s', 
			__FUNCTION__, 
			'<a target="_blank" href="' . esc_url( $url ) . '">',
			esc_url( $url ),
			'</a>'
			), 'notice notice-warning', 'debug' );


		$response = wp_remote_get( $url );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $feedID, $response->get_error_code(), __FUNCTION__, esc_url( $url ), $response->get_error_message() ) );
		}

		$data = json_decode( wp_remote_retrieve_body( $response ) );

		if ( isset( $data->error ) ) {
			$this->vbp->info_message( sprintf( esc_html__( '%s %s(%s) - DailyMotion API returned %s: %s', 'video-blogster' ), $feedID, __FUNCTION__, esc_url( $url ), $data->error->code, $data->error->message ) );
		}

		if ( ! $this->total ) {
			$this->total = isset( $data->total ) ? $data->total : 0;
			if ( $this->total ) {
				$this->vbp->info_message( sprintf( esc_html__( '%s DailyMotion API returned %s total results.', 'video-blogster' ), $feedID, $this->total ), 'updated', 'video_import' );
			}
		}

		return $data;
	}

	private function get_best_thumbnail( $video ) {
		$img = array();
		$match = false;

		if ( ! $match && isset( $video->thumbnail_60_url ) ) $img[] = (string) $video->thumbnail_60_url;
		if ( $this->query_fields['qExtraParams2'] == '60px' ) $match = true;

		if ( ! $match && isset( $video->thumbnail_120_url ) ) $img[] = (string) $video->thumbnail_120_url;
		if ( $this->query_fields['qExtraParams2'] == '120px' ) $match = true;

		if ( ! $match  && isset( $video->thumbnail_180_url ) ) $img[] = (string) $video->thumbnail_180_url;
		if ( $this->query_fields['qExtraParams2'] == '180px' ) $match = true;

		if ( ! $match && isset( $video->thumbnail_240_url ) ) $img[] = (string) $video->thumbnail_240_url;
		if ( $this->query_fields['qExtraParams2'] == '240px' ) $match = true;

		if ( ! $match && isset( $video->thumbnail_360_url ) ) $img[] = (string) $video->thumbnail_360_url;
		if ( $this->query_fields['qExtraParams2'] == '360px' ) $match = true;

		if ( ! $match && isset( $video->thumbnail_480_url ) ) $img[] = (string) $video->thumbnail_480_url;
		if ( $this->query_fields['qExtraParams2'] == '480px' ) $match = true;

		if ( ! $match && isset( $video->thumbnail_720_url ) ) $img[] = (string) $video->thumbnail_720_url;
		if ( $this->query_fields['qExtraParams2'] == '720px' ) $match = true;

		// catchall
		if ( isset( $video->thumbnail_url ) ) $img[] = (string) $video->thumbnail_url;

		$img = array_reverse( $img );

		$image = $this->vbp->verify_thumbnails( $img );

		return $image;
	}

	/**
	 * Extract video details into our generic videoInfo array
	 */
	private function get_video_info( $video ) {
		$videoInfo = array();

		$videoInfo['videoSource'] = 'DailyMotion';
		$videoInfo['videoID'] = isset( $video->id ) ? (string) $video->id : '';
		$videoInfo['orig_title'] = $videoInfo['title'] = isset( $video->title ) ? (string) $video->title : '';

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}

		$videoInfo['url'] = isset( $video->embed_url ) ? $video->embed_url : '';
		$videoInfo['videoEmbed'] = isset( $video->embed_html ) ? $video->embed_html : '';
		
		$videoInfo['orig_desc'] = $videoInfo['desc'] = isset( $video->description ) ? (string) $video->description : '';
		$videoInfo['association'] = isset( $video->{"owner.username"}) ? (string) $video->{"owner.username"} : '';
		$videoInfo['authorUrl'] = isset( $video->{"owner.username"}) ? 'https://www.dailymotion.com/' . (string) $video->{"owner.username"} : '';
		$videoInfo['authorTitle'] = isset( $video->{"owner.username"}) ? (string) $video->{"owner.username"} : '';
		$videoInfo['img'] = $this->get_best_thumbnail( $video );

		$videoInfo['allow_embed'] = isset( $video->allow_embed ) ? $video->allow_embed : true;

		$publishedAt = isset( $video->updated_time ) ? (string) $video->updated_time : current_time( 'Y-m-d H:i:s' );
		// video time is UTC - convert to date string
		$publishedAt = date( 'Y-m-d H:i:s', $publishedAt );
		$date = $this->vbp->getDateTime( $publishedAt, $videoInfo['url'] );
		$videoInfo['publishedAt'] = isset( $date ) ? $date->format( 'Y-m-d H:i:s' ) : $publishedAt;
		$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

		$videoInfo['duration'] = isset( $video->duration_formatted ) ? $video->duration_formatted : 0;

		$videoInfo['sourceCategory'] = $videoInfo['tags'] = '';
		if ( isset( $video->channel ) ) {
			$videoInfo['sourceCategoryName'] = (string) $video->channel;
		}
		if ( isset( $video->tags ) ) {
			$videoInfo['tags'] = implode( ',', $video->tags );
		}

		$videoInfo['viewCount'] = isset( $video->views_total) ? (string) $video->views_total : 0;
		//convert DailyMotion ratings to likes/dislikes
		$rating = isset( $video->rating ) ? (string) $video->rating : 0;
		$ratings_total = isset( $video->ratings_total ) ? (string) $video->ratings_total : 0;
		$videoInfo['likeCount'] = round( $ratings_total * ( $rating * 0.2 ), 0) ; // 1-5 rating system
		$videoInfo['dislikeCount'] = $ratings_total - $videoInfo['likeCount'];

		$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
		$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $video );

		$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );

		return $videoInfo;
	}

	/**
	 * Do a batch query search for up to $this->batch_limit comments at a time (DailyMotion limit)
	 */
	private function grab_comments_batch( $url, $postID ) {
		$this->vbp->info_message(__FUNCTION__ . ': ' . htmlentities( $url ), 'notice notice-warning', 'debug' );
        	$response = $this->queryApi( $url );
		$total=0;
		$comment_date = current_time( 'Y-m-d H:i:s' );
		foreach ( $response->list as $comment ) {
			$author = isset( $comment->{"owner.screenname"} ) ? trim( $comment->{"owner.screenname"} ) : '';
			$content = isset( $comment->message ) ? trim( $comment->message ) : '';
			if ( empty( $author ) || empty( $content ) ) {
				continue;
			}
			if ( TRUE == $this->query_fields['cUsePublishedDate'] ) {
				$commentAt = isset( $comment->created_time ) ? $comment->created_time : current_time( 'Y-m-d H:i:s' );
				$commentDate = $this->vbp->getDateTime( $commentAt, $url );
				$comment_date = isset( $commentDate ) ? $commentDate->format('Y-m-d H:i:s') : $commentAt;
			}

			$commentdata = array(
				'comment_post_ID'	=> $postID,
				'comment_author'	=> $author,
				'comment_content'	=> $content,
				'comment_date'     	=> $comment_date,
				'comment_approved'	=> 1
			);
			$commentdata = apply_filters( 'vbp_dailymotion_comment', $commentdata );
			$comment_id = $this->vbp->save_the_comment( $commentdata, $this->query_fields );
			if ( $comment_id ) {
				$total++;
			}
		}
		$this->vbp->info_message( sprintf( esc_html__( '%s comments imported successfully to post id %s.', 'video-blogster' ), $total, $postID ), 'updated', 'video_import' );

		return isset( $response->has_more ) ? $response->has_more : 0;
	}

	/**
	 * Grab and save comments for a video
	 * Toggle comment approval and email notification if set
	 * Break up process into batches
	 */
	private function grab_comments( $videoID, $postID ) {
		
		$cm = $cn = false;
		$this->vbp->bulk_comment_start( $cm, $cn);

		$totalComments = $this->query_fields['qNumComments'];
        	$query_args = array(
			'fields'	=> 'owner.screenname,message,created_time',
                	'limit'		=> $totalComments > $this->batch_limit || $totalComments < 0 ? $this->batch_limit : $totalComments,
                	'page'		=> '1'
        	);

		$query_args = apply_filters( 'vbp_dailymotion_comments_query', $query_args, $this->query_fields );

		while ( $totalComments ) {
			$url = $this->getApi( 'video.data' ) . $videoID . '/comments' . '?' . http_build_query( $query_args );
			if ( ! $this->grab_comments_batch( $url, $postID ) ) {
				break;	// no more comments
			}
			$totalComments -= $query_args['limit'];
                	$query_args['limit'] = $totalComments > $this->batch_limit || $totalComments < 0 ? $this->batch_limit : $totalComments;
                	$query_args['page']++;
		}

		$this->vbp->bulk_comment_end( $cm, $cn );
	}

	/**
	 * Takes an array of video details to process and create posts
	 */
	private function save_videos( $videoDetails ) {
                if ( ! $videoDetails ) {
			return 0;
                }
		foreach ( $videoDetails as $video ) {

			$videoInfo = $this->get_video_info( $video );
			if ( ! $videoInfo ) continue;
			$this->vbp->info_message( sprintf( esc_html__( 'Checking Daily Motion video: [%s] at %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'] ), 'notice notice-warning', 'debug' );

			// DailyMotion specific:
			if ( isset( $videoInfo['allow_embed'] ) && $videoInfo['allow_embed'] === false ) {
				$this->vbp->info_message( esc_html__( 'Daily Motion video flag allow_embed is set to false. Skipping.', 'video-blogster' ),  'notice notice-warning', 'video_skip' );
				$this->num_skipped++;
				continue;
			}

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID ) { $this->num_skipped++; continue; }

			if ( ! empty( $this->query_fields['qNumComments'] ) ) {
				$this->grab_comments( $videoInfo['videoID'], $postID );
			}

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

                        if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
                        else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		}
		$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
		if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		return 1;
	}

	/**
	 * Grab a single video by the video ID stored in qAssoc.
	 */
	private function grab_videos_by_ids() {
		$fields = 'id,title,channel,thumbnail_60_url,thumbnail_120_url,thumbnail_180_url,thumbnail_240_url,thumbnail_360_url,thumbnail_480_url,thumbnail_720_url,thumbnail_url,allow_embed,updated_time,description,duration_formatted,embed_url,embed_html,rating,ratings_total,status,tags,views_total,owner.username';
		$query_args = array(
			'fields'		=> $fields
		);

		$ids = explode( PHP_EOL, $this->query_fields['qAssoc'] );
		foreach ( $ids as $id ) {
			$url = $this->getApi( 'video.data' ) . $id . '?' . http_build_query( $query_args );
			$response = $this->queryApi( $url );
			if ( ! isset( $response->error ) ) {
				$this->save_videos( array( $response ) );
			}
		}
		$this->vbp->import_finished( 'DailyMotion videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
	}


	/**
	 * Make a batch request for videos
	 */
	private function grab_videos_batch( $url ) {
		$response = $this->queryApi( $url );
                if ( $this->total == 0 && empty( $response->list ) ) {
			$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] : "";
			return $this->vbp->info_message( sprintf( esc_html__( '%s No results for DailyMotion query (%s)', 'video-blogster' ), $feedID, htmlentities( $url ) ), 'notice notice-warning', 'critical' );
                }

		$list = isset( $response->list ) ? $response->list : null;
		if ( ! empty( $list ) ) {
			$this->vbp->info_message( sprintf( esc_html__( 'DailyMotion response data (%s total results): %s', 'video-blogster' ), $this->total, htmlentities( print_r( $list,true ) ) ), 'notice notice-warning', 'debug' );
			if ( ! $this->save_videos( $list ) ) return 0;
		}
		return $response->has_more;
	}

	/**
	 * Process the video query and break down into batches
	 * http://developer.dailymotion.com/tools
	 */
	public function grab_videos() {
		$keyphrase = $this->query_fields['qKeyphrase'];
		$assoc = $this->query_fields['qAssoc'];
		$assocType = $this->query_fields['qAssocType'];
		$base_url = $this->getApi( 'videos.list' );
		$filters = ! empty( $this->query_fields['qExtraParams'] ) ? "&" . ltrim( $this->query_fields['qExtraParams'], "&" ) : '';

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
                if ( empty( $keyphrase ) && empty( $assoc ) && empty( $assocType ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: No DailyMotion keyphrase or association not set properly.', 'video-blogster' ), $feed ), 'error', 'critical' );
                }

		$fields = 'id,title,channel,thumbnail_60_url,thumbnail_120_url,thumbnail_180_url,thumbnail_240_url,thumbnail_360_url,thumbnail_480_url,thumbnail_720_url,thumbnail_url,allow_embed,updated_time,description,duration_formatted,embed_url,embed_html,rating,ratings_total,status,tags,views_total,owner.username';

		$totalVids = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;

		$query_args = array(
			'sort'			=> $this->query_fields['qOrderBy'],
			'limit'			=> $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids,
			'page'			=> '1',
			'fields'		=> $fields
		);

		if ( $keyphrase ) {
			$query_args['search'] = $keyphrase;
		}

		$not_supported = array();
		if ( $assoc && $assocType == 'channel' ) {
			$query_args['channel'] = $assoc;
			$not_supported = array( 'country', 'language' );
		}
		else if ( $assoc && $assocType == 'playlist' ) {
			$base_url = $this->getApi( 'playlist.data' ) . $assoc . '/videos';
			$not_supported = array( 'country', 'language', 'created_after', 'created_before' );
		}
		else if ( $assoc && $assocType == 'user' ) {
			$base_url = $this->getApi( 'user.data' ) . $assoc . '/videos';
			$not_supported = array( 'country', 'language' );
		}
		else if ( $assoc && $assocType == 'video' ) {
			return $this->grab_videos_by_ids();
		}

		if ( ! in_array( 'country', $not_supported ) && $this->query_fields['qRegionCode'] ) {
			$query_args['country'] = $this->query_fields['qRegionCode'];
		}
		if ( ! in_array( 'language', $not_supported ) && $this->query_fields['qLanguage'] ) {
			$query_args['languages'] = $this->query_fields['qLanguage'];
		}

		if ( ! in_array( 'country', $not_supported ) && $this->query_fields['qPublishedAfter'] ) {
			$date = $this->vbp->getDateTime( $this->query_fields['qPublishedAfter'], 'API query', false );
			if ( $date ) {
				$query_args['created_after'] = $date->format('U');
			}
			else {
				$this->vbp->info_message( sprintf( esc_html__( '%s Error: publishedAfter date %s not recognized.', 'video-blogster' ), $feedID, $query_args['publishedAfter'] ) );
			}
		}
		if ( ! in_array( 'country', $not_supported ) && $this->query_fields['qPublishedBefore'] ) {
			$date = $this->vbp->getDateTime( $this->query_fields['qPublishedBefore'], 'API query', false );
			if ( $date ) {
				$query_args['created_before'] = $date->format( 'U' );
			}
			else {
				$this->vbp->info_message( sprintf( esc_html__( '%s Error: publishedBefore date %s not recognized.', 'video-blogster' ), $feedID, $query_args['publishedBefore'] ) );
			}
		}

		$query_args = apply_filters( 'vbp_dailymotion_search_query', $query_args, $this->query_fields );

		while ( $totalVids ) {
			$url = $base_url . '?' . http_build_query( $query_args );
			if ( $filters ) $url .= $filters;
			if ( ! $this->grab_videos_batch( $url ) ) {
				break; // reached limit
			}
			if ( $this->query_fields['qQueryBehavior'] == 'strict' && $totalVids > 0 ) {
				$totalVids -= $query_args['limit'];
                                if ( $totalVids <= 0 ) break;   // failsafe for strict behavior
			}
			$query_argsp['limit'] = $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids;
			$query_args['page']++;
		}
		$this->vbp->import_finished( 'DailyMotion videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
		return $query_args['page'];
	}


	/**
	 * Checks a batch of video ids to see if they're been deleted from DailyMotion
	 * Query result list without id means that video has been deleted.
	 */
	private function check_for_deleted_batch( $videoIDs ) {
		if ( empty( $videoIDs ) ) {
			return 0;
		}

		foreach ( $videoIDs as $key => $value ) {
			$list[] = $value;
		}
		$commaList = implode( ',', $list );

		$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: checking DailyMotion video IDs: %s ', 'video-blogster' ), $commaList ), 'updated', 'utility_funcs' );

        	$query_args = array(
			'fields'	=> 'id,status',
                	'ids'		=> $commaList,
			'limit'		=> $this->batch_limit
        	);
		$url = $this->getApi( 'videos.list' ) . '?' . http_build_query( $query_args );
		$data = $this->queryApi( $url );

		if ( ! $data ) { // bad error occurred. Try again another time
			return $this->vbp->info_message( esc_html__( 'Query Error: Skipping this batch.', 'video-blogster' ) );
		}
		if ( isset( $data->error ) ) { // error occurred - probably timeout. Try again another time
			return $this->vbp->info_message( sprintf( esc_html__( 'DailyMotion error: %s ', 'video-blogster' ), $data->error ) );
		}

		$videoDetails = isset( $data->list ) ? $data->list: array();

		$validIDs = array();
		// remove each ID found from query from our original list of IDs to check...
		foreach ( $videoDetails as $video )  {
			if ( ( $key = array_search( $video->id, $videoIDs ) ) !== FALSE && ( $video->status == 'ready' || $video->status == 'published' ) ) {
				unset( $videoIDs[$key] );
			}
		}

		// any remaining IDs in our list means they weren't found on DailyMotion
		foreach ( $videoIDs as $key => $value ) {
			$key = apply_filters( 'vbp_delete_video', $key );
			if ( ! $key ) {
				continue;
			}
			wp_trash_post( $key );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: DailyMotion video %s not found. Sending post %s to trash.', 'video-blogster' ), $value, $key ), 'updated', 'critical' );
		}
		return 0;
	}

	/**
	 * Creates batches of DailyMotion ids from save posts to check
	 */
	public function check_for_deleted_videos() {
		// get comma delimited videoIDs from post metadata
		$getMore = true;
		$offset = 0;
		$chunksize = apply_filters( 'vbp_posts_chunksize', 1000 );

		while ( $getMore ) {
		$posts = $this->vbp->get_posts_by_site( 'DailyMotion', $offset );
		if ( ! $posts ) return;

		$num_results = count( $posts );

		if ( $num_results && $offset == 0 ) {
			$this->vbp->info_message( sprintf( esc_html__( 'DailyMotion %s Start', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
		}

		$num=0;
		foreach ( $posts as $post_id ) {
			if ( ! $num ) {
				$videoIDs = array();
			}
			$videoID = get_post_meta( $post_id, 'VideoID', TRUE );
			if ( ! $videoID ) {	// mvb compatability check:
				$videoID = get_post_meta( $post_id, 'mvb_vid_code', TRUE );
			}
			if ( ! $videoID ) {
				continue; 
			}
			$videoIDs[$post_id] = $videoID;
			if ( ++$num >= $this->batch_limit ) {
				$num = $this->check_for_deleted_batch( $videoIDs );
			}
		} // end foreach posts
			if ( $num_results < $chunksize ) break;
			$offset += $chunksize;
			$query_args['offset'] = $offset;
		} // end getMore
		if ( $num ) {
			$this->check_for_deleted_batch( $videoIDs );	// still some left
		}
		if ( $num_results || $offset ) $this->vbp->info_message( sprintf( esc_html__( 'DailyMotion %s End', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
	}

    } // END class Video_Blogster_DailyMotion
} // END if ( ! class_exists( 'Video_Blogster_DailyMotion' ) )


// Fix for DailyMotion returning http instead of https
add_filter( 'get_video_info', 'vbp_dm_verify_https' );
function vbp_dm_verify_https( $videoInfo ) {
  if ( $videoInfo['videoSource'] == 'DailyMotion' && is_ssl() ) {
	$videoInfo['videoEmbed'] = str_replace( 'http:', 'https:', $videoInfo['videoEmbed'] );
  }
  return $videoInfo;
}
?>
