<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster Pro GiantBomb support
 * http://www.giantbomb.com/api/
 */
if ( ! class_exists( 'Video_Blogster_GiantBomb' )) {
    class Video_Blogster_GiantBomb {

	private $vbp = null;			// will point to main Video_Blogster instance
        private $query_fields = array();        // options for current query

        protected $apiKey;
        
	private $APIs = array(
		'search'		=> 'http://www.giantbomb.com/api/search',
		'video'			=> 'http://www.giantbomb.com/api/video',
		'videos'		=> 'http://www.giantbomb.com/api/videos',
		'video_types'		=> 'http://www.giantbomb.com/api/video_types',
		);

	private $batch_limit = 100;             // max amount we can request per query 

	private $total = 0;			// total number of results
        private $num_skipped= 0;                // a sum of all videos skipped for this request
        private $num_updated = 0;               // a sum of all videos updated for this request
        private $num_imported = 0;              // a sum of all videos imported for this request

	/**
	 * Create GiantBomb video source
	 * Point back to Video Blogster Pro object to use common functions
	 * Save the query fields for easy access
	 */
	public function __construct( $vbp, $query_fields, $apiKey ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->apiKey = $apiKey;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

	/**
	 * Shortcut to proper API url
	 */
	private function getApi( $type ) {
		return $this->APIs[$type];
	}

	/**
	 * Make the query and check for errors.
	 */
	private function queryApi( $url ) {
		$this->vbp->info_message( sprintf( '%s : %s%s%s', 
			__FUNCTION__, 
			'<a target="_blank" href="' . esc_url( $url ) . '">',
			esc_url( $url ),
			'</a>'
			), 'notice notice-warning', 'debug' );


                $args = array( 
                        'headers' => array( 'User-Agent'    => 'video blogster pro; (http://www.videoblogsterpro.com/)' ) 
                );

		$response = wp_remote_get( $url, $args );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $feedID, $response->get_error_code(), __FUNCTION__, $url, $response->get_error_message() ) );
		}
		$data = new SimpleXMLElement( $response['body'],  LIBXML_NOCDATA | LIBXML_COMPACT | LIBXML_PARSEHUGE );

		if ( ! $this->total ) {
			$this->total = isset( $data->number_of_total_results ) ? $data->number_of_total_results : 0;
			if ( $this->total ) {
				$this->vbp->info_message( sprintf( esc_html__( '%s GiantBomb API returned %s total results.', 'video-blogster' ), $feedID, $this->total ), 'updated', 'video_import' );
			}
		}

		return $data;
	}

	/**
	 * Extract video details into our generic videoInfo array
	 */
	private function get_video_info( $video ) {
		$videoInfo = array();

		$videoInfo['videoSource'] = 'GiantBomb';
		$videoInfo['videoID'] = isset( $video->id ) ? (string) $video->id : '';
		$videoInfo['orig_title'] = $videoInfo['title'] = isset( $video->name ) ? (string) $video->name: '';

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}

		$videoInfo['url'] = isset( $video->site_detail_url ) ? (string) $video->site_detail_url : '';
		$videoInfo['orig_desc'] = $videoInfo['desc'] = isset( $video->deck ) ? (string) $video->deck : '';
		$videoInfo['association'] = isset( $video->user ) ? (string) $video->user : '';
		$videoInfo['tags'] = (string) $videoInfo['association'];
		$videoInfo['img'] = $this->get_best_thumbnail( $video );

		$videoInfo['videoEmbed'] = isset( $video->embed_player ) ? "<iframe width='800' height='600' src='" . (string) $video->embed_player . "' frameborder='0' scrolling='no' allowfullscreen></iframe>" : '';


		$publishedAt = isset( $video->publish_date ) ? (string) $video->publish_date : current_time( 'Y-m-d H:i:s' );
		$date = $this->vbp->getDateTime( $publishedAt, $videoInfo['url'] );
		$videoInfo['publishedAt'] = isset( $date ) ? $date->format( 'Y-m-d H:i:s' ) : $publishedAt;
		$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

		$d = isset( $video->length_seconds ) ? (int) $video->length_seconds : 0;
		$videoInfo['duration'] = '';
		if ( $d >= 3600 ) {
			$videoInfo['duration'] .= sprintf( '%02d:', ($d/3600) );
		}
		$videoInfo['duration'] .= sprintf('%02d:%02d', ($d/60%60), $d%60);

		$videoInfo['sourceCategory'] = '';
		$cat = isset( $video->video_type ) ? (string) $video->video_type : '';
		$videoInfo['sourceCategoryName'] = $cat;

		$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
		$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $video );

		$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );

		return $videoInfo;
	}

	/**
	 * Takes an array of video details to process and create posts
	 */
	private function save_videos( $videoDetails ) {
                if ( ! $videoDetails ) {
			return 0;
                }
		foreach ( $videoDetails->video as $video ) {

			$videoInfo = $this->get_video_info( $video );
			if ( ! $videoInfo ) continue;
			$this->vbp->info_message( sprintf( esc_html__( 'Checking GiantBomb video: [%s] at %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'] ), 'notice notice-warning', 'debug' );

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID ) { $this->num_skipped++; continue; }

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

                        if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
                        else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		}
		$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
		if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		return 1;
	}

	/**
	 * Make a batch request for videos
	 */
	private function grab_videos_batch( $url ) {
		$response = $this->queryApi( $url );
		$results = isset( $response->results ) ? $response->results : null;
		if ( ! empty( $results ) ) {
			$this->vbp->info_message( sprintf( esc_html__( 'GiantBomb response data (%s total results): %s', 'video-blogster' ), $this->total, htmlentities( print_r( $results,true ) ) ), 'notice notice-warning', 'debug' );
			if ( ! $this->save_videos( $results ) ) return 0;
		}
		else if ( $response ) {
			return $this->vbp->info_message( esc_html__( 'No results returned for given query.', 'video-blogster' ), 'notice notice-warning', 'critical' );
		}
		return 1;
	}

	/**
	 * Process the video query and break down into batches
	 */
	public function grab_videos() {

		$totalVids = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;
		$keyphrase = $this->query_fields['qKeyphrase'];
		$cat = $this->query_fields['qCategory'];
		$sort = $this->query_fields['qOrderBy'];

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( empty( $this->apiKey ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: no API Key found for Giant Bomb in Main Settings.', 'video-blogster' ), $feedID ), 'error', 'critical' );
		}

		if ( ! empty( $keyphrase ) && false === strpos( $keyphrase, ':' ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: Giant Bomb filters should be in name:value format.', 'video-blogster' ), $feedID ), 'error', 'critical' );
		}
		if ( false === strpos( $sort, ':' ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s Error: Giant Bomb sort results should be in field:direction format.', 'video-blogster' ), $feedID ), 'error', 'critical' );
		}

		$keyphrase = str_replace( 'filter=', '', $keyphrase ); // just in case user added filter=

		$query_args = array(
			'filter'		=> $keyphrase,
			'offset'		=> 0,
			'limit'			=> $totalVids > $this->batch_limit || $totalVids < 0 ? $this->batch_limit : $totalVids,
			'sort'			=> $sort,
			'subscriber_only'	=> 0,
			'field_list'		=> 'deck,embed_player,id,image,length_seconds,name,publish_date,site_detail_url,url,user,video_type',
			'api_key'		=> $this->apiKey,
		);
		if ( ! empty( $cat ) ) $query_args['video_type'] = $cat;

		$query_args = apply_filters( 'vbp_giantbomb_search_query', $query_args, $this->query_fields );

		while ( $totalVids ) {
			$url = $this->getApi( 'videos' ) . '?' . http_build_query( $query_args );
			if ( ! $this->grab_videos_batch( $url ) ) {
				break;
			}
			if ( $this->query_fields['qQueryBehavior'] == 'strict' && $totalVids > 0 ) {
				$totalVids -= $query_args['limit'];
				if ( $totalVids <= 0 ) break;   // failsafe for strict behavior
			}
			$query_args['offset'] += $query_args['limit'];
			$query_argsp['limit'] = $totalVids > $this->batch_limit ? $this->batch_limit : $totalVids;
		}
		$this->vbp->import_finished( 'GiantBomb videos', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
		return $query_args['offset'];
	}

	/*
	 * Get GiantBomb video types aka categories
	 */
	public function grab_categories() {
		if ( empty( $this->apiKey ) ) {
                        return $this->vbp->info_message( esc_html__( 'Error: no API Key found for Giant Bomb in Main Settings.', 'video-blogster' ), 'error', 'critical' );
		}

		$query_args = array(
			'field_list'		=> 'id,name',
			'api_key'		=> $this->apiKey,
		);
		$url = $this->getApi( 'video_types' ) . '?' . http_build_query( $query_args );
		$response = $this->queryApi( $url );
		if ( empty( $response ) ) return array();
		return $response;
	}

	/**
	 * Checks video ids to see if they're been deleted from GiantBomb
	 * No batch checking allowed - have to query each video individually :(
	 */
	public function check_for_deleted_videos() {

		$getMore = true;
		$offset = 0;
		$chunksize = apply_filters( 'vbp_posts_chunksize', 1000 );

		while ( $getMore ) {
		$posts = $this->vbp->get_posts_by_site( 'GiantBomb', $offset );
		if ( ! $posts ) return;

		$num_results = count( $posts );

		if ( $num_results && $offset == 0 ) {
			$this->vbp->info_message( sprintf( esc_html__( 'GiantBomb %s Start', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
		}

		foreach ( $posts as $post_id ) {
			$videoID = get_post_meta( $post_id, 'VideoID', TRUE );
			if ( ! $videoID ) {
				continue;
			}
			$url = get_post_meta( $post_id, 'VideoUrl', TRUE );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: checking GiantBomb video ID %s, url: %s ', 'video-blogster' ), $videoID, $url ), 'updated', 'utility_funcs' );
			$query_args = array(
				'field_list'		=> 'id,name',
				'filter'		=> 'id:' . $videoID,
				'api_key'		=> $this->apiKey,
				);
			$url = $this->getApi( 'videos' ) . '?' . http_build_query( $query_args );
			$response = $this->queryApi( $url );
			if ( $response->number_of_total_results == 0 ) {
				$postID = apply_filters( 'vbp_delete_video', $post_id );
				if ( ! $postID ) {
					continue;
				}
				wp_trash_post( $postID );
				$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: GiantBomb video %s not found. Sending post %s to trash.', 'video-blogster' ), $videoID, $postID ), 'updated', 'critical' );
			}
		} // end foreach posts
			if ( $num_results < $chunksize ) break;
			$offset += $chunksize;
			$query_args['offset'] = $offset;
		} // end getMore
		if ( $num_results || $offset ) $this->vbp->info_message( sprintf( esc_html__( 'GiantBomb %s End', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
	}

	/**
	 * Gets best thumbnail for a video request
	 */
	private function get_best_thumbnail ( $video ) {
		$img = array();
		$match = false;

		if ( ! $match && isset( $video->image->small_url ) ) $img[] = (string) $video->image->small_url;
		if ( $this->query_fields['qExtraParams2'] == 'small' ) $match = true;

		if ( ! $match && isset( $video->image->medium_url ) ) $img[] = (string) $video->image->medium_url;
		if ( $this->query_fields['qExtraParams2'] == 'medium' ) $match = true;

		if ( ! $match && isset( $video->image->super_url ) ) $img[] = (string) $video->image->super_url;
		if ( $this->query_fields['qExtraParams2'] == 'super' ) $match = true;

		$img = array_reverse( $img );

		$image = $this->vbp->verify_thumbnails( $img );

		return $image;
	}

    } // END class Video_Blogster_GiantBomb
} // END if ( ! class_exists( 'Video_Blogster_GiantBomb' ) )
?>
