<?php

defined( 'ABSPATH' ) or die( "Oops! This is a WordPress plugin and should not be called directly.\n" );

/**
 * Class for Video Blogster MixCloud support
 * https://www.mixcloud.com/developers/
 */
if ( ! class_exists( 'Video_Blogster_MixCloud' )) {
    class Video_Blogster_MixCloud {

	private $vbp = null;			// will point to main Video_Blogster instance
        private $query_fields = array();        // options for current query
        private $comment_num = 0;       	// Internal counter
        private $batch_limit = 100;       	// Max amount of tracks we can request per API query

	private $total = 0;			// total number of results
        private $num_skipped = 0;               // a sum of all videos skipped for this request
        private $num_updated = 0;               // a sum of all videos updated for this request
        private $num_imported = 0;              // a sum of all videos imported for this request

	/**
	 * Create MixCloud video source
	 * Point back to Video Blogster object to use common functions
	 * Save the query fields for easy access
	 */
	public function __construct( $vbp, $query_fields ) {
		$this->vbp = $vbp;
		$this->query_fields = $query_fields;
		$this->vbp->info_message( sprintf( esc_html__( 'Creating %s resource', 'video-blogster' ), $query_fields[ 'videoSource' ] ), 'notice notice-warning', 'debug' );
	}

        /**
         * Make the query and check for errors.
         */
        private function queryApi( $url ) {
		$this->vbp->info_message( sprintf( '%s : %s%s%s', 
			__FUNCTION__, 
			'<a target="_blank" href="' . esc_url( $url ) . '">',
			esc_url( $url ),
			'</a>'
			), 'notice notice-warning', 'debug' );


		$response = wp_remote_get( $url );

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
                if ( is_wp_error( $response ) ) {
			return $this->vbp->info_message( sprintf( esc_html__( '%s WP error %s in %s(%s) - %s ', 'video-blogster' ), $feedID, $response->get_error_code(), __FUNCTION__, $url, $response->get_error_message() ) );
                }
                $data = json_decode( wp_remote_retrieve_body( $response ) );

                if ( isset( $data->error ) && false === stripos( $data->error->message, 'Upload does not exist' ) ) {
                        return $this->vbp->info_message( sprintf( esc_html__( '%s(%s) - MixCloud API returned %s', 'video-blogster' ), __FUNCTION__, $url, print_r( $data,true ) ) );
                }
                return $data;
        }


	private function grab_comments_batch( $url, $postID ) {
		$totalComments = $this->query_fields['qNumComments'];

		$result = $this->queryApi( $url );

		$comments = isset( $result->data ) ? $result->data : $result;

		if ( empty( $comments ) ) {
			return null;
		}

		$comment_date = current_time( 'Y-m-d H:i:s' );
		foreach ( $comments as $comment ) {
			if ( $this->comment_num < 50 ) {
				$this->vbp->info_message( sprintf (__( 'COMMENT: %s', 'video-blogster' ), htmlentities( print_r($comment,TRUE) ) ), 'notice notice-warning', 'debug' );
			}
			$author = isset( $comment->user->username ) ? trim( $comment->user->username ) : '';
			$author_url = isset( $comment->user->url ) ? trim( $comment->user->url ) : '';
			$content = isset( $comment->comment ) ? trim( $comment->comment ) : '';
			if ( empty( $author ) || empty( $content ) ) {
				continue;
			}
			if ( TRUE == $this->query_fields['cUsePublishedDate'] ) {
				$commentAt = isset( $comment->submit_date ) ? $comment->submit_date : current_time( 'Y-m-d H:i:s' );
				$commentDate = $this->vbp->getDateTime( $commentAt, $url );
				$comment_date = isset( $commentDate ) ? $commentDate->format('Y-m-d H:i:s') : $commentAt;
			}
			$commentdata = array(
				'comment_post_ID'       => $postID,
				'comment_author'        => $author,
				'comment_author_url'    => $author_url,
				'comment_content'       => $content,
				'comment_date'          => $comment_date,
				'comment_approved'      => 1
				);
			$commentdata = apply_filters( 'vbp_mixcloud_comment', $commentdata);
			$comment_id = $this->vbp->save_the_comment( $commentdata, $this->query_fields );
			if ( $comment_id ) {
				$this->comment_num++;
			}
			if ( $totalComments > 0 && $this->comment_num >= $totalComments ) {
				return null;
			}
		}
		return isset( $result->next_href ) ? $result->next_href : null;
	}


	/**
	 * Get MixCloud comments on a track
	 */
	private function grab_comments( $url, $postID ) {
		$cm = $cn = false;
		$this->vbp->bulk_comment_start( $cm, $cn);

		// don't see any way to just request X comments. JPH
		$url = str_replace("www", "api", $url."comments/");

		$this->comment_num = 0;

		$url = apply_filters( 'vbp_mixcloud_comments_query', $url, $this->query_fields );

		while ( $url ) {
			$url = $this->grab_comments_batch( $url, $postID );
		}

		$this->vbp->info_message( sprintf( esc_html__( '%s comments imported successfully to post id %s.', 'video-blogster' ), $this->comment_num, $postID ), 'updated', 'video_import' );

		$this->vbp->bulk_comment_end( $cm, $cn );
	}

        /**
         * Takes an array of video details to process and create posts
         */
	private function save_tracks( $tracks, $url ) {

                if ( ! $tracks ) {
                        return 0;
                }

		foreach ( $tracks as $track ) {

			// if 'cloudcast' then request track/show as it has more data like 'description'
			if ( ( isset( $track->url ) && false !== stripos( $url, 'cloudcast' ) )
					|| $url == 'https://api.mixcloud.com/popular/'
					|| $url == 'https://api.mixcloud.com/hot/'
					|| $url == 'https://api.mixcloud.com/new/'
					) {
				$trackUrl = str_replace( "www", "api", $track->url );
				$track = $this->queryApi( $trackUrl );
			}

			$videoInfo = $this->get_video_info( $track );
			if ( ! $videoInfo ) continue;

			$postID = $this->vbp->save_the_video( $this->query_fields, $videoInfo );

			if ( $postID < 0 ) return 0; // user abort!
			if ( ! $postID ) { $this->num_skipped++; continue; }

			if ( ! empty( $this->query_fields['qNumComments'] ) && $videoInfo['commentCount'] ) {
				$this->grab_comments( $videoInfo['url'], $postID );
			}

			$this->vbp->publish_the_video( $postID, $this->query_fields, $videoInfo );

                        if ( $videoInfo['action'] == 'saved' ) $this->num_imported++;
                        else if ( $videoInfo['action'] == 'updated' ) $this->num_updated++;

			$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
			if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		}
		$processed = $this->num_skipped + $this->num_updated + $this->num_imported;
		if ( $this->vbp->reached_import_limit( $this->query_fields['qNumVideos'], $processed, $this->num_imported, $this->query_fields['qQueryBehavior'] ) ) return 0;
		return 1;
	}

	/*
         * Find the best match thumbnail to what the user requested. 
	 * If it is not defined this code will cascade to the closest match.
         */
	private function get_best_thumbnail( $track ) {
		$img = array();
		$match = false;

		if ( ! $match && isset( $track->pictures->small ) ) $img[] = $track->pictures->small;
		if ( $this->query_fields['qExtraParams2'] == 'small' ) $match = true;

		if ( ! $match && isset( $track->pictures->thumbnail ) ) $img[] = $track->pictures->thumbnail;
		if ( $this->query_fields['qExtraParams2'] == 'thumbnail' ) $match = true;

		if ( ! $match && isset( $track->pictures->medium_mobile ) ) $img[] = $track->pictures->medium_mobile;
		if ( $this->query_fields['qExtraParams2'] == 'medium_mobile' ) $match = true;

		if ( ! $match && isset( $track->pictures->medium ) ) $img[] = $track->pictures->medium;
		if ( $this->query_fields['qExtraParams2'] == 'medium' ) $match = true;

		if ( ! $match && isset( $track->pictures->large ) ) $img[] = $track->pictures->large;
		if ( $this->query_fields['qExtraParams2'] == 'large' ) $match = true;

		if ( ! $match && isset( $track->pictures->{'320wx320h'} ) ) $img[] = $track->pictures->{'320wx320h'};
		if ( $this->query_fields['qExtraParams2'] == '320wx320h' ) $match = true;

		if ( ! $match && isset( $track->pictures->extra_large ) ) $img[] = $track->pictures->extra_large;
		if ( $this->query_fields['qExtraParams2'] == 'extra_large' ) $match = true;

		if ( ! $match && isset( $track->pictures->{'640wx640h'} ) ) $img[] = $track->pictures->{'640wx640h'};
		if ( $this->query_fields['qExtraParams2'] == '640wx640h' ) $match = true;

		if ( ! $match && isset( $track->pictures->{'768wx768h'} ) ) $img[] = $track->pictures->{'768wx768h'};
		if ( $this->query_fields['qExtraParams2'] == '768wx768h' ) $match = true;

		if ( ! $match && isset( $track->pictures->{'1024wx1024h'} ) ) $img[] = $track->pictures->{'1024wx1024h'};
		if ( $this->query_fields['qExtraParams2'] == '1024wx1024h' ) $match = true;

		$img = array_reverse( $img );

		$image = $this->vbp->verify_thumbnails( $img );

		return $image;
	}

        /**
         * Extract video details into our generic videoInfo array
         */
        private function get_video_info( $track ) {
		$videoInfo = array();

		$videoInfo['videoSource'] = 'MixCloud';
		$videoInfo['videoID'] = isset( $track->slug ) ? $track->slug : '';
		$videoInfo['orig_title'] = $videoInfo['title'] = isset( $track->name ) ? $track->name : '';

		// that's enough to check for duplicate post
		if ( $this->vbp->check_post_duplicate( $this->query_fields, $videoInfo ) ) {
			$this->num_skipped++;
			return null;
		}

		$videoInfo['orig_desc'] = $videoInfo['desc'] = isset( $track->description ) ? $track->description : '';
		$videoInfo['association'] = isset( $track->user->username ) ? $track->user->username : '';

		$videoInfo['authorUrl'] = isset( $track->user->url ) ? $track->user->url : '';
		$videoInfo['authorTitle'] = isset( $track->user->name ) ? $track->user->name : '';

		$videoInfo['url'] = isset( $track->url ) ? $track->url : '';

		$this->vbp->info_message( sprintf( esc_html__( 'Checking MixCloud track: [%s] at %s', 'video-blogster' ), $videoInfo['title'], $videoInfo['url'] ), 'notice notice-warning', 'debug' );

		$videoInfo['duration'] = isset( $track->audio_length ) ? $track->audio_length : 0;
		$videoInfo['duration'] = $this->vbp->seconds_to_time( $videoInfo['duration'] );
		$videoInfo['viewCount'] = isset( $track->play_count ) ? $track->play_count : '';
		$videoInfo['likeCount'] = $videoInfo['favoriteCount'] = isset( $track->favorite_count ) ? $track->favorite_count : '';
		$videoInfo['dislikeCount'] = 0;
		$videoInfo['commentCount'] = isset( $track->comment_count ) ? $track->comment_count : '';
		$videoInfo['img'] = $this->get_best_thumbnail( $track );


		// Make a separate call to MixCloud oembed for embed code 
		$url = "https://www.mixcloud.com/oembed?url=" . htmlentities( $videoInfo['url'] ) . "&format=json";
		$videoInfo['videoEmbed'] = $this->vbp->queryoEmbed( $url );

		$publishedAt = isset( $track->created_time ) ? $track->created_time : null;
		$date = $this->vbp->getDateTime( $publishedAt, $videoInfo['url'] );
		$videoInfo['publishedAt'] = $date->format( 'Y-m-d H:i:s' );
		$videoInfo['publishedAt8601'] = isset( $date ) ? $date->format( 'c' ) : '';

		$videoInfo['tags'] = '';
		if ( isset( $track->tags ) ) {
			$tags = array();
			foreach ( $track->tags as $tag ) {
				$tags[] = $tag->name;
			}
			$videoInfo['tags'] = implode( ",", $tags );
		}

		$videoInfo['feedID'] = ! empty ( $this->query_fields['id'] ) ? $this->query_fields['id'] : "";
		$videoInfo = apply_filters( 'vbp_get_video_info', $videoInfo, $track );

		$this->vbp->info_message( sprintf( esc_html__( '%s: %s', 'video-blogster' ), __FUNCTION__, htmlentities( print_r($videoInfo,true) ) ), 'notice notice-warning', 'debug' );

		return $videoInfo;
	}


	public function grab_videos_batch( $url ) {

		$result = $this->queryApi( $url );
		$tracks = isset( $result->data ) ? $result->data : $result;

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( empty( $result ) && empty( $tracks ) ) {
			return $this->vbp->info_message( sprintf (__( '%s %s: No results returned for %s', 'video-blogster' ), $feedID, __FUNCTION__, $url ), 'notice notice-warning', 'critical' );
		}

		if ( ! is_array( $tracks ) ) {
			// specific track is a standard object, not an array, so make array of the stdobj
			$stdObj = $tracks;
			$tracks = array( $stdObj );
		}

		$this->total += count( $tracks );

		$this->vbp->info_message( sprintf( esc_html__( 'MixCloud response data (%s total results): %s', 'video-blogster' ), $this->total, htmlentities( print_r( $tracks,true ) ) ), 'notice notice-warning', 'debug' );

		if ( ! $this->save_tracks( $tracks, $url ) ) return 0;

		return isset( $result->paging->next ) ? $result->paging->next : null;
	}

	public function grab_videos() {
		$maxItems = $this->query_fields['qQueryBehavior'] == 'strict' ? $this->query_fields['qNumVideos'] : -1;
		$limit = $maxItems > $this->batch_limit || $maxItems < 0 ? $this->batch_limit : $maxItems;
		$keyphrase = urlencode( $this->query_fields['qKeyphrase'] );
		$assocType = $this->query_fields['qAssocType'];
		$assoc =  $this->query_fields['qAssoc'];	# by id or url 
		$filters = ! empty( $this->query_fields['qExtraParams'] ) ? "&" . ltrim( $this->query_fields['qExtraParams'], "&" ) : '';

		$feedID = ! empty ( $this->query_fields['id'] ) ? "Feed " . $this->query_fields['id'] . ", " : "";
		if ( empty( $keyphrase ) && empty( $assoc ) ) {
			if ( $assocType == 'popular' ) ;
			else if ( $assocType == 'hot' ) ;
			else if ( $assocType == 'new' ) ;
			else return $this->vbp->info_message( sprintf( esc_html__( '%s Error: No MixCloud keyphrase or association not set properly.', 'video-blogster' ), $feedID ), 'error', 'critical' );
		}

		$url = $assoc;
		
		if ( $assocType == 'popular' ) $url = "https://api.mixcloud.com/popular/?limit={$limit}";
		else if ( $assocType == 'hot' ) $url = "https://api.mixcloud.com/popular/hot/?limit={$limit}";
		else if ( $assocType == 'new' ) $url = "https://api.mixcloud.com/new/?limit={$limit}";
		else if ( stripos( $assoc, 'http://' ) !== FALSE || stripos( $assoc, 'https://' ) !== FALSE ) {
			$url = str_replace( "www", "api", $assoc );
			$url = rtrim( $url, '/' ) . '/';

			if ( $assocType == 'user' ) {
				$url .= 'cloudcasts/';
			}
			else if ( $assocType == 'playlist' ) {
				$url .= 'cloudcasts/';
			}
			else if ( $assocType == 'show' ) {
			}

			$url .= "?limit=" . $limit;
		}
		else if ( $keyphrase && ! $assoc ) {
			$url = "http://api.mixcloud.com/search/?type=cloudcast&q={$keyphrase}&limit={$limit}";
		}
		else {
			return $this->vbp->info_message( sprintf( esc_html__( '%s Error: Missing URL or keyphrase.', 'video-blogster' ), $feedID ), 'error', 'critical' );
		}

		if ( $filters ) {
			$url .= $filters;
		}

		$url = apply_filters( 'vbp_mixcloud_search_query', $url, $this->query_fields );

		while ( $url ) {
			$url = $this->grab_videos_batch( $url );
                        if ( $this->query_fields['qQueryBehavior'] == 'strict' && $maxItems > 0 ) { // failsafe
                                if ( $this->num_skipped + $this->num_imported >= $maxItems ) break;
                        }
		}

		if ( $this->query_fields['qQueryBehavior'] == 'strict' ) $this->total = 0; // don't know total
		$this->vbp->import_finished( 'MixCloud tracks', $this->num_skipped, $this->num_updated, $this->num_imported, $this->total, $this->query_fields['id'], $this->query_fields['qQueryBehavior'] );
		return 0;
	}

	/**
	 * MixCloud 
	 */
	public function check_for_deleted_videos() {

		$getMore = true;
		$offset = 0;
		$chunksize = apply_filters( 'vbp_posts_chunksize', 1000 );

		while ( $getMore ) {
		$posts = $this->vbp->get_posts_by_site( 'MixCloud', $offset );
		if ( ! $posts ) return;

		$num_results = count( $posts );

		if ( $num_results && $offset == 0 ) {
			$this->vbp->info_message( sprintf( esc_html__( 'MixCloud %s Start', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
		}

		foreach ( $posts as $post_id ) {
			$VideoUrl = get_post_meta( $post_id, 'VideoUrl', TRUE );
			$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: checking MixCloud track url: %s ', 'video-blogster' ), $VideoUrl ), 'updated', 'utility_funcs' );
			$url = str_replace("www", "api", $VideoUrl );
			$track = $this->queryApi( $url );
			if ( isset( $track->error->message ) && $track->error->message == 'Upload does not exist.' ) {
				// are there other error messages we should check for? unknown.
				$postID = apply_filters( 'vbp_delete_video', $post_id );
				if ( ! $postID ) {
					continue;
				}
				wp_trash_post( $postID );
				$this->vbp->info_message( sprintf( esc_html__( 'Schedule Checker: MixCloud track url %s not found. Sending post %s to trash.', 'video-blogster' ), $VideoUrl, $postID ), 'updated', 'critical' );
			}
		} // end foreach posts
			if ( $num_results < $chunksize ) break;
			$offset += $chunksize;
			$query_args['offset'] = $offset;
		} // end getMore
		if ( $num_results || $offset ) $this->vbp->info_message( sprintf( esc_html__( 'MixCloud %s End', 'video-blogster' ), __FUNCTION__ ), 'updated', 'utility_funcs' );
	}

    } // END class Video_Blogster_MixCloud
} // END if ( ! class_exists( 'Video_Blogster_MixCloud' ) )


?>
